<?php
// 会话验证
require_once 'session_check.php';


/**
 * 数据库连接工具函数
 * @return PDO 数据库连接实例
 * @throws PDOException 连接失败时抛出异常
 */
function getDbConnection() {
    require_once 'db.php';
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    return $pdo;
}

/**
 * 验证BotId是否唯一
 * @param PDO $pdo 数据库连接
 * @param string $adminUserName 管理员用户名
 * @param string $botId 机器人编号
 * @param int $excludeId 排除的ID（编辑时使用）
 * @return bool 唯一返回true，否则false
 */
function isBotIdUnique($pdo, $adminUserName, $botId, $excludeId = 0) {
    $sql = "SELECT COUNT(*) FROM TelSms_BotConfig WHERE AdminUserName = :adminUserName AND BotId = :botId";
    if ($excludeId > 0) {
        $sql .= " AND Id != :excludeId";
    }
    
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':adminUserName', $adminUserName);
    $stmt->bindParam(':botId', $botId);
    
    if ($excludeId > 0) {
        $stmt->bindParam(':excludeId', $excludeId);
    }
    
    $stmt->execute();
    return $stmt->fetchColumn() == 0;
}

// 处理AJAX请求
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax'])) {
    try {
        $pdo = getDbConnection();
        $currentUser = $_SESSION['admin_username'];
        $response = ['success' => false, 'message' => ''];

        // 处理添加操作
        if ($_POST['action'] === 'add') {
            $botId = trim($_POST['BotId'] ?? '');
            $userName = trim($_POST['UserName'] ?? '');
            $pwd = trim($_POST['Pwd'] ?? '');
            $payType = $_POST['PayType'] ?? '1'; // 获取支付类型，默认1
            
            if (empty($botId) || empty($userName) || empty($pwd)) {
                throw new Exception("机器人编号、HG用户名和密码为必填项");
            }
            
            if (!is_numeric($botId)) {
                throw new Exception("机器人编号必须为数字");
            }
            
            if (!isBotIdUnique($pdo, $currentUser, $botId)) {
                throw new Exception("该机器人编号已存在");
            }
            
            $accountNoList = isset($_POST['AccountNoList']) ? array_filter($_POST['AccountNoList']) : [];
            $accountNoList = implode(',', $accountNoList);
            
            $pwdSafe = trim($_POST['Pwd_Safe'] ?? '');
            $pwdSafe = empty($pwdSafe) ? 'none' : $pwdSafe;
            
            $loginLayer = empty($pwdSafe) || $pwdSafe === 'none' ? 'ag' : 'su';
            
            $hgAdminList = trim($_POST['HgAdminList'] ?? '');
            $formattedHgAdminList = empty($hgAdminList) ? $userName : $userName . '(' . $hgAdminList . ')';
            
            $selMaxId = 10000000000;
            
            $stmt = $pdo->prepare("INSERT INTO TelSms_BotConfig 
                                  (AdminUserName, BotId, AccountNoList, Username, Pwd, Pwd_Safe, 
                                   Login_Layer, SelMaxId, HgAdminList,Pay_Type, CreateDate) 
                                  VALUES 
                                  (:adminUserName, :botId, :accountNoList, :userName, :pwd, :pwdSafe, 
                                   :loginLayer, :selMaxId, :hgAdminList,:payType, NOW())");
            
            $stmt->execute([
                ':adminUserName' => $currentUser,
                ':botId' => $botId,
                ':accountNoList' => $accountNoList,
                ':userName' => $userName,
                ':pwd' => $pwd,
                ':pwdSafe' => $pwdSafe,
                ':loginLayer' => $loginLayer,
                ':selMaxId' => $selMaxId,
                ':hgAdminList' => $formattedHgAdminList,
                ':payType' => $payType
            ]);
            
            $response['success'] = true;
            $response['message'] = "添加HG子账号成功";
        }

        // 处理修改操作
        if ($_POST['action'] === 'edit') {
            $id = $_POST['id'] ?? 0;
            $botId = trim($_POST['BotId'] ?? '');
            $userName = trim($_POST['UserName'] ?? '');
            $pwd = trim($_POST['Pwd'] ?? '');
            $payType = $_POST['PayType'] ?? '1'; // 获取支付类型，默认1
            
            if (empty($id) || empty($botId) || empty($userName) || empty($pwd)) {
                throw new Exception("参数不完整");
            }
            
            if (!is_numeric($botId)) {
                throw new Exception("机器人编号必须为数字");
            }
            
            if (!isBotIdUnique($pdo, $currentUser, $botId, $id)) {
                throw new Exception("该机器人编号已存在");
            }
            
            $accountNoList = isset($_POST['AccountNoList']) ? array_filter($_POST['AccountNoList']) : [];
            $accountNoList = implode(',', $accountNoList);
            
            $pwdSafe = trim($_POST['Pwd_Safe'] ?? '');
            $pwdSafe = empty($pwdSafe) ? 'none' : $pwdSafe;
            
            $loginLayer = empty($pwdSafe) || $pwdSafe === 'none' ? 'ag' : 'su';
            
            $hgAdminList = trim($_POST['HgAdminList'] ?? '');
            $formattedHgAdminList = empty($hgAdminList) ? $userName : $userName . '(' . $hgAdminList . ')';
            
            $selMaxId = 1000000;
            
            $stmt = $pdo->prepare("UPDATE TelSms_BotConfig SET 
                                  BotId = :botId, 
                                  AccountNoList = :accountNoList, 
                                  Username = :userName, 
                                  Pwd = :pwd, 
                                  Pwd_Safe = :pwdSafe, 
                                  Login_Layer = :loginLayer, 
                                 
                                  HgAdminList = :hgAdminList,
                                  Pay_Type = :payType 
                                  WHERE Id = :id AND AdminUserName = :adminUserName");
            
            $stmt->execute([
                ':botId' => $botId,
                ':accountNoList' => $accountNoList,
                ':userName' => $userName,
                ':pwd' => $pwd,
                ':pwdSafe' => $pwdSafe,
                ':loginLayer' => $loginLayer,
               
                ':hgAdminList' => $formattedHgAdminList,
                ':payType' => $payType,
                ':id' => $id,
                ':adminUserName' => $currentUser
            ]);
            
            $response['success'] = true;
            $response['message'] = "修改HG子账号成功";
        }

        // 处理删除操作
        if ($_POST['action'] === 'delete') {
            $id = $_POST['id'] ?? 0;
            
            if (empty($id)) {
                throw new Exception("参数不完整");
            }
            
            $stmt = $pdo->prepare("DELETE FROM TelSms_BotConfig WHERE Id = :id AND AdminUserName = :adminUserName");
            $stmt->execute([
                ':id' => $id,
                ':adminUserName' => $currentUser
            ]);
            
            if ($stmt->rowCount() === 0) {
                throw new Exception("未找到记录或没有权限删除");
            }
            
            $response['success'] = true;
            $response['message'] = "删除HG子账号成功";
        }

        echo json_encode($response);
        exit;
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        exit;
    }
}

try {
    // 数据库连接
    $pdo = getDbConnection();
    
    // 获取当前登录用户
    $currentUser = $_SESSION['admin_username'];

    // 获取当前用户的飞机群记录
    $accountNos = [];
    $stmt = $pdo->prepare("SELECT AccountNo FROM TelSms_Account WHERE Username = :userName ORDER BY AccountNo");
    $stmt->execute([':userName' => $currentUser]);
    $accountNos = $stmt->fetchAll(PDO::FETCH_COLUMN, 0);

    // 获取当前用户的HG子账号记录
    $hgAccounts = [];
    $stmt = $pdo->prepare("SELECT * FROM TelSms_BotConfig WHERE AdminUserName = :adminUserName ORDER BY BotId");
    $stmt->execute([':adminUserName' => $currentUser]);
    $hgAccounts = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // 获取要编辑的记录
    $editHgAccount = null;
    $editId = 0;
    if (isset($_GET['edit']) && is_numeric($_GET['edit'])) {
        $editId = intval($_GET['edit']);
        $stmt = $pdo->prepare("SELECT * FROM TelSms_BotConfig WHERE Id = :id AND AdminUserName = :adminUserName");
        $stmt->execute([
            ':id' => $editId,
            ':adminUserName' => $currentUser
        ]);
        $editHgAccount = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$editHgAccount) {
            throw new Exception("未找到编辑的记录或没有权限");
        }
        
        // 处理AccountNoList数组
        $editHgAccount['AccountNoListArray'] = [];
        if (!empty($editHgAccount['AccountNoList'])) {
            $accountNoArray = explode(',', $editHgAccount['AccountNoList']);
            $editHgAccount['AccountNoListArray'] = array_filter(array_map('trim', $accountNoArray));
        }
        
        // 提取备注部分
        $userName = $editHgAccount['Username'];
        if (strpos($editHgAccount['HgAdminList'], "($userName)") !== false) {
            $editHgAccount['HgAdminListRaw'] = preg_replace("/^$userName\((.*)\)$/", "$1", $editHgAccount['HgAdminList']);
        } else {
            $editHgAccount['HgAdminListRaw'] = '';
        }
    }
} catch (Exception $e) {
    $error = $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>HG子账号管理 - 比赛数据推送</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdn.jsdelivr.net/npm/font-awesome@4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#165DFF',
                        secondary: '#FF4D4F',
                        success: '#52C41A',
                        neutral: '#1D2129',
                        'neutral-light': '#4E5969',
                        'neutral-lighter': '#86909C',
                        'neutral-lightest': '#F2F3F5'
                    },
                    fontFamily: {
                        sans: ['Inter', 'system-ui', 'sans-serif'],
                    },
                }
            }
        }
    </script>
    <style type="text/tailwindcss">
        @layer utilities {
            .content-auto {
                content-visibility: auto;
            }
            .card-shadow {
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            }
            .tab-active {
                border-bottom: 2px solid #165DFF;
                color: #165DFF;
                font-weight: 500;
            }
            .notice-slide {
                animation: slideDown 0.3s ease-out forwards;
            }
            @keyframes slideDown {
                from { transform: translateY(-100%); opacity: 0; }
                to { transform: translateY(0); opacity: 1; }
            }
            .account-card {
                transition: transform 0.2s, box-shadow 0.2s;
            }
            .account-card:hover {
                transform: translateY(-2px);
                box-shadow: 0 6px 24px rgba(0, 0, 0, 0.12);
            }
            .checkbox-grid {
                display: grid;
                grid-template-columns: repeat(auto-fill, minmax(120px, 1fr));
                gap: 0.5rem;
            }
        }
    </style>
</head>
<body class="bg-neutral-lightest min-h-screen font-sans">
    <!-- 顶部导航栏 -->
    <header class="bg-white shadow-md fixed top-0 left-0 right-0 z-10 transition-all duration-300">
        <div class="container mx-auto px-4 py-3 flex items-center justify-between">
            <div class="flex items-center space-x-2">
                <i class="fa fa-bolt text-primary text-xl"></i>
                <h1 class="text-lg font-bold text-neutral">系统配置</h1>
            </div>
            <div class="flex items-center space-x-4">
                <button id="refreshBtn" class="text-primary hover:text-primary/80 transition-colors">
                    <i class="fa fa-refresh"></i>
                </button>
                <a href="bet_my.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="text-neutral-light hover:text-neutral transition-colors">
                    <i class="fa fa-arrow-left"></i>
                </a>
            </div>
        </div>
        
        <!-- 标签页导航 -->
        <div class="bg-white border-t border-gray-100">
            <div class="container mx-auto">
                <div class="flex overflow-x-auto scrollbar-hide">
                    <a href="my_config.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" 
                       class="tab-link flex-1 py-3 px-4 text-center">
                        Telegram群
                    </a>
                    <a href="my_config_hg.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" 
                       class="tab-link flex-1 py-3 px-4 text-center tab-active">
                        HG子账号
                    </a>
                    <a href="admin2/index.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" 
                       class="tab-link flex-1 py-3 px-4 text-center">
                        Telegram机器人
                    </a>
                    <a href="admin1/index.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" 
                       class="tab-link flex-1 py-3 px-4 text-center">
                        HG机器人
                    </a>
                </div>
            </div>
        </div>
    </header>

    <!-- 提示信息容器 -->
    <div id="noticeContainer" class="fixed top-28 left-0 right-0 z-50 px-4 pointer-events-none"></div>

    <!-- 主内容区 -->
    <main class="container mx-auto px-4 pt-32 pb-32">
      
        
        <!-- 功能按钮区 -->
        <div class="flex justify-between items-center mb-6">
            <h3 class="text-lg font-medium text-neutral"></h3>
            <button id="addBtn" class="bg-primary hover:bg-primary/90 text-white px-4 py-2 rounded-lg transition-colors flex items-center">
                <i class="fa fa-plus mr-2"></i> 添加HG子账号
            </button>
        </div>
        
        <!-- 记录列表容器 -->
        <div id="accountListContainer">
            <?php if (empty($hgAccounts)): ?>
                <div class="text-center py-10 bg-neutral-lightest rounded-lg">
                    <i class="fa fa-user-o text-4xl text-neutral-lighter mb-3"></i>
                    <p class="text-neutral-lighter">暂无HG子账号记录</p>
                    <p class="text-xs text-neutral-lighter mt-1">点击"添加HG子账号"按钮创建新记录</p>
                </div>
            <?php else: ?>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <?php foreach ($hgAccounts as $account): ?>
                        <!-- HG子账号卡片 -->
                        <div class="account-card bg-white border border-gray-100 rounded-xl p-5 card-shadow" data-id="<?php echo $account['Id']; ?>">
                            <div class="flex justify-between items-start mb-4">
                                <h4 class="font-medium text-neutral">HG子账号 #<?php echo htmlspecialchars($account['BotId']); ?></h4>
                                <div class="flex space-x-2">
                                    <!-- 编辑链接 -->
                                    <a href="my_config_hg.php?<?php echo isset($_GET['AccountNo']) ? 'AccountNo=' . urlencode($_GET['AccountNo']) . '&' : ''; ?>edit=<?php echo $account['Id']; ?>" 
                                       class="text-primary hover:text-primary/80 transition-colors p-1 edit-link"
                                       data-id="<?php echo $account['Id']; ?>"
                                       title="编辑">
                                        <i class="fa fa-edit"></i>
                                    </a>
                                    <button class="delete-btn text-secondary hover:text-secondary/80 transition-colors p-1" 
                                            data-id="<?php echo $account['Id']; ?>"
                                            data-bot-id="<?php echo $account['BotId']; ?>"
                                            title="删除">
                                        <i class="fa fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                            
                            <div class="space-y-3 text-sm">
                                <div class="flex items-center">
                                    <i class="fa fa-user text-primary w-6"></i>
                                    <span class="text-neutral-light ml-2">HG用户名：</span>
                                    <span class="text-neutral truncate ml-1 flex-1"><?php echo htmlspecialchars($account['Username']); ?></span>
                                </div>
                                
                                <div class="flex items-center">
                                    <i class="fa fa-lock text-primary w-6"></i>
                                    <span class="text-neutral-light ml-2">密码：</span>
                                    <span class="text-neutral truncate ml-1 flex-1"><?php echo htmlspecialchars($account['Pwd']); ?></span>
                                </div>
                                
                                <div class="flex items-center">
                                    <i class="fa fa-shield text-primary w-6"></i>
                                    <span class="text-neutral-light ml-2">安全码：</span>
                                    <span class="text-neutral truncate ml-1 flex-1">
                                        <?php echo $account['Pwd_Safe'] === 'none' ? '未设置' : htmlspecialchars($account['Pwd_Safe']); ?>
                                    </span>
                                </div>
                                
                                <div class="flex items-start">
                                    <i class="fa fa-sitemap text-primary w-6 mt-1"></i>
                                    <span class="text-neutral-light ml-2 mt-1">关联群：</span>
                                    <div class="text-neutral ml-1 flex-1">
                                        <?php 
                                        $accountNos2 = [];
                                        if (!empty($account['AccountNoList'])) {
                                            $rawList = explode(',', $account['AccountNoList']);
                                            $accountNos2 = array_filter(array_map('trim', $rawList));
                                        }
                                        
                                        if (empty($accountNos2)) {
                                            echo '未关联任何群';
                                        } else {
                                            echo implode('、', array_map('htmlspecialchars', $accountNos2));
                                        }
                                        ?>
                                    </div>
                                </div>
                                
                                <div class="flex items-center">
                                    <i class="fa fa-sticky-note text-primary w-6"></i>
                                    <span class="text-neutral-light ml-2">备注：</span>
                                    <span class="text-neutral truncate ml-1 flex-1"><?php echo htmlspecialchars($account['HgAdminList']); ?></span>
                                </div>
                                
                                <div class="flex items-center text-xs text-neutral-lighter">
                                    <i class="fa fa-clock-o w-5"></i>
                                    <span>创建时间：<?php echo htmlspecialchars($account['CreateDate']); ?></span>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <!-- 添加/编辑HG子账号模态框 -->
    <div id="accountModal" class="fixed inset-0 bg-black/50 z-50 hidden flex items-center justify-center p-4">
        <div class="bg-white rounded-xl w-full max-w-lg overflow-hidden transform transition-all duration-300 scale-95 opacity-0 max-h-[90vh] overflow-y-auto" id="modalContent">
            <div class="p-6">
                <div class="flex justify-between items-center mb-4">
                    <h3 id="modalTitle" class="text-lg font-bold text-neutral">
                        <?php echo $editHgAccount ? '修改HG子账号' : '添加HG子账号'; ?>
                    </h3>
                    <button id="closeModal" class="text-neutral-light hover:text-neutral transition-colors">
                        <i class="fa fa-times"></i>
                    </button>
                </div>
                
                <form id="accountForm" method="post">
                    <input type="hidden" name="ajax" value="1">
                    <input type="hidden" name="action" value="<?php echo $editHgAccount ? 'edit' : 'add'; ?>">
                    <input type="hidden" name="id" id="accountId" value="<?php echo $editHgAccount ? $editHgAccount['Id'] : ''; ?>">
                    
                    <div class="space-y-4">
                        <div>
                            <label for="BotId" class="block text-sm font-medium text-neutral-light mb-1">机器人编号 <span class="text-secondary">*</span></label>
                            <input type="text" id="BotId" name="BotId" 
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary transition-colors"
                                placeholder="请输入数字编号"
                                value="<?php echo $editHgAccount ? htmlspecialchars($editHgAccount['BotId']) : ''; ?>">
                            <p class="text-xs text-neutral-lighter mt-1">必须为数字，且唯一</p>
                        </div>
                        
                        <div>
                            <label for="UserName" class="block text-sm font-medium text-neutral-light mb-1">HG用户名 <span class="text-secondary">*</span></label>
                            <input type="text" id="UserName" name="UserName" 
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary transition-colors"
                                placeholder="请输入HG用户名"
                                value="<?php echo $editHgAccount ? htmlspecialchars($editHgAccount['Username']) : ''; ?>">
                        </div>
                        
                        <div>
                            <label for="Pwd" class="block text-sm font-medium text-neutral-light mb-1">HG密码 <span class="text-secondary">*</span></label>
                            <input type="text" id="Pwd" name="Pwd" 
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary transition-colors"
                                placeholder="请输入HG密码"
                                value="<?php echo $editHgAccount ? htmlspecialchars($editHgAccount['Pwd']) : ''; ?>">
                        </div>
                        
                        <div>
                            <label for="Pwd_Safe" class="block text-sm font-medium text-neutral-light mb-1">安全代码</label>
                            <input type="text" id="Pwd_Safe" name="Pwd_Safe" 
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary transition-colors"
                                placeholder="请输入安全代码（可选）"
                                value="<?php echo $editHgAccount && $editHgAccount['Pwd_Safe'] !== 'none' ? htmlspecialchars($editHgAccount['Pwd_Safe']) : ''; ?>">
                            <p class="text-xs text-neutral-lighter mt-1">不输入则默认为"none"</p>
                        </div>

                        <!-- 新增Pay_Type字段 -->
                        <div>
                            <label for="PayType" class="block text-sm font-medium text-neutral-light mb-1">数据模式 <span class="text-secondary">*</span></label>
                            <select id="PayType" name="PayType" 
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary transition-colors">
                                <option value="1" <?php echo (!$editHgAccount || (isset($editHgAccount['Pay_Type']) && $editHgAccount['Pay_Type'] == 1)) ? 'selected' : ''; ?>>1 - 默认模式</option>
                                <option value="0" <?php echo (isset($editHgAccount['Pay_Type']) && $editHgAccount['Pay_Type'] == 0) ? 'selected' : ''; ?>>0 - 兼容模式</option>
                            </select>
                            <p class="text-xs text-neutral-lighter mt-1">默认选择1，若无法抓取比赛结果，请改为0</p>
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-neutral-light mb-1">关联飞机群</label>
                            <?php if (empty($accountNos)): ?>
                                <p class="text-neutral-lighter text-sm py-2">暂无可用的飞机群，请先创建飞机群</p>
                            <?php else: ?>
                                <div class="checkbox-grid">
                                    <?php foreach ($accountNos as $accountNo): ?>
                                        <label class="flex items-center space-x-2 p-2 border border-gray-100 rounded-lg hover:bg-neutral-lightest/50 cursor-pointer">
                                            <input type="checkbox" name="AccountNoList[]" value="<?php echo htmlspecialchars($accountNo); ?>"
                                                <?php echo ($editHgAccount && in_array($accountNo, $editHgAccount['AccountNoListArray'])) ? 'checked' : ''; ?>
                                                class="rounded text-primary focus:ring-primary/50">
                                            <span><?php echo htmlspecialchars($accountNo); ?></span>
                                        </label>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <div>
                            <label for="HgAdminList" class="block text-sm font-medium text-neutral-light mb-1">备注（可选）</label>
                            <input type="text" id="HgAdminList" name="HgAdminList" 
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary transition-colors"
                                placeholder="请输入备注信息"
                                value="<?php echo $editHgAccount ? htmlspecialchars($editHgAccount['HgAdminListRaw']) : ''; ?>">
                            <p class="text-xs text-neutral-lighter mt-1">将显示为 "用户名(备注)"</p>
                        </div>
                        
                        <div class="pt-2">
                            <button type="submit" class="w-full bg-primary hover:bg-primary/90 text-white font-medium py-2 px-4 rounded-lg transition-colors">
                                <?php echo $editHgAccount ? '保存修改' : '添加'; ?>
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- 删除确认模态框 -->
    <div id="deleteModal" class="fixed inset-0 bg-black/50 z-50 hidden flex items-center justify-center p-4">
        <div class="bg-white rounded-xl w-full max-w-md overflow-hidden transform transition-all duration-300 scale-95 opacity-0" id="deleteModalContent">
            <div class="p-6 text-center">
                <div class="inline-flex items-center justify-center w-16 h-16 rounded-full bg-secondary/10 text-secondary mb-4 mx-auto">
                    <i class="fa fa-exclamation-triangle text-2xl"></i>
                </div>
                <h3 class="text-lg font-bold text-neutral mb-2">确认删除</h3>
                <p class="text-neutral-light mb-6" id="deleteConfirmText">您确定要删除这条记录吗？此操作不可撤销。</p>
                
                <form id="deleteForm" method="post">
                    <input type="hidden" name="ajax" value="1">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" id="deleteId" value="">
                    
                    <div class="flex space-x-3">
                        <button type="button" id="cancelDelete" class="flex-1 px-4 py-2 bg-neutral-lightest hover:bg-gray-200 text-neutral font-medium rounded-lg transition-colors">
                            取消
                        </button>
                        <button type="submit" class="flex-1 px-4 py-2 bg-secondary hover:bg-secondary/90 text-white font-medium rounded-lg transition-colors">
                            确认删除
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- 底部导航 -->
    <nav class="fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 py-2 px-4 z-20">
        <div class="flex justify-around">
            <a href="bet_list.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-home text-lg"></i>
                <span class="text-xs mt-1">首页</span>
            </a>
            <a href="bet_good.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-line-chart text-lg"></i>
                <span class="text-xs mt-1">数据</span>
            </a>
            <a href="bet_result.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-trophy text-lg"></i>
                <span class="text-xs mt-1">结果</span>
            </a>
            <a href="bet_notice.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-bell text-lg"></i>
                <span class="text-xs mt-1">通知</span>
            </a>
            <a href="bet_my.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-secondary">
                <i class="fa fa-user text-lg"></i>
                <span class="text-xs mt-1">我的</span>
            </a>
        </div>
    </nav>

    <script>
        // DOM元素
        const addBtn = document.getElementById('addBtn');
        const accountModal = document.getElementById('accountModal');
        const modalContent = document.getElementById('modalContent');
        const closeModal = document.getElementById('closeModal');
        const accountForm = document.getElementById('accountForm');
        const deleteBtns = document.querySelectorAll('.delete-btn');
        const deleteModal = document.getElementById('deleteModal');
        const deleteModalContent = document.getElementById('deleteModalContent');
        const cancelDelete = document.getElementById('cancelDelete');
        const deleteForm = document.getElementById('deleteForm');
        const deleteId = document.getElementById('deleteId');
        const deleteConfirmText = document.getElementById('deleteConfirmText');
        const refreshBtn = document.getElementById('refreshBtn');
        const noticeContainer = document.getElementById('noticeContainer');
        const accountListContainer = document.getElementById('accountListContainer');
        const modalTitle = document.getElementById('modalTitle');
        const actionInput = document.querySelector('input[name="action"]');
        const accountIdInput = document.getElementById('accountId');

        // 显示通知
        function showNotice(message, type = 'success') {
            const existingNotices = noticeContainer.querySelectorAll('div');
            existingNotices.forEach(notice => notice.remove());
            
            const notice = document.createElement('div');
            notice.className = `bg-${type === 'success' ? 'success' : 'secondary'} text-white py-2 px-4 notice-slide pointer-events-auto`;
            notice.innerHTML = `
                <div class="container mx-auto">
                    <div class="flex items-center">
                        <i class="fa ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-2"></i>
                        <span>${message}</span>
                        <button class="ml-auto text-white hover:text-white/80 close-notice">
                            <i class="fa fa-times"></i>
                        </button>
                    </div>
                </div>
            `;
            
            noticeContainer.appendChild(notice);
            
            notice.querySelector('.close-notice').addEventListener('click', () => {
                notice.style.opacity = '0';
                setTimeout(() => notice.remove(), 300);
            });
            
            setTimeout(() => {
                notice.style.opacity = '0';
                setTimeout(() => notice.remove(), 300);
            }, 3000);
        }

        // 显示模态框
        function showModal(modal, content) {
            modal.classList.remove('hidden');
            document.body.style.overflow = 'hidden';
            setTimeout(() => {
                content.classList.remove('scale-95', 'opacity-0');
                content.classList.add('scale-100', 'opacity-100');
            }, 10);
        }

        // 隐藏模态框
        function hideModal(modal, content) {
            content.classList.remove('scale-100', 'opacity-100');
            content.classList.add('scale-95', 'opacity-0');
            document.body.style.overflow = '';
            setTimeout(() => {
                modal.classList.add('hidden');
            }, 300);
        }

        // 刷新账号列表
        function refreshAccountList() {
            window.location.reload();
        }

        // 添加按钮点击事件
        addBtn.addEventListener('click', function() {
            // 重置为添加模式
            actionInput.value = 'add';
            modalTitle.textContent = '添加HG子账号';
            accountIdInput.value = '';
            // 清空表单
            document.getElementById('BotId').value = '';
            document.getElementById('UserName').value = '';
            document.getElementById('Pwd').value = '';
            document.getElementById('Pwd_Safe').value = '';
            document.getElementById('HgAdminList').value = '';
            document.getElementById('PayType').value = '1'; // 默认选中1
            // 取消所有复选框选中状态
            document.querySelectorAll('input[name="AccountNoList[]"]').forEach(checkbox => {
                checkbox.checked = false;
            });
            showModal(accountModal, modalContent);
        });

        // 关闭模态框
        closeModal.addEventListener('click', () => hideModal(accountModal, modalContent));

        // 删除按钮点击事件
        deleteBtns.forEach(btn => {
            btn.addEventListener('click', function() {
                const id = this.getAttribute('data-id');
                const botId = this.getAttribute('data-bot-id');
                deleteId.value = id;
                deleteConfirmText.textContent = `您确定要删除机器人 #${botId} 的记录吗？此操作不可撤销。`;
                showModal(deleteModal, deleteModalContent);
            });
        });

        // 取消删除
        cancelDelete.addEventListener('click', () => hideModal(deleteModal, deleteModalContent));

        // 刷新按钮点击事件
        refreshBtn.addEventListener('click', refreshAccountList);

        // 表单提交处理（添加/编辑）
        accountForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            
            try {
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    hideModal(accountModal, modalContent);
                    showNotice(result.message, 'success');
                    refreshAccountList();
                } else {
                    showNotice(result.message, 'error');
                }
            } catch (error) {
                showNotice('操作失败，请重试', 'error');
                console.error('Error:', error);
            }
        });

        // 删除表单提交处理
        deleteForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            
            try {
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    hideModal(deleteModal, deleteModalContent);
                    showNotice(result.message, 'success');
                    refreshAccountList();
                } else {
                    showNotice(result.message, 'error');
                }
            } catch (error) {
                showNotice('删除失败，请重试', 'error');
                console.error('Error:', error);
            }
        });

        // 点击模态框背景关闭
        accountModal.addEventListener('click', e => {
            if (e.target === accountModal) hideModal(accountModal, modalContent);
        });

        deleteModal.addEventListener('click', e => {
            if (e.target === deleteModal) hideModal(deleteModal, deleteModalContent);
        });

        // 表单提交验证
        accountForm.addEventListener('submit', function(e) {
            const botId = document.getElementById('BotId').value.trim();
            const userName = document.getElementById('UserName').value.trim();
            const pwd = document.getElementById('Pwd').value.trim();
            
            if (!botId) {
                e.preventDefault();
                showNotice('请输入机器人编号', 'error');
                return;
            }
            
            if (!/^\d+$/.test(botId)) {
                e.preventDefault();
                showNotice('机器人编号必须为数字', 'error');
                return;
            }
            
            if (!userName) {
                e.preventDefault();
                showNotice('请输入HG用户名', 'error');
                return;
            }
            
            if (!pwd) {
                e.preventDefault();
                showNotice('请输入HG密码', 'error');
                return;
            }
        });

        // 页面加载时，如果是编辑状态，自动打开模态框
        <?php if ($editHgAccount): ?>
            document.addEventListener('DOMContentLoaded', function() {
                showModal(accountModal, modalContent);
            });
        <?php endif; ?>

        // 显示初始错误信息
        <?php if (isset($error)): ?>
            document.addEventListener('DOMContentLoaded', function() {
                showNotice('<?php echo addslashes($error); ?>', 'error');
            });
        <?php endif; ?>
    </script>
</body>
</html>