<?php
// 会话验证
require_once 'session_check.php';

/**
 * 数据库连接工具函数
 * @return PDO 数据库连接实例
 * @throws PDOException 连接失败时抛出异常
 */
function getDbConnection() {
    require_once 'db.php';
    
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    return $pdo;
}

/**
 * 验证群编号格式（英文+数字）
 * @param string $accountNo 群编号
 * @return bool 验证通过返回true，否则false
 */
function validateAccountNo($accountNo) {
    return preg_match('/^[A-Za-z0-9]+$/', $accountNo);
}

/**
 * 发送Telegram测试消息
 * @param string $botToken 机器人Token
 * @param string $chatId 聊天ID
 * @param string $message 消息内容
 * @return array 包含成功状态和消息的数组
 */
function sendTelegramTestMessage($botToken, $chatId, $message) {
    try {
        // Telegram API URL
        $url = "https://api.telegram.org/bot{$botToken}/sendMessage";
        
        // 准备请求数据
        $data = [
            'chat_id' => $chatId,
            'text' => $message
        ];
        
        // 初始化cURL
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // 生产环境应设为true
        
        // 执行请求
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        // 检查错误
        if(curl_errno($ch)) {
            throw new Exception('cURL错误: ' . curl_error($ch));
        }
        
        curl_close($ch);
        
        // 解析响应
        $responseData = json_decode($response, true);
        
        if($httpCode == 200 && $responseData['ok'] == true) {
            return [
                'success' => true,
                'message' => '消息发送成功',
                'details' => '消息ID: ' . $responseData['result']['message_id']
            ];
        } else {
            $errorMsg = isset($responseData['description']) ? $responseData['description'] : '未知错误';
            return [
                'success' => false,
                'message' => '消息发送失败',
                'details' => $errorMsg
            ];
        }
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => '发送过程出错',
            'details' => $e->getMessage()
        ];
    }
}

try {
    // 处理测试消息发送
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'test') {
        $botToken = trim($_POST['botToken'] ?? '');
        $chatId = trim($_POST['chatId'] ?? '');
        $message = trim($_POST['message'] ?? '');
        
        if (empty($botToken) || empty($chatId)) {
            throw new Exception("机器人Token和会话ID不能为空");
        }
        
        if (empty($message)) {
            $message = "这是一条来自系统的测试消息";
        }
        
        // 发送测试消息
        $result = sendTelegramTestMessage($botToken, $chatId, $message);
        
        // 返回JSON结果
        header('Content-Type: application/json');
        echo json_encode($result);
        exit;
    }

    // 数据库连接
    $pdo = getDbConnection();
    
    // 获取当前登录用户
    $currentUser = $_SESSION['admin_username'];

    // 获取当前用户的ServerIP（用于SiteUrl默认值）
    $serverIP = '';
    try {
        $stmt = $pdo->prepare("SELECT ServerIP FROM TelSms_Admin WHERE UserName = :userName");
        $stmt->bindParam(':userName', $currentUser);
        $stmt->execute();
        $adminInfo = $stmt->fetch(PDO::FETCH_ASSOC);
        $serverIP = $adminInfo['ServerIP'] ?? '';
    } catch (PDOException $e) {
        throw new Exception("获取服务器信息失败: " . $e->getMessage());
    }

    // 获取机器人列表（TelSms_Bot）
    $bots = [];
    try {
        $stmt = $pdo->query("SELECT Token, Name FROM TelSms_Bot ORDER BY Name");
        $bots = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        throw new Exception("获取机器人列表失败: " . $e->getMessage());
    }

    // 获取当前用户的飞机群记录
    $accounts = [];
    $accountCount = 0;
    try {
        $stmt = $pdo->prepare("SELECT * FROM TelSms_Account WHERE UserName = :userName ORDER BY AccountNo");
        $stmt->bindParam(':userName', $currentUser);
        $stmt->execute();
        $accounts = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $accountCount = count($accounts);
    } catch (PDOException $e) {
        throw new Exception("获取飞机群列表失败: " . $e->getMessage());
    }

    // 处理添加操作
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add') {
        // 验证记录数不超过3条
        if ($accountCount >= 3) {
            throw new Exception("每个账号最多只能创建3条记录");
        }
        
        // 验证必填字段
        $accountNo = trim($_POST['AccountNo'] ?? '');
        $chatId = trim($_POST['ChatId'] ?? '');
        $botToken = trim($_POST['BotToken'] ?? '');
        
        if (empty($accountNo) || empty($chatId) || empty($botToken)) {
            throw new Exception("群编号、会话ID和机器人为必填项");
        }
        
        // 验证AccountNo格式
        if (!validateAccountNo($accountNo)) {
            throw new Exception("群编号只能包含英文和数字");
        }
        
        // 检查AccountNo是否已存在
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM TelSms_Account WHERE AccountNo = :accountNo AND UserName = :userName");
        $stmt->bindParam(':accountNo', $accountNo);
        $stmt->bindParam(':userName', $currentUser);
        $stmt->execute();
        if ($stmt->fetchColumn() > 0) {
            throw new Exception("该群编号已存在");
        }
        
        // 插入新记录
        $stmt = $pdo->prepare("INSERT INTO TelSms_Account (AccountNo, ChatId, BotToken, SiteUrl, UserName, CreateDate) 
                              VALUES (:accountNo, :chatId, :botToken, :siteUrl, :userName, NOW())");
        $stmt->bindParam(':accountNo', $accountNo);
        $stmt->bindParam(':chatId', $chatId);
        $stmt->bindParam(':botToken', $botToken);
        $stmt->bindParam(':siteUrl', $serverIP);
        $stmt->bindParam(':userName', $currentUser);
        $stmt->execute();
        
        $success = "添加成功";
        // 重新加载页面
        header("Refresh: 0");
        exit;
    }

    // 处理修改操作
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'edit') {
        $id = $_POST['id'] ?? 0;
        $accountNo = trim($_POST['AccountNo'] ?? '');
        $chatId = trim($_POST['ChatId'] ?? '');
        $botToken = trim($_POST['BotToken'] ?? '');
        
        if (empty($id) || empty($accountNo) || empty($chatId) || empty($botToken)) {
            throw new Exception("参数不完整");
        }
        
        // 验证AccountNo格式
        if (!validateAccountNo($accountNo)) {
            throw new Exception("群编号只能包含英文和数字");
        }
        
        // 检查AccountNo是否已存在（排除当前记录）
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM TelSms_Account WHERE AccountNo = :accountNo AND UserName = :userName AND Id != :id");
        $stmt->bindParam(':accountNo', $accountNo);
        $stmt->bindParam(':userName', $currentUser);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        if ($stmt->fetchColumn() > 0) {
            throw new Exception("该群编号已存在");
        }
        
        // 更新记录
        $stmt = $pdo->prepare("UPDATE TelSms_Account SET AccountNo = :accountNo, ChatId = :chatId, BotToken = :botToken 
                              WHERE Id = :id AND UserName = :userName");
        $stmt->bindParam(':accountNo', $accountNo);
        $stmt->bindParam(':chatId', $chatId);
        $stmt->bindParam(':botToken', $botToken);
        $stmt->bindParam(':id', $id);
        $stmt->bindParam(':userName', $currentUser);
        $stmt->execute();
        
        if ($stmt->rowCount() === 0) {
            throw new Exception("未找到记录或没有权限修改");
        }
        
        $success = "修改成功";
        // 重新加载页面
        header("Refresh: 0");
        exit;
    }

    // 处理删除操作
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete') {
        $id = $_POST['id'] ?? 0;
        
        if (empty($id)) {
            throw new Exception("参数不完整");
        }
        
        $stmt = $pdo->prepare("DELETE FROM TelSms_Account WHERE Id = :id AND UserName = :userName");
        $stmt->bindParam(':id', $id);
        $stmt->bindParam(':userName', $currentUser);
        $stmt->execute();
        
        if ($stmt->rowCount() === 0) {
            throw new Exception("未找到记录或没有权限删除");
        }
        
        $success = "删除成功";
        // 重新加载页面
        header("Refresh: 0");
        exit;
    }

    // 获取要编辑的记录
    $editAccount = null;
    if (isset($_GET['edit'])) {
        $editId = $_GET['edit'];
        $stmt = $pdo->prepare("SELECT * FROM TelSms_Account WHERE Id = :id AND UserName = :userName");
        $stmt->bindParam(':id', $editId);
        $stmt->bindParam(':userName', $currentUser);
        $stmt->execute();
        $editAccount = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$editAccount) {
            throw new Exception("未找到编辑的记录或没有权限");
        }
    }
} catch (Exception $e) {
    $error = $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>系统配置 - 比赛数据推送</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdn.jsdelivr.net/npm/font-awesome@4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#165DFF',
                        secondary: '#FF4D4F',
                        success: '#52C41A',
                        neutral: '#1D2129',
                        'neutral-light': '#4E5969',
                        'neutral-lighter': '#86909C',
                        'neutral-lightest': '#F2F3F5'
                    },
                    fontFamily: {
                        sans: ['Inter', 'system-ui', 'sans-serif'],
                    },
                }
            }
        }
    </script>
    <style type="text/tailwindcss">
        @layer utilities {
            .content-auto {
                content-visibility: auto;
            }
            .card-shadow {
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            }
            .tab-active {
                border-bottom: 2px solid #165DFF;
                color: #165DFF;
                font-weight: 500;
            }
            .notice-slide {
                animation: slideDown 0.3s ease-out forwards;
            }
            @keyframes slideDown {
                from { transform: translateY(-100%); opacity: 0; }
                to { transform: translateY(0); opacity: 1; }
            }
            .account-card {
                transition: transform 0.2s, box-shadow 0.2s;
            }
            .account-card:hover {
                transform: translateY(-2px);
                box-shadow: 0 6px 24px rgba(0, 0, 0, 0.12);
            }
            .fade-in {
                animation: fadeIn 0.3s ease-out forwards;
            }
            @keyframes fadeIn {
                from { opacity: 0; }
                to { opacity: 1; }
            }
        }
    </style>
</head>
<body class="bg-neutral-lightest min-h-screen font-sans">
    <!-- 顶部导航栏 -->
    <header class="bg-white shadow-md fixed top-0 left-0 right-0 z-10 transition-all duration-300">
        <div class="container mx-auto px-4 py-3 flex items-center justify-between">
            <div class="flex items-center space-x-2">
                <i class="fa fa-bolt text-primary text-xl"></i>
                <h1 class="text-lg font-bold text-neutral">系统配置</h1>
            </div>
            <div class="flex items-center space-x-4">
                <button id="refreshBtn" class="text-primary hover:text-primary/80 transition-colors">
                    <i class="fa fa-refresh"></i>
                </button>
                <a href="bet_my.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="text-neutral-light hover:text-neutral transition-colors">
                    <i class="fa fa-arrow-left"></i>
                </a>
            </div>
        </div>
        
        <!-- 标签页导航 - 置顶于头部 -->
        <div class="bg-white border-t border-gray-100">
            <div class="container mx-auto">
                <div class="flex overflow-x-auto scrollbar-hide">
                    <a href="my_config.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" 
                       class="tab-link flex-1 py-3 px-4 text-center tab-active">
                        Telegram群
                    </a>
                    <a href="my_config_hg.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" 
                       class="tab-link flex-1 py-3 px-4 text-center">
                        HG子账号
                    </a>
                    <a href="admin2/index.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" 
                       class="tab-link flex-1 py-3 px-4 text-center">
                        Telegram机器人
                    </a>
                    <a href="admin1/index.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" 
                       class="tab-link flex-1 py-3 px-4 text-center">
                        HG机器人
                    </a>
                </div>
            </div>
        </div>
    </header>

    <!-- 提示信息 -->
    <?php if (isset($success)): ?>
        <div class="fixed top-28 left-0 right-0 z-50 bg-success text-white py-2 px-4 notice-slide" id="successNotice">
            <div class="container mx-auto">
                <div class="flex items-center">
                    <i class="fa fa-check-circle mr-2"></i>
                    <span><?php echo htmlspecialchars($success); ?></span>
                    <button class="ml-auto text-white hover:text-white/80 close-notice">
                        <i class="fa fa-times"></i>
                    </button>
                </div>
            </div>
        </div>
    <?php endif; ?>

    <?php if (isset($error)): ?>
        <div class="fixed top-28 left-0 right-0 z-50 bg-secondary text-white py-2 px-4 notice-slide" id="errorNotice">
            <div class="container mx-auto">
                <div class="flex items-center">
                    <i class="fa fa-exclamation-circle mr-2"></i>
                    <span><?php echo htmlspecialchars($error); ?></span>
                    <button class="ml-auto text-white hover:text-white/80 close-notice">
                        <i class="fa fa-times"></i>
                    </button>
                </div>
            </div>
        </div>
    <?php endif; ?>

    <!-- 主内容区 -->
    <main class="container mx-auto px-4 pt-32 pb-32">
        <!-- 飞机群管理内容 -->
        <div class="bg-white rounded-xl p-6 card-shadow">
            <div class="flex justify-between items-center mb-6">
                <h3 class="text-lg font-medium text-neutral"></h3>
                <button id="addBtn" class="bg-primary hover:bg-primary/90 text-white px-4 py-2 rounded-lg transition-colors flex items-center" 
                        <?php echo $accountCount >= 3 ? 'disabled title="每个账号最多只能创建3条记录"' : ''; ?>>
                    <i class="fa fa-plus mr-2"></i> 添加飞机群
                </button>
            </div>
            
            <!-- 记录列表 - 卡片形式 -->
            <?php if (empty($accounts)): ?>
                <div class="text-center py-10 bg-neutral-lightest rounded-lg">
                    <i class="fa fa-inbox text-4xl text-neutral-lighter mb-3"></i>
                    <p class="text-neutral-lighter">暂无飞机群记录</p>
                    <p class="text-xs text-neutral-lighter mt-1">点击"添加飞机群"按钮创建新记录</p>
                </div>
            <?php else: ?>
                <!-- 卡片网格布局 - 移动端单列，平板双列 -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <?php foreach ($accounts as $account): ?>
                        <?php
                        // 查找对应的机器人名称
                        $botName = '未知';
                        foreach ($bots as $bot) {
                            if ($bot['Token'] == $account['BotToken']) {
                                $botName = $bot['Name'];
                                break;
                            }
                        }
                        ?>
                        <!-- 飞机群卡片 -->
                        <div class="account-card bg-white border border-gray-100 rounded-xl p-5 card-shadow">
                            <!-- 卡片标题 -->
                            <div class="flex justify-between items-start mb-4">
                                <h4 class="font-medium text-neutral"><?php echo htmlspecialchars($account['AccountNo']); ?></h4>
                                <div class="flex space-x-2">
                                    <button class="test-btn text-primary hover:text-primary/80 transition-colors p-1" 
                                            data-id="<?php echo $account['Id']; ?>"
                                            data-bot-token="<?php echo htmlspecialchars($account['BotToken']); ?>"
                                            data-chat-id="<?php echo htmlspecialchars($account['ChatId']); ?>"
                                            title="测试发送消息">
                                        <i class="fa fa-paper-plane"></i>
                                    </button>
                                    <a href="my_config.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>&edit=<?php echo $account['Id']; ?>" 
                                       class="text-primary hover:text-primary/80 transition-colors p-1"
                                       title="编辑">
                                        <i class="fa fa-edit"></i>
                                    </a>
                                    <button class="delete-btn text-secondary hover:text-secondary/80 transition-colors p-1" 
                                            data-id="<?php echo $account['Id']; ?>"
                                            title="删除">
                                        <i class="fa fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                            
                            <!-- 卡片内容 -->
                            <div class="space-y-3 text-sm">
                                <div class="flex items-center">
                                    <i class="fa fa-comments-o text-primary w-6"></i>
                                    <span class="text-neutral-light ml-2">会话ID：</span>
                                    <span class="text-neutral truncate ml-1 flex-1"><?php echo htmlspecialchars($account['ChatId']); ?></span>
                                </div>
                                
                                <div class="flex items-center">
                                    <i class="fa fa-robot text-primary w-6"></i>
                                    <span class="text-neutral-light ml-2">机器人：</span>
                                    <span class="text-neutral truncate ml-1 flex-1"><?php echo htmlspecialchars($botName); ?></span>
                                </div>
                                
                                <div class="flex items-center">
                                    <i class="fa fa-server text-primary w-6"></i>
                                    <span class="text-neutral-light ml-2">服务器：</span>
                                    <span class="text-neutral truncate ml-1 flex-1"><?php echo htmlspecialchars($account['SiteUrl'] ?? $serverIP); ?></span>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <!-- 添加/编辑飞机群模态框 -->
    <div id="accountModal" class="fixed inset-0 bg-black/50 z-50 hidden flex items-center justify-center p-4">
        <div class="bg-white rounded-xl w-full max-w-md overflow-hidden transform transition-all duration-300 scale-95 opacity-0" id="modalContent">
            <div class="p-6">
                <div class="flex justify-between items-center mb-4">
                    <h3 id="modalTitle" class="text-lg font-bold text-neutral">添加飞机群</h3>
                    <button id="closeModal" class="text-neutral-light hover:text-neutral transition-colors">
                        <i class="fa fa-times"></i>
                    </button>
                </div>
                
                <form id="accountForm" method="post">
                    <input type="hidden" name="action" value="add">
                    <input type="hidden" name="id" id="accountId" value="">
                    
                    <div class="space-y-4">
                        <div>
                            <label for="AccountNo" class="block text-sm font-medium text-neutral-light mb-1">群编号 <span class="text-secondary">*</span></label>
                            <input type="text" id="AccountNo" name="AccountNo" 
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary transition-colors"
                                placeholder="请输入群编号（英文+数字）"
                                value="<?php echo $editAccount ? htmlspecialchars($editAccount['AccountNo']) : ''; ?>">
                            <p class="text-xs text-neutral-lighter mt-1">只能包含英文和数字</p>
                        </div>
                        
                        <div>
                            <label for="ChatId" class="block text-sm font-medium text-neutral-light mb-1">飞机群会话ID <span class="text-secondary">*</span></label>
                            <input type="text" id="ChatId" name="ChatId" 
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary transition-colors"
                                placeholder="请输入飞机群会话ID"
                                value="<?php echo $editAccount ? htmlspecialchars($editAccount['ChatId']) : ''; ?>">
                        </div>
                        
                        <div>
                            <label for="BotToken" class="block text-sm font-medium text-neutral-light mb-1">群机器人 <span class="text-secondary">*</span></label>
                            <select id="BotToken" name="BotToken" 
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary transition-colors">
                                <option value="">请选择机器人</option>
                                <?php foreach ($bots as $bot): ?>
                                    <option value="<?php echo htmlspecialchars($bot['Token']); ?>" 
                                        <?php echo ($editAccount && $editAccount['BotToken'] == $bot['Token']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($bot['Name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div>
                            <label for="SiteUrl" class="block text-sm font-medium text-neutral-light mb-1">服务器地址</label>
                            <input type="text" id="SiteUrl" name="SiteUrl" 
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg bg-neutral-lightest/50 cursor-not-allowed"
                                value="<?php echo $editAccount ? htmlspecialchars($editAccount['SiteUrl']) : htmlspecialchars($serverIP); ?>"
                                readonly>
                            <p class="text-xs text-neutral-lighter mt-1">由系统自动生成，不可修改</p>
                        </div>
                        
                        <div class="pt-2">
                            <button type="submit" class="w-full bg-primary hover:bg-primary/90 text-white font-medium py-2 px-4 rounded-lg transition-colors">
                                <?php echo $editAccount ? '保存修改' : '添加'; ?>
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- 测试发送消息模态框 -->
    <div id="testModal" class="fixed inset-0 bg-black/50 z-50 hidden flex items-center justify-center p-4">
        <div class="bg-white rounded-xl w-full max-w-md overflow-hidden transform transition-all duration-300 scale-95 opacity-0" id="testModalContent">
            <div class="p-6">
                <div class="flex justify-between items-center mb-4">
                    <h3 class="text-lg font-bold text-neutral">发送测试消息</h3>
                    <button id="closeTestModal" class="text-neutral-light hover:text-neutral transition-colors">
                        <i class="fa fa-times"></i>
                    </button>
                </div>
                
                <form id="testForm" method="post">
                    <input type="hidden" name="action" value="test">
                    <input type="hidden" name="botToken" id="testBotToken" value="">
                    <input type="hidden" name="chatId" id="testChatId" value="">
                    
                    <div class="space-y-4">
                        <div>
                            <label for="testMessage" class="block text-sm font-medium text-neutral-light mb-1">消息内容</label>
                            <textarea id="testMessage" name="message" rows="4"
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary transition-colors"
                                placeholder="请输入要发送的测试消息...">这是一条来自系统的测试消息，用于验证机器人和会话ID是否配置正确。</textarea>
                        </div>
                        
                        <!-- 发送结果区域 -->
                        <div id="testResult" class="hidden p-3 rounded-lg text-sm"></div>
                        
                        <div class="pt-2">
                            <button type="submit" id="sendTestBtn" class="w-full bg-primary hover:bg-primary/90 text-white font-medium py-2 px-4 rounded-lg transition-colors flex items-center justify-center">
                                <i class="fa fa-paper-plane mr-2"></i> 发送测试消息
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- 删除确认模态框 -->
    <div id="deleteModal" class="fixed inset-0 bg-black/50 z-50 hidden flex items-center justify-center p-4">
        <div class="bg-white rounded-xl w-full max-w-md overflow-hidden transform transition-all duration-300 scale-95 opacity-0" id="deleteModalContent">
            <div class="p-6 text-center">
                <div class="inline-flex items-center justify-center w-16 h-16 rounded-full bg-secondary/10 text-secondary mb-4 mx-auto">
                    <i class="fa fa-exclamation-triangle text-2xl"></i>
                </div>
                <h3 class="text-lg font-bold text-neutral mb-2">确认删除</h3>
                <p class="text-neutral-light mb-6">您确定要删除这条记录吗？此操作不可撤销。</p>
                
                <form id="deleteForm" method="post">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" id="deleteId" value="">
                    
                    <div class="flex space-x-3">
                        <button type="button" id="cancelDelete" class="flex-1 px-4 py-2 bg-neutral-lightest hover:bg-gray-200 text-neutral font-medium rounded-lg transition-colors">
                            取消
                        </button>
                        <button type="submit" class="flex-1 px-4 py-2 bg-secondary hover:bg-secondary/90 text-white font-medium rounded-lg transition-colors">
                            确认删除
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- 底部导航 -->
    <nav class="fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 py-2 px-4 z-20">
        <div class="flex justify-around">
            <a href="bet_list.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-home text-lg"></i>
                <span class="text-xs mt-1">首页</span>
            </a>
            <a href="bet_good.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-line-chart text-lg"></i>
                <span class="text-xs mt-1">数据</span>
            </a>
            <a href="bet_result.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-trophy text-lg"></i>
                <span class="text-xs mt-1">结果</span>
            </a>
            <a href="bet_notice.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-bell text-lg"></i>
                <span class="text-xs mt-1">通知</span>
            </a>
            <a href="bet_my.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-secondary">
                <i class="fa fa-user text-lg"></i>
                <span class="text-xs mt-1">我的</span>
            </a>
        </div>
    </nav>

    <script>
        // DOM元素
        const addBtn = document.getElementById('addBtn');
        const accountModal = document.getElementById('accountModal');
        const modalContent = document.getElementById('modalContent');
        const closeModal = document.getElementById('closeModal');
        const accountForm = document.getElementById('accountForm');
        const deleteBtns = document.querySelectorAll('.delete-btn');
        const deleteModal = document.getElementById('deleteModal');
        const deleteModalContent = document.getElementById('deleteModalContent');
        const cancelDelete = document.getElementById('cancelDelete');
        const deleteForm = document.getElementById('deleteForm');
        const deleteId = document.getElementById('deleteId');
        const closeNotices = document.querySelectorAll('.close-notice');
        const refreshBtn = document.getElementById('refreshBtn');
        const successNotice = document.getElementById('successNotice');
        const errorNotice = document.getElementById('errorNotice');
        // 测试相关元素
        const testBtns = document.querySelectorAll('.test-btn');
        const testModal = document.getElementById('testModal');
        const testModalContent = document.getElementById('testModalContent');
        const closeTestModal = document.getElementById('closeTestModal');
        const testForm = document.getElementById('testForm');
        const testBotToken = document.getElementById('testBotToken');
        const testChatId = document.getElementById('testChatId');
        const testMessage = document.getElementById('testMessage');
        const testResult = document.getElementById('testResult');
        const sendTestBtn = document.getElementById('sendTestBtn');

        // 自动关闭通知
        function autoCloseNotice(notice, delay = 3000) {
            if (notice) {
                setTimeout(() => {
                    notice.style.opacity = '0';
                    setTimeout(() => notice.style.display = 'none', 300);
                }, delay);
            }
        }

        // 初始化自动关闭通知
        document.addEventListener('DOMContentLoaded', () => {
            autoCloseNotice(successNotice);
            autoCloseNotice(errorNotice);
        });

        // 显示模态框
        function showModal(modal, content) {
            modal.classList.remove('hidden');
            // 阻止背景滚动
            document.body.style.overflow = 'hidden';
            setTimeout(() => {
                content.classList.remove('scale-95', 'opacity-0');
                content.classList.add('scale-100', 'opacity-100');
            }, 10);
        }

        // 隐藏模态框
        function hideModal(modal, content) {
            content.classList.remove('scale-100', 'opacity-100');
            content.classList.add('scale-95', 'opacity-0');
            // 恢复背景滚动
            document.body.style.overflow = '';
            setTimeout(() => {
                modal.classList.add('hidden');
            }, 300);
        }

        // 添加按钮点击事件
        if (addBtn) {
            addBtn.addEventListener('click', function() {
                // 重置表单为添加模式
                document.querySelector('input[name="action"]').value = 'add';
                document.getElementById('modalTitle').textContent = '添加飞机群';
                document.getElementById('accountId').value = '';
                accountForm.reset();
                
                showModal(accountModal, modalContent);
            });
        }

        // 关闭模态框
        closeModal.addEventListener('click', () => hideModal(accountModal, modalContent));

        // 测试按钮点击事件
        testBtns.forEach(btn => {
            btn.addEventListener('click', function() {
                const botToken = this.getAttribute('data-bot-token');
                const chatId = this.getAttribute('data-chat-id');
                
                // 设置测试表单数据
                testBotToken.value = botToken;
                testChatId.value = chatId;
                testResult.classList.add('hidden');
                
                showModal(testModal, testModalContent);
            });
        });

        // 关闭测试模态框
        closeTestModal.addEventListener('click', () => hideModal(testModal, testModalContent));

        // 测试表单提交
        testForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            // 显示加载状态
            sendTestBtn.disabled = true;
            sendTestBtn.innerHTML = '<i class="fa fa-spinner fa-spin mr-2"></i> 发送中...';
            testResult.classList.add('hidden');
            
            try {
                const formData = new FormData(this);
                
                // 发送请求
                const response = await fetch('', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                // 显示结果
                testResult.classList.remove('hidden', 'bg-success/10', 'text-success', 'bg-secondary/10', 'text-secondary');
                
                if (result.success) {
                    testResult.classList.add('bg-success/10', 'text-success');
                    testResult.innerHTML = `
                        <div class="flex items-start">
                            <i class="fa fa-check-circle mt-0.5 mr-2"></i>
                            <div>
                                <div>${result.message}</div>
                                <div class="mt-1 text-xs opacity-80">${result.details}</div>
                            </div>
                        </div>
                    `;
                } else {
                    testResult.classList.add('bg-secondary/10', 'text-secondary');
                    testResult.innerHTML = `
                        <div class="flex items-start">
                            <i class="fa fa-exclamation-circle mt-0.5 mr-2"></i>
                            <div>
                                <div>${result.message}</div>
                                <div class="mt-1 text-xs opacity-80">${result.details}</div>
                            </div>
                        </div>
                    `;
                }
                
                // 滚动到结果区域
                testResult.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                testResult.classList.add('fade-in');
                
            } catch (error) {
                testResult.classList.remove('hidden');
                testResult.classList.add('bg-secondary/10', 'text-secondary');
                testResult.innerHTML = `
                    <div class="flex items-start">
                        <i class="fa fa-exclamation-circle mt-0.5 mr-2"></i>
                        <div>请求失败，请稍后重试</div>
                    </div>
                `;
            } finally {
                // 恢复按钮状态
                sendTestBtn.disabled = false;
                sendTestBtn.innerHTML = '<i class="fa fa-paper-plane mr-2"></i> 发送测试消息';
            }
        });

        // 删除按钮点击事件
        deleteBtns.forEach(btn => {
            btn.addEventListener('click', function() {
                const id = this.getAttribute('data-id');
                deleteId.value = id;
                showModal(deleteModal, deleteModalContent);
            });
        });

        // 取消删除
        cancelDelete.addEventListener('click', () => hideModal(deleteModal, deleteModalContent));

        // 关闭提示信息
        closeNotices.forEach(btn => {
            btn.addEventListener('click', function() {
                const notice = this.closest('div');
                notice.style.opacity = '0';
                setTimeout(() => notice.style.display = 'none', 300);
            });
        });

        // 刷新页面
        refreshBtn.addEventListener('click', () => window.location.reload());

        // 页面加载时，如果是编辑状态，自动打开模态框
        <?php if ($editAccount): ?>
            document.addEventListener('DOMContentLoaded', function() {
                // 设置表单为编辑模式
                document.querySelector('input[name="action"]').value = 'edit';
                document.getElementById('modalTitle').textContent = '修改飞机群';
                document.getElementById('accountId').value = '<?php echo $editAccount['Id']; ?>';
                
                showModal(accountModal, modalContent);
            });
        <?php endif; ?>

        // 点击模态框背景关闭
        accountModal.addEventListener('click', e => {
            if (e.target === accountModal) hideModal(accountModal, modalContent);
        });

        deleteModal.addEventListener('click', e => {
            if (e.target === deleteModal) hideModal(deleteModal, deleteModalContent);
        });

        testModal.addEventListener('click', e => {
            if (e.target === testModal) hideModal(testModal, testModalContent);
        });

        // 表单提交验证
        accountForm.addEventListener('submit', function(e) {
            const accountNo = document.getElementById('AccountNo').value.trim();
            const chatId = document.getElementById('ChatId').value.trim();
            const botToken = document.getElementById('BotToken').value;
            
            if (!accountNo) {
                e.preventDefault();
                alert('请输入群编号');
                return;
            }
            
            if (!/^[A-Za-z0-9]+$/.test(accountNo)) {
                e.preventDefault();
                alert('群编号只能包含英文和数字');
                return;
            }
            
            if (!chatId) {
                e.preventDefault();
                alert('请输入飞机群会话ID');
                return;
            }
            
            if (!botToken) {
                e.preventDefault();
                alert('请选择群机器人');
                return;
            }
        });
    </script>
</body>
</html>