<?php
// 配置数据库连接
require_once 'db.php';

// 全局配置（固定自动执行参数，无需手动设置）
const PUSH_INTERVAL = 2; // 投注推送间隔(秒) - 固定
const BOT_CHECK_INTERVAL = 30; // 机器人状态检查间隔(秒) - 固定
const SUMMARY_PUSH_INTERVAL = 10800; // 汇总推送间隔(秒) 3小时=3*3600=10800 - 固定
const MAX_RETRIES = 1;
const RETRY_DELAY = 2; // 秒

// 初始化会话
session_start();

// CSRF令牌处理
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function verifyCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// 错误处理
function handleError($message, $isFatal = false, $logOnly = false) {
    $errorData = [
        'error' => true,
        'message' => $message,
        'timestamp' => date('Y-m-d H:i:s')
    ];
    
    error_log("系统错误: " . $message);
    
    if ($logOnly) return null;
    
    if ($isFatal) {
        header('Content-Type: application/json');
        http_response_code(500);
        echo json_encode($errorData);
        exit;
    }
    return $errorData;
}

// 数据库连接
try {
    $pdo = new PDO("mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
    $pdo->setAttribute(PDO::ATTR_TIMEOUT, 10);
} catch(PDOException $e) {
    handleError("数据库连接失败: " . $e->getMessage(), true);
}

// 账号验证
function authenticateAccount($pdo, $accountNo) {
    try {
        $stmt = $pdo->prepare("SELECT AccountNo FROM TelSms_Account WHERE AccountNo = :accountNo");
        $stmt->execute(['accountNo' => $accountNo]);
        return $stmt->fetch() !== false;
    } catch (Exception $e) {
        handleError("账号验证失败: " . $e->getMessage(), true);
        return false;
    }
}

// 获取账号参数
$accountNo = isset($_GET['AccountNo']) ? trim($_GET['AccountNo']) : '';
if ($accountNo && !authenticateAccount($pdo, $accountNo)) {
    handleError("账号验证失败，无权访问", true);
}

// 获取账号配置
function getAccountConfig($pdo, $accountNo) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM TelSms_Account WHERE AccountNo = :accountNo");
        $stmt->execute(['accountNo' => $accountNo]);
        $result = $stmt->fetch();
        if (!$result) throw new Exception("未找到账号配置: $accountNo");
        return $result;
    } catch (Exception $e) {
        throw new Exception("获取账号配置失败: " . $e->getMessage());
    }
}

// 获取管理员配置（用于有效期判断）
function getAdminConfig($pdo, $username) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM TelSms_Admin WHERE UserName = :username");
        $stmt->execute(['username' => $username]);
        $result = $stmt->fetch();
        if (!$result) throw new Exception("未找到管理员配置: $username");
        return $result;
    } catch (Exception $e) {
        throw new Exception("获取管理员配置失败: " . $e->getMessage());
    }
}

// 获取待推送数据
function getPendingBetData($pdo, $accountNo) {
    try {
        $stmt = $pdo->prepare("
            SELECT *
            FROM TelSms_BetList 
            WHERE AccountNo = :accountNo AND smsStatus = 0 
            ORDER BY BetTime ASC 
            LIMIT 1
        ");
        $stmt->execute(['accountNo' => $accountNo]);
        return $stmt->fetch();
    } catch (Exception $e) {
        throw new Exception("获取待推送数据失败: " . $e->getMessage());
    }
}

// 记录推送日志
function logPushResult($pdo, $accountNo, $title, $content, $success, $message = '') { 
    try { 
        $logContent = json_encode([ 
            'title' => $title, 
            'content' => $content, 
            'status' => $success ? '成功' : '失败', 
            'message' => $message,
            'timestamp' => date('Y-m-d H:i:s')
        ], JSON_UNESCAPED_UNICODE); 

        $stmt = $pdo->prepare(" 
            INSERT INTO TelSms_Logs (AccountNo, Contents, CreateDate) 
            VALUES (:accountNo, :contents, NOW()) 
        "); 
        $stmt->bindParam(':accountNo', $accountNo, PDO::PARAM_STR); 
        $stmt->bindParam(':contents', $logContent, PDO::PARAM_STR); 
        return $stmt->execute(); 
    } catch (Exception $e) { 
        throw new Exception("记录推送日志失败: " . $e->getMessage()); 
    } 
}

// 更新数据状态
function updateBetStatus($pdo, $id, $status) {
    try {
        $pdo->beginTransaction();
        $stmt = $pdo->prepare("
            UPDATE TelSms_BetList 
            SET smsStatus = :status, SendTime = NOW() 
            WHERE Id = :id AND smsStatus = 0
        ");
        $result = $stmt->execute(['status' => $status, 'id' => $id]);
        if ($result && $stmt->rowCount() === 0) {
            throw new Exception("数据状态已更新或不存在，ID: $id");
        }
        $pdo->commit();
        return $result && $stmt->rowCount() > 0;
    } catch (Exception $e) {
        $pdo->rollBack();
        throw new Exception("更新数据状态失败: " . $e->getMessage());
    }
}

// 发送到Telegram
function sendToTelegram($botToken, $chatId, $content) {
    $url = "https://api.telegram.org/bot$botToken/sendMessage";
    $data = [
        'chat_id' => $chatId,
        'text' => $content,
        'parse_mode' => 'HTML'
    ];

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($data),
        CURLOPT_TIMEOUT => 10,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);

    if ($error) {
        return ['ok' => false, 'description' => "请求失败: $error"];
    }
    if ($httpCode !== 200) {
        return ['ok' => false, 'description' => "状态码: $httpCode, 响应: $response"];
    }

    $result = json_decode($response, true);
    return $result ?: ['ok' => false, 'description' => "解析响应失败: $response"];
}

// 带重试的发送
function sendToTelegramWithRetry($botToken, $chatId, $content) {
    for ($i = 0; $i < MAX_RETRIES; $i++) {
        $result = sendToTelegram($botToken, $chatId, $content);
        if ($result['ok']) return $result;
        sleep(RETRY_DELAY);
    }
    return $result;
}

// 构建推送内容（含有效期检查）
function buildPushContent($accountConfig, $betData, $adminConfig) {
    // 创建当前时间对象并设置为北京时间
    $now = new DateTime('now', new DateTimeZone('Asia/Shanghai'));
    $startDate = new DateTime($adminConfig['StartDate']);
    $endDate = new DateTime($adminConfig['EndDate']);

   

    // 正常投注内容
    if($betData && $betData['BetContent']) {
        // 账号过期检查
        if ($now < $startDate || $now > $endDate) {
            return "⚠️ 账号已过期\n群编号：{$accountConfig['AccountNo']}\n软件账号：{$accountConfig['UserName']}\n有效期：{$adminConfig['StartDate']} - {$adminConfig['EndDate']}\n\n当前时间：{$now->format('Y-m-d H:i:s')}";
        }
        // 提取最后一条记录
        $text = $betData['BetContent'];
        $pattern = '/时间:  \d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}/';
        preg_match_all($pattern, $text, $matches, PREG_OFFSET_CAPTURE);

        if (!empty($matches[0])) {
            $lastPos = end($matches[0])[1];
            $lastRecord = substr($text, $lastPos);
            $text = $lastRecord;
        } 
        
        $betno = substr($betData['BetNo'], -4);
        return "{$betData['HgAdmin']}监控 会员:{$betData['BetAccount']} 单号:***{$betno} \n{$text}"
        ."\n\n数据中台:<a href=\"{$accountConfig['SiteUrl']}/bet_list.php?AccountNo={$accountConfig['AccountNo']}\">点击查看</a>";
    }

    return null; // 无数据时返回null
}

// 机器人状态检查与更新（全自动执行核心函数）
function checkAndUpdateBotStatus($pdo, $username) {
    try {
        $twoMinutesAgo = date('Y-m-d H:i:s', strtotime('-2 minutes'));
        
        // 查询需要检查的机器人：AdminUserName匹配且状态为2且超过2分钟未更新
        $stmt = $pdo->prepare("
            SELECT * FROM TelSms_BotConfig 
            WHERE AdminUserName = :username AND Status = 2 
            AND UpdateTime < :twoMinutesAgo
        ");
        $stmt->execute([
            'username' => $username,
            'twoMinutesAgo' => $twoMinutesAgo
        ]);
        $bots = $stmt->fetchAll();
        
        if (empty($bots)) return [];
        
        // 更新状态为1（异常）
        $botIds = array_column($bots, 'Id');
        $stmt = $pdo->prepare("
            UPDATE TelSms_BotConfig 
            SET Status = 1, UpdateTime = NOW() 
            WHERE Id IN (" . implode(',', array_fill(0, count($botIds), '?')) . ")
        ");
        $stmt->execute($botIds);
        
        return $bots;
    } catch (Exception $e) {
        throw new Exception("机器人状态检查失败: " . $e->getMessage());
    }
}

// 获取所有机器人状态（用于汇总）
function getAllBotStatus($pdo, $username) {
    try {
        $stmt = $pdo->prepare("
            SELECT * FROM TelSms_BotConfig 
            WHERE AdminUserName = :username
        ");
        $stmt->execute(['username' => $username]);
        return $stmt->fetchAll();
    } catch (Exception $e) {
        throw new Exception("获取机器人状态失败: " . $e->getMessage());
    }
}

// 构建机器人状态汇总内容
function buildBotSummaryContent($bots, $username) {
    $total = count($bots);
    $active = 0;
    $inactive = 0;
    $unknown = 0;
    
    foreach ($bots as $bot) {
        switch ($bot['Status']) {
            case 1: $inactive++; break;
            case 2: $active++; break;
            default: $unknown++;
        }
    }
    
    $now = new DateTime('now', new DateTimeZone('Asia/Shanghai'));
    return "🤖 机器人状态汇总 ({$username})\n" .
           "📅 时间: {$now->format('Y-m-d H:i:s')}\n" .
           "📊 总数: {$total}\n" .
           "✅ 正常运行: {$active}\n" .
           "❌ 已停止-空闲: {$inactive}\n" .
           "❓ 机器异常-需重置: {$unknown}";
}

// API请求处理
if (isset($_GET['api'])) {
    header('Content-Type: application/json');
    try {
        switch ($_GET['api']) {
            case 'switch-account':
                $accountNo = $_GET['accountNo'] ?? '';
                if (!$accountNo) throw new Exception("账号不能为空");
                if (!authenticateAccount($pdo, $accountNo)) throw new Exception("账号验证失败");
                
                $config = getAccountConfig($pdo, $accountNo);
                $adminConfig = getAdminConfig($pdo, $config['UserName']);
                $betData = getPendingBetData($pdo, $accountNo);
                
                echo json_encode([
                    'success' => true,
                    'account' => $config,
                    'admin' => $adminConfig,
                    'betData' => $betData
                ]);
                exit;

            case 'fetch-data':
                $accountNo = $_GET['accountNo'] ?? '';
                if (!$accountNo || !authenticateAccount($pdo, $accountNo)) throw new Exception("无权访问");
                
                $config = getAccountConfig($pdo, $accountNo);
                $adminConfig = getAdminConfig($pdo, $config['UserName']);
                $betData = getPendingBetData($pdo, $accountNo);
                
                echo json_encode([
                    'success' => true,
                    'betData' => $betData,
                    'timestamp' => date('Y-m-d H:i:s')
                ]);
                exit;

            case 'push-data':
                if ($_SERVER['REQUEST_METHOD'] !== 'POST') throw new Exception("请使用POST请求");

                $accountNo = $_GET['accountNo']?? '';
                $title = $_GET['title']?? '定时推送';
                $csrfToken = $_GET['csrf_token']?? '';
                
                if (!verifyCSRFToken($csrfToken)) throw new Exception("CSRF验证失败");
                if (!$accountNo) throw new Exception("缺少账号");
                
                $config = getAccountConfig($pdo, $accountNo);
                $adminConfig = getAdminConfig($pdo, $config['UserName']);
                $betData = getPendingBetData($pdo, $accountNo);
                $content = buildPushContent($config, $betData, $adminConfig);

                // 无新投注数据时不执行任何操作
                if ($content === null && empty($betData)) {
                    $newBetData = getPendingBetData($pdo, $accountNo);
                    echo json_encode([
                        'success' => true,
                        'message' => '无新投注数据，无需推送',
                        'newBetData' => $newBetData
                    ]);
                    exit;
                }

                $result = sendToTelegramWithRetry($config['BotToken'], $config['ChatId'], $content);
                $success = $result['ok'];
                $message = $success ? '推送成功' : ($result['description'] ?? '未知错误');

                // 记录日志
                $logContents = $content;
                if (!empty($betData)) {
                    $logContents = "AccountNo:{$accountNo}; BetAccount:{$betData['BetAccount']}; BetNo:{$betData['BetNo']}; Content:{$content}";
                    $title = "投注推送:{$betData['BetNo']}";
                }
                logPushResult($pdo, $accountNo, $title, $logContents, $success, $message);

                // 更新投注状态
                if ($success && $betData) {
                    $updateResult = updateBetStatus($pdo, $betData['Id'], 1);
                    if (!$updateResult) {
                        logPushResult($pdo, $accountNo, "状态更新失败", $content, false, "更新TelSms_BetList状态失败，ID: {$betData['Id']}");
                    }
                }

                $newBetData = getPendingBetData($pdo, $accountNo);
                echo json_encode([
                    'success' => $success,
                    'message' => $message,
                    'newBetData' => $newBetData
                ]);
                exit;

            // 机器人状态自动检查接口
            case 'auto-check-bots':
                $accountNo = $_GET['accountNo'] ?? '';
                if (!$accountNo || !authenticateAccount($pdo, $accountNo)) throw new Exception("无权访问");
                
                $config = getAccountConfig($pdo, $accountNo);
                $updatedBots = checkAndUpdateBotStatus($pdo, $config['UserName']);
                
                // 记录机器人状态更新日志
                if (!empty($updatedBots)) {
                    $botIds = array_column($updatedBots, 'Id');
                    logPushResult(
                        $pdo, 
                        $accountNo, 
                        "机器人状态自动更新", 
                        "更新机器人ID: " . implode(',', $botIds), 
                        true, 
                        "共更新" . count($updatedBots) . "个机器人状态"
                    );
                }
                
                echo json_encode([
                    'success' => true,
                    'updatedCount' => count($updatedBots),
                    'timestamp' => date('Y-m-d H:i:s')
                ]);
                exit;

            // 机器人状态汇总自动推送接口
            case 'auto-push-summary':
                $accountNo = $_GET['accountNo'] ?? '';
                if (!$accountNo || !authenticateAccount($pdo, $accountNo)) throw new Exception("无权访问");
                
                $config = getAccountConfig($pdo, $accountNo);
                $bots = getAllBotStatus($pdo, $config['UserName']);
                $content = buildBotSummaryContent($bots, $config['UserName']);
                
                $result = sendToTelegramWithRetry($config['BotToken'], $config['ChatId'], $content);
                $success = $result['ok'];
                
                logPushResult($pdo, $accountNo, "机器人状态汇总（自动）", $content, $success, $result['description'] ?? '');
                
                echo json_encode([
                    'success' => $success,
                    'message' => $success ? '汇总推送成功' : ($result['description'] ?? '推送失败'),
                    'botCount' => count($bots)
                ]);
                exit;

            case 'push-logs':
                $accountNo = $_GET['accountNo'] ?? '';
                if (!$accountNo || !authenticateAccount($pdo, $accountNo)) throw new Exception("无权访问");
                
                $stmt = $pdo->prepare("
                    SELECT Id, AccountNo, Contents, CreateDate 
                    FROM TelSms_Logs 
                    WHERE AccountNo = :accountNo 
                    ORDER BY CreateDate DESC 
                    LIMIT 10
                ");
                $stmt->execute(['accountNo' => $accountNo]);
                echo json_encode(['success' => true, 'data' => $stmt->fetchAll()]);
                exit;

            case 'push-log-detail':
                $id = (int)($_GET['id'] ?? 0);
                if (!$id) throw new Exception("缺少id参数");
                
                $stmt = $pdo->prepare("
                    SELECT l.*, a.UserName 
                    FROM TelSms_Logs l
                    JOIN TelSms_Account a ON l.AccountNo = a.AccountNo
                    WHERE l.Id = :id
                ");
                $stmt->execute(['id' => $id]);
                $log = $stmt->fetch();
                if (!$log) throw new Exception("日志不存在");
                if ($accountNo && $log['AccountNo'] !== $accountNo) throw new Exception("无权访问");
                
                echo json_encode(['success' => true, 'data' => $log]);
                exit;

            case 'test-connection':
                if ($_SERVER['REQUEST_METHOD'] !== 'POST') throw new Exception("请使用POST请求");
                $accountNo = $_GET['accountNo'] ?? '';
                if (!$accountNo || !authenticateAccount($pdo, $accountNo)) throw new Exception("无权访问");
                
                $config = getAccountConfig($pdo, $accountNo);
                $content = "✅ 测试连接成功\n账号: {$accountNo}\n时间: " . date('Y-m-d H:i:s');
                $result = sendToTelegram($config['BotToken'], $config['ChatId'], $content);
                
                echo json_encode([
                    'success' => $result['ok'],
                    'error' => $result['ok'] ? '' : $result['description']
                ]);
                exit;

            default:
                throw new Exception("未知API端点");
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        exit;
    }
}

// 加载账号配置和待推送数据
$accountConfig = null;
$adminConfig = null;
$pendingBetData = null;
$errorMessage = '';
try {
    if ($accountNo) {
        $accountConfig = getAccountConfig($pdo, $accountNo);
        $adminConfig = getAdminConfig($pdo, $accountConfig['UserName']);
        $pendingBetData = getPendingBetData($pdo, $accountNo);
    }
} catch (Exception $e) {
    $errorMessage = $e->getMessage();
}

$csrfToken = generateCSRFToken();
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>全自动Telegram推送系统</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdn.jsdelivr.net/npm/font-awesome@4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#3b82f6',
                        success: '#10b981',
                        danger: '#ef4444',
                        dark: '#1e293b',
                    }
                }
            }
        }
    </script>
    <style type="text/tailwindcss">
        @layer utilities {
            .card-shadow { box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1); }
            .btn-hover { 
                transition: all 0.3s; 
                @apply hover:shadow-lg hover:-translate-y-0.5;
            }
            .input-focus { 
                @apply focus:ring-2 focus:ring-primary/50 focus:border-primary transition duration-200;
            }
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen flex flex-col">
    <header class="bg-white shadow-md sticky top-0 z-50">
        <div class="container mx-auto px-4 py-3 flex justify-between items-center">
            <div class="flex items-center space-x-2">
                <i class="fa fa-paper-plane text-primary text-2xl"></i>
                <h1 class="text-xl font-bold text-dark">全自动Telegram推送系统</h1>
            </div>
            <div class="flex items-center space-x-4">
                <span class="text-sm text-gray-600" id="current-account">当前账号: <?php echo htmlspecialchars($accountConfig['UserName'] ?? '未选择'); ?></span>
                <span class="text-xs px-2 py-1 rounded-full bg-green-100 text-green-800 hidden" id="auto-status">
                    <i class="fa fa-refresh fa-spin mr-1"></i>自动运行中
                </span>
            </div>
        </div>
    </header>

    <main class="flex-grow container mx-auto px-4 py-6 grid grid-cols-1 lg:grid-cols-3 gap-6">
        <!-- 左侧配置面板 -->
        <div class="lg:col-span-1 space-y-6">
            <div class="bg-white rounded-xl p-6 card-shadow">
                <h2 class="text-lg font-bold text-dark mb-4 flex items-center">
                    <i class="fa fa-user-circle text-primary mr-2"></i>选择账号
                </h2>
                <div class="flex">
                    <input type="text" id="account-input" placeholder="输入账号ID" 
                        class="flex-1 px-4 py-2 border border-gray-300 rounded-l-lg input-focus"
                        value="<?php echo htmlspecialchars($accountNo); ?>">
                    <button id="select-account" class="bg-primary text-white px-4 py-2 rounded-r-lg btn-hover">
                        <i class="fa fa-search mr-1"></i> 查找
                    </button>
                </div>
            </div>

            <div id="account-info" class="bg-white rounded-xl p-6 card-shadow hidden">
                <h2 class="text-lg font-bold text-dark mb-4 flex items-center">
                    <i class="fa fa-user-circle text-primary mr-2"></i>账号信息
                </h2>
                <div class="space-y-3">
                    <div class="grid grid-cols-2 gap-2">
                        <div><p class="text-xs text-gray-500">用户名</p><p class="font-medium" id="account-username">-</p></div>
                        <div><p class="text-xs text-gray-500">账号ID</p><p class="font-medium" id="account-id">-</p></div>
                    </div>
                    <div>
                        <p class="text-xs text-gray-500">有效期</p>
                        <p class="font-medium" id="account-validity">-</p>
                    </div>
                    <div class="mt-2 px-3 py-1 rounded-full text-sm font-medium" id="account-status">
                        <span class="bg-gray-100 text-gray-800">未加载</span>
                    </div>
                </div>
            </div>

            <div id="telegram-config" class="bg-white rounded-xl p-6 card-shadow hidden">
                <h2 class="text-lg font-bold text-dark mb-4 flex items-center">
                    <i class="fa fa-telegram text-primary mr-2"></i>Telegram配置
                </h2>
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Bot Token</label>
                        <input type="text" id="bot-token" class="w-full px-4 py-2 border border-gray-300 rounded-lg input-focus" disabled>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">聊天ID</label>
                        <input type="text" id="chat-id" class="w-full px-4 py-2 border border-gray-300 rounded-lg input-focus" disabled>
                    </div>
                    <button id="test-telegram" class="w-full bg-primary text-white px-4 py-2 rounded-lg flex items-center justify-center btn-hover">
                        <i class="fa fa-paper-plane-o mr-2"></i>测试连接
                    </button>
                </div>
            </div>

            <div class="bg-white rounded-xl p-6 card-shadow">
                <h2 class="text-lg font-bold text-dark mb-4 flex items-center">
                    <i class="fa fa-cogs text-primary mr-2"></i>自动执行状态
                </h2>
                <div class="space-y-4">
                    <div class="p-3 bg-gray-50 rounded-lg">
                        <p class="text-sm font-medium">投注推送间隔</p>
                        <p class="text-xs text-gray-500 mt-1"><?php echo PUSH_INTERVAL; ?>秒（自动执行）</p>
                    </div>
                    <div class="p-3 bg-gray-50 rounded-lg">
                        <p class="text-sm font-medium">机器人状态检查间隔</p>
                        <p class="text-xs text-gray-500 mt-1"><?php echo BOT_CHECK_INTERVAL; ?>秒（自动执行）</p>
                    </div>
                    <div class="p-3 bg-gray-50 rounded-lg">
                        <p class="text-sm font-medium">状态汇总推送间隔</p>
                        <p class="text-xs text-gray-500 mt-1"><?php echo (SUMMARY_PUSH_INTERVAL/3600); ?>小时（自动执行）</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- 右侧数据管理 -->
        <div class="lg:col-span-2 space-y-6">
            <div id="bet-data-container" class="bg-white rounded-xl p-6 card-shadow hidden">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-lg font-bold text-dark flex items-center">
                        <i class="fa fa-table text-primary mr-2"></i>待推送数据
                    </h2>
                    <div class="text-sm text-gray-500" id="last-refresh-time">上次刷新: --</div>
                </div>
                
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">内容</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">创建时间</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">状态</th>
                            </tr>
                        </thead>
                        <tbody id="data-table-body" class="bg-white divide-y divide-gray-200">
                            <tr><td colspan="3" class="px-6 py-12 text-center text-gray-500"><i class="fa fa-info-circle mr-2"></i>请选择账号加载数据</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <div id="logs-container" class="bg-white rounded-xl p-6 card-shadow hidden">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-lg font-bold text-dark flex items-center">
                        <i class="fa fa-history text-primary mr-2"></i>推送日志
                    </h2>
                </div>
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">推送时间</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">标题</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">状态</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">操作</th>
                            </tr>
                        </thead>
                        <tbody id="log-table-body" class="bg-white divide-y divide-gray-200">
                            <tr><td colspan="4" class="px-6 py-12 text-center text-gray-500"><i class="fa fa-info-circle mr-2"></i>请选择账号加载日志</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <div id="system-status" class="bg-white rounded-xl p-6 card-shadow hidden">
                <h2 class="text-lg font-bold text-dark mb-4 flex items-center">
                    <i class="fa fa-area-chart text-primary mr-2"></i>系统状态
                </h2>
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div class="p-4 bg-gray-50 rounded-lg border border-gray-200">
                        <p class="text-sm text-gray-500">总推送次数</p>
                        <p class="text-2xl font-bold text-primary mt-1" id="total-pushes">0</p>
                    </div>
                    <div class="p-4 bg-gray-50 rounded-lg border border-gray-200">
                        <p class="text-sm text-gray-500">成功推送</p>
                        <p class="text-2xl font-bold text-success mt-1" id="successful-pushes">0</p>
                    </div>
                    <div class="p-4 bg-gray-50 rounded-lg border border-gray-200">
                        <p class="text-sm text-gray-500">失败推送</p>
                        <p class="text-2xl font-bold text-danger mt-1" id="failed-pushes">0</p>
                    </div>
                </div>
                <div class="mt-4 p-4 bg-gray-50 rounded-lg border border-gray-200">
                    <p class="text-sm font-medium mb-2">最近活动</p>
                    <div id="recent-activities" class="text-sm text-gray-600 space-y-2 max-h-40 overflow-auto">
                        <p class="text-gray-400 italic">系统启动中...</p>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <footer class="bg-white border-t border-gray-200 py-4">
        <div class="container mx-auto px-4 text-center text-gray-500 text-sm">
            <p>© 2025 全自动Telegram推送系统 | 完全自动化运行</p>
        </div>
    </footer>

    <!-- 机器人状态模态框 -->
    <div id="bot-status-modal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
        <div class="bg-white rounded-lg p-6 max-w-2xl w-full max-h-[80vh] overflow-auto">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-bold">机器人状态</h3>
                <button id="close-bot-modal" class="text-gray-400 hover:text-gray-600"><i class="fa fa-times"></i></button>
            </div>
            <div id="bot-status-content" class="space-y-4">
                <div class="text-center text-gray-500">
                    <i class="fa fa-spinner fa-spin mr-2"></i>加载中...
                </div>
            </div>
        </div>
    </div>

    <!-- 日志详情模态框 -->
    <div id="log-detail-modal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
        <div class="bg-white rounded-lg p-6 max-w-2xl w-full max-h-[80vh] overflow-auto">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-bold">推送详情</h3>
                <button id="close-log-detail" class="text-gray-400 hover:text-gray-600"><i class="fa fa-times"></i></button>
            </div>
            <div id="log-detail-content" class="space-y-4"></div>
        </div>
    </div>

    <!-- 通知提示 -->
    <div id="notification" class="fixed top-4 right-4 max-w-sm bg-white rounded-lg shadow-lg p-4 transform transition-all duration-300 translate-x-full opacity-0 z-50 flex items-start">
        <div id="notification-icon" class="mr-3 text-xl"></div>
        <div class="flex-1">
            <h3 id="notification-title" class="font-medium text-gray-900"></h3>
            <p id="notification-message" class="text-sm text-gray-600 mt-1"></p>
        </div>
        <button id="close-notification" class="ml-2 text-gray-400 hover:text-gray-600"><i class="fa fa-times"></i></button>
    </div>

    <script>
        // 全局常量（固定间隔，不可修改）
        const PUSH_INTERVAL = <?php echo PUSH_INTERVAL; ?>; // 投注推送间隔(秒)
        const BOT_CHECK_INTERVAL = <?php echo BOT_CHECK_INTERVAL; ?>; // 机器人检查间隔(秒)
        const SUMMARY_PUSH_INTERVAL = <?php echo SUMMARY_PUSH_INTERVAL; ?>; // 汇总推送间隔(秒)
        
        // 全局变量
        let currentAccount = '<?php echo htmlspecialchars($accountNo); ?>';
        let pushTimer = null;         // 投注推送定时器
        let botCheckTimer = null;     // 机器人状态检查定时器
        let summaryPushTimer = null;  // 汇总推送定时器
        const csrfToken = '<?php echo htmlspecialchars($csrfToken); ?>';
        let activityLog = [];
        let pushStats = { total: 0, success: 0, failed: 0 };

        // DOM元素
        const elements = {
            accountInput: document.getElementById('account-input'),
            selectAccount: document.getElementById('select-account'),
            currentAccountEl: document.getElementById('current-account'),
            autoStatus: document.getElementById('auto-status'),
            accountInfo: document.getElementById('account-info'),
            accountUsername: document.getElementById('account-username'),
            accountId: document.getElementById('account-id'),
            accountValidity: document.getElementById('account-validity'),
            accountStatus: document.getElementById('account-status'),
            telegramConfig: document.getElementById('telegram-config'),
            botToken: document.getElementById('bot-token'),
            chatId: document.getElementById('chat-id'),
            testTelegram: document.getElementById('test-telegram'),
            lastRefreshTime: document.getElementById('last-refresh-time'),
            betDataContainer: document.getElementById('bet-data-container'),
            dataTableBody: document.getElementById('data-table-body'),
            logsContainer: document.getElementById('logs-container'),
            logTableBody: document.getElementById('log-table-body'),
            logDetailModal: document.getElementById('log-detail-modal'),
            closeLogDetail: document.getElementById('close-log-detail'),
            logDetailContent: document.getElementById('log-detail-content'),
            botStatusModal: document.getElementById('bot-status-modal'),
            closeBotModal: document.getElementById('close-bot-modal'),
            botStatusContent: document.getElementById('bot-status-content'),
            notification: document.getElementById('notification'),
            notificationIcon: document.getElementById('notification-icon'),
            notificationTitle: document.getElementById('notification-title'),
            notificationMessage: document.getElementById('notification-message'),
            closeNotification: document.getElementById('close-notification'),
            systemStatus: document.getElementById('system-status'),
            totalPushes: document.getElementById('total-pushes'),
            successfulPushes: document.getElementById('successful-pushes'),
            failedPushes: document.getElementById('failed-pushes'),
            recentActivities: document.getElementById('recent-activities')
        };

        // 页面加载完成后自动初始化
        document.addEventListener('DOMContentLoaded', () => {
            // 绑定事件
            elements.selectAccount.addEventListener('click', switchAccount);
            elements.accountInput.addEventListener('keypress', e => e.key === 'Enter' && switchAccount());
            elements.testTelegram.addEventListener('click', testTelegramConnection);
            elements.closeLogDetail.addEventListener('click', () => elements.logDetailModal.classList.add('hidden'));
            elements.closeBotModal.addEventListener('click', () => elements.botStatusModal.classList.add('hidden'));
            elements.closeNotification.addEventListener('click', hideNotification);

            // 页面加载后自动启动（如果有账号）
            if (currentAccount) {
                switchAccount();
            } else {
                showNotification('提示', '请选择账号以启动全自动模式', 'info');
            }
        });

        // 切换账号并自动启动所有定时任务
        function switchAccount() {
            const accountNo = elements.accountInput.value.trim();
            if (!accountNo) {
                showNotification('错误', '请输入账号ID', 'danger');
                return;
            }

            addActivity('切换账号中...');
            fetch(`?api=switch-account&accountNo=${encodeURIComponent(accountNo)}`)
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        currentAccount = accountNo;
                        // 更新UI显示
                        updateAccountUI(data.account, data.admin);
                        updateBetDataUI(data.betData);
                        
                        // 显示功能区域
                        elements.accountInfo.classList.remove('hidden');
                        elements.telegramConfig.classList.remove('hidden');
                        elements.betDataContainer.classList.remove('hidden');
                        elements.logsContainer.classList.remove('hidden');
                        elements.systemStatus.classList.remove('hidden');
                        
                        // 加载日志
                        loadPushLogs();
                        
                        // 启动所有自动任务（核心：全自动运行）
                        startAllAutoTasks();
                        
                        addActivity('账号切换完成，自动任务已启动');
                        showNotification('成功', '账号切换完成，自动任务已启动', 'success');
                    } else {
                        throw new Error(data.error || '切换账号失败');
                    }
                })
                .catch(err => {
                    addActivity(`切换账号失败: ${err.message}`, 'error');
                    showNotification('错误', err.message, 'danger');
                    console.error('切换账号失败:', err);
                });
        }

        // 启动所有自动任务（页面加载后自动执行）
        function startAllAutoTasks() {
            // 先停止已有定时器，避免重复
            stopAllAutoTasks();
            
            if (!currentAccount) return;
            
            // 1. 启动投注定时推送（每PUSH_INTERVAL秒）
            pushTimer = setInterval(() => {
                autoPushData();
            }, PUSH_INTERVAL * 1000);
            
            // 2. 启动机器人状态自动检查（每BOT_CHECK_INTERVAL秒）
            botCheckTimer = setInterval(() => {
                autoCheckBotStatus();
            }, BOT_CHECK_INTERVAL * 1000);
            
            // 3. 启动状态汇总自动推送（每SUMMARY_PUSH_INTERVAL秒）
            summaryPushTimer = setInterval(() => {
                autoPushBotSummary();
            }, SUMMARY_PUSH_INTERVAL * 1000);
            
            // 显示自动运行状态
            elements.autoStatus.classList.remove('hidden');
            
            addActivity(`全自动任务已启动: 投注推送(${PUSH_INTERVAL}秒), 机器人检查(${BOT_CHECK_INTERVAL}秒), 汇总推送(${SUMMARY_PUSH_INTERVAL/3600}小时)`);
            console.log(`全自动任务已启动:
                - 投注推送: ${PUSH_INTERVAL}秒
                - 机器人检查: ${BOT_CHECK_INTERVAL}秒
                - 汇总推送: ${SUMMARY_PUSH_INTERVAL/3600}小时`);
        }

        // 停止所有自动任务
        function stopAllAutoTasks() {
            if (pushTimer) clearInterval(pushTimer);
            if (botCheckTimer) clearInterval(botCheckTimer);
            if (summaryPushTimer) clearInterval(summaryPushTimer);
            pushTimer = null;
            botCheckTimer = null;
            summaryPushTimer = null;
            elements.autoStatus.classList.add('hidden');
            addActivity('所有自动任务已停止');
        }

        // 自动推送数据（无手动触发）
        function autoPushData() {
            if (!currentAccount) return;

            // 先检查是否有数据，无数据则不发送请求
            fetch(`?api=fetch-data&accountNo=${currentAccount}`)
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        // 更新数据显示
                        updateBetDataUI(data.betData);
                        elements.lastRefreshTime.textContent = `上次刷新: ${data.timestamp}`;
                        
                        // 无新投注数据时不执行推送
                        if (!data.betData) {
                            addActivity('无新投注数据，跳过推送');
                            return;
                        }
                        
                        // 有数据才执行推送
                        addActivity('发现新投注数据，准备推送');
                        return fetch(`?api=push-data&accountNo=${currentAccount}&title=定时推送&csrf_token=${csrfToken}`, {
                            method: 'POST'
                        });
                    }
                    throw new Error(data.error || '获取数据失败');
                })
                .then(res => res && res.json())
                .then(data => {
                    if (data && data.success) {
                        updateBetDataUI(data.newBetData);
                        loadPushLogs();
                        
                        // 更新统计数据
                        pushStats.total++;
                        if (data.message === '推送成功') {
                            pushStats.success++;
                            addActivity('投注数据推送成功');
                        } else {
                            addActivity(`推送结果: ${data.message}`);
                        }
                        updatePushStats();
                    } else if (data) {
                        pushStats.total++;
                        pushStats.failed++;
                        updatePushStats();
                        throw new Error(data.error || data.message || '推送失败');
                    }
                })
                .catch(err => {
                    if (!err.message.includes('无新投注数据')) {
                        pushStats.total++;
                        pushStats.failed++;
                        updatePushStats();
                        addActivity(`推送失败: ${err.message}`, 'error');
                    }
                    console.error('自动推送数据失败:', err);
                });
        }

        // 机器人状态自动检查（定时任务）
        function autoCheckBotStatus() {
            if (!currentAccount) return;

            addActivity('执行机器人状态检查');
            fetch(`?api=auto-check-bots&accountNo=${currentAccount}`)
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        if (data.updatedCount > 0) {
                            addActivity(`自动更新了 ${data.updatedCount} 个异常机器人状态`);
                            loadPushLogs(); // 刷新日志
                        } else {
                            addActivity('机器人状态检查完成，未发现异常');
                        }
                    } else {
                        throw new Error(data.error || '机器人状态检查失败');
                    }
                })
                .catch(err => {
                    addActivity(`机器人状态检查失败: ${err.message}`, 'error');
                    console.error('机器人自动检查失败:', err);
                });
        }

        // 机器人状态汇总自动推送（定时任务）
        function autoPushBotSummary() {
            if (!currentAccount) return;

            addActivity('执行机器人状态汇总推送');
            fetch(`?api=auto-push-summary&accountNo=${currentAccount}`, { method: 'POST' })
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        addActivity('机器人状态汇总已自动推送');
                        loadPushLogs(); // 刷新日志
                    } else {
                        throw new Error(data.message || '汇总推送失败');
                    }
                })
                .catch(err => {
                    addActivity(`汇总推送失败: ${err.message}`, 'error');
                    console.error('汇总自动推送失败:', err);
                });
        }

        // 更新账号信息UI
        function updateAccountUI(account, admin) {
            elements.currentAccountEl.textContent = `当前账号: ${account.UserName}`;
            elements.accountUsername.textContent = account.UserName;
            elements.accountId.textContent = account.AccountNo;
            elements.accountValidity.textContent = `${admin.StartDate} - ${admin.EndDate}`;
            elements.botToken.value = maskToken(account.BotToken);
            elements.chatId.value = account.ChatId || '';
            
            // 账号状态判断（有效期检查）
            const now = new Date();
            const start = new Date(admin.StartDate);
            const end = new Date(admin.EndDate);
            
            if (now < start) {
                elements.accountStatus.innerHTML = '<span class="bg-yellow-100 text-yellow-800">未生效</span>';
                addActivity('账号尚未生效', 'warning');
            } else if (now > end) {
                elements.accountStatus.innerHTML = '<span class="bg-red-100 text-red-800">已过期</span>';
                addActivity('账号已过期', 'error');
            } else {
                elements.accountStatus.innerHTML = '<span class="bg-green-100 text-green-800">正常</span>';
            }
        }

        // 掩码处理Token
        function maskToken(token) {
            if (!token) return '';
            return token.length > 10 ? `${token.substring(0, 6)}****${token.substring(token.length - 4)}` : token;
        }

        // 更新数据表格UI
        function updateBetDataUI(betData) {
            if (betData) {
                elements.dataTableBody.innerHTML = `
                    <tr>
                        <td class="px-6 py-4"><pre class="whitespace-pre-wrap break-words">${escapeHtml(betData.BetContent)}</pre></td>
                        <td class="px-6 py-4 whitespace-nowrap">${formatDateTime(betData.CreateDate)}</td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="px-2 py-1 text-xs font-medium bg-blue-100 text-blue-800 rounded-full">待推送</span>
                        </td>
                    </tr>
                `;
            } else {
                elements.dataTableBody.innerHTML = `
                    <tr><td colspan="3" class="px-6 py-12 text-center text-gray-500"><i class="fa fa-info-circle mr-2"></i>暂无待推送数据</td></tr>
                `;
            }
        }

        // 加载推送日志
        function loadPushLogs() {
            if (!currentAccount) return;

            fetch(`?api=push-logs&accountNo=${currentAccount}`)
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        elements.logTableBody.innerHTML = '';
                        
                        // 重置统计数据
                        pushStats = { total: 0, success: 0, failed: 0 };
                        
                        data.data.forEach(log => {
                            const content = JSON.parse(log.Contents || '{}');
                            
                            // 更新统计数据
                            pushStats.total++;
                            if (content.status === '成功') {
                                pushStats.success++;
                            } else {
                                pushStats.failed++;
                            }

                            const row = document.createElement('tr');
                            row.innerHTML = `
                                <td class="px-6 py-4 whitespace-nowrap">${formatDateTime(log.CreateDate)}</td>
                                <td class="px-6 py-4">${content.title || '无标题'}</td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="px-2 py-1 text-xs font-medium ${content.status === '成功' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'} rounded-full">
                                        ${content.status}
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <button onclick="showLogDetail(${log.Id})" class="text-primary hover:text-primary/80">查看详情</button>
                                </td>
                            `;
                            elements.logTableBody.appendChild(row);
                        });

                        updatePushStats();

                    } else {
                        throw new Error(data.error || '加载日志失败');
                    }
                })
                .catch(err => {
                    addActivity(`加载日志失败: ${err.message}`, 'error');
                    showNotification('错误', `加载日志失败: ${err.message}`, 'danger');
                    console.error('加载日志失败:', err);
                });
        }

        // 更新推送统计数据
        function updatePushStats() {
            elements.totalPushes.textContent = pushStats.total;
            elements.successfulPushes.textContent = pushStats.success;
            elements.failedPushes.textContent = pushStats.failed;
        }

        // 添加活动日志
        function addActivity(message, type = 'info') {
            const timestamp = new Date().toLocaleTimeString();
            activityLog.unshift({ timestamp, message, type });
            
            // 限制日志数量
            if (activityLog.length > 20) {
                activityLog.pop();
            }
            
            // 更新UI
            renderActivityLog();
        }

        // 渲染活动日志
        function renderActivityLog() {
            elements.recentActivities.innerHTML = '';
            
            activityLog.forEach(item => {
                const activityEl = document.createElement('p');
                const typeClass = item.type === 'error' ? 'text-red-500' : 
                                 item.type === 'warning' ? 'text-yellow-500' : 'text-gray-600';
                
                activityEl.className = `${typeClass} flex items-start`;
                activityEl.innerHTML = `
                    <span class="text-gray-400 mr-2">${item.timestamp}</span>
                    <span>${item.message}</span>
                `;
                
                elements.recentActivities.appendChild(activityEl);
            });
        }

        // 测试Telegram连接
        function testTelegramConnection() {
            if (!currentAccount) {
                showNotification('错误', '请先选择账号', 'danger');
                return;
            }

            addActivity('测试Telegram连接中...');
            fetch(`?api=test-connection&accountNo=${currentAccount}`, { method: 'POST' })
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        addActivity('Telegram测试连接成功');
                        showNotification('成功', '测试消息发送成功', 'success');
                    } else {
                        throw new Error(data.error || '测试连接失败');
                    }
                })
                .catch(err => {
                    addActivity(`Telegram测试失败: ${err.message}`, 'error');
                    showNotification('失败', `测试失败: ${err.message}`, 'danger');
                    console.error('测试连接失败:', err);
                });
        }

        // 日志详情
        function showLogDetail(id) {
            fetch(`?api=push-log-detail&id=${id}`)
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        const log = data.data;
                        const content = JSON.parse(log.Contents || '{}');
                        
                        elements.logDetailContent.innerHTML = `
                            <p><strong>标题:</strong> ${content.title || '无标题'}</p>
                            <p><strong>状态:</strong> ${content.status || '未知'}</p>
                            <p><strong>时间:</strong> ${formatDateTime(log.CreateDate)}</p>
                            <p><strong>消息:</strong> ${content.message || '无'}</p>
                            <p><strong>内容:</strong></p>
                            <pre class="bg-gray-50 p-3 rounded">${content.content || '无内容'}</pre>
                        `;
                        elements.logDetailModal.classList.remove('hidden');
                    } else {
                        throw new Error(data.error || '获取日志详情失败');
                    }
                })
                .catch(err => {
                    addActivity(`加载日志详情失败: ${err.message}`, 'error');
                    showNotification('错误', `加载详情失败: ${err.message}`, 'danger');
                    console.error('加载日志详情失败:', err);
                });
        }

        // 通知提示
        function showNotification(title, message, type) {
            const iconClass = {
                success: 'fa-check-circle text-success',
                danger: 'fa-exclamation-circle text-danger',
                info: 'fa-info-circle text-primary',
                warning: 'fa-exclamation-triangle text-yellow-500'
            }[type] || 'fa-info-circle text-primary';

            elements.notificationIcon.className = `fa ${iconClass}`;
            elements.notificationTitle.textContent = title;
            elements.notificationMessage.textContent = message;
            elements.notification.classList.remove('translate-x-full', 'opacity-0');

            setTimeout(hideNotification, 5000);
        }

        function hideNotification() {
            elements.notification.classList.add('translate-x-full', 'opacity-0');
        }

        // 工具函数
        function formatDateTime(dateString) {
            const date = new Date(dateString);
            return date.toLocaleString();
        }

        function escapeHtml(unsafe) {
            if (!unsafe) return '';
            return unsafe
                .replace(/&/g, "&amp;")
                .replace(/</g, "&lt;")
                .replace(/>/g, "&gt;")
                .replace(/"/g, "&quot;")
                .replace(/'/g, "&#039;");
        }

        // 页面关闭前停止定时器
        window.addEventListener('beforeunload', () => {
            stopAllAutoTasks();
        });
    </script>
</body>
</html>
    