<?php
// 数据库配置
require_once 'db.php';

// 初始化数据库连接
try {
    $pdo = new PDO("mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    die("数据库连接失败: " . $e->getMessage());
}

// 处理清空数据请求
if (isset($_POST['clear_all'])) {
    try {
        $pdo->exec("TRUNCATE TABLE TelSms_BetResults");
        $success_msg = "所有数据已成功清空";
    } catch(PDOException $e) {
        $error_msg = "清空数据失败: " . $e->getMessage();
    }
}

// 获取过滤参数
$accountNo = isset($_GET['account_no']) ? $_GET['account_no'] : '';
$username = isset($_GET['username']) ? $_GET['username'] : '';
$gameType = isset($_GET['game_type']) ? $_GET['game_type'] : '';
$hgAdmin = isset($_GET['hg_admin']) ? $_GET['hg_admin'] : '';
$wagerType = isset($_GET['wager_type']) ? $_GET['wager_type'] : '';

// 时间范围过滤参数 - 默认不限制时间范围
$timeRangeType = isset($_GET['time_range_type']) ? $_GET['time_range_type'] : '';
$startDate = isset($_GET['start_date']) ? $_GET['start_date'] : '';
$endDate = isset($_GET['end_date']) ? $_GET['end_date'] : '';

// 根据时间范围类型计算对应的日期范围
list($calculatedStartDate, $calculatedEndDate) = calculateDateRange($timeRangeType);

// 如果是自定义时间范围且提供了日期，则使用提供的日期
if ($timeRangeType == 'custom' && !empty($startDate) && !empty($endDate)) {
    $calculatedStartDate = $startDate;
    $calculatedEndDate = $endDate;
}

// 获取所有账户编号（用于下拉选择）
$accountNumbers = getAccountNumbers($pdo);

// 获取所有用户名（用于下拉选择）
$userNames = getUsernames($pdo);

// 获取所有比赛类型（用于过滤下拉框）
$gameTypes = getGameTypes($pdo);

// 获取所有HgAdmin子账号（用于过滤下拉框）
$hgAdmins = getHgAdmins($pdo);

// 获取所有投注类型（用于过滤下拉框）
$wagerTypes = getWagerTypes($pdo);

// 构建头部统计数据的WHERE子句
$whereClause = buildWhereClause($accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $calculatedStartDate, $calculatedEndDate);

// 总投注数
$totalBetsStmt = $pdo->prepare("SELECT COUNT(Id) as total FROM TelSms_BetResults $whereClause");
bindFilterParams($totalBetsStmt, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $calculatedStartDate, $calculatedEndDate);
$totalBetsStmt->execute();
$totalBets = $totalBetsStmt->fetch(PDO::FETCH_ASSOC)['total'];

// 总投注额
$totalAmountStmt = $pdo->prepare("SELECT SUM(BetAmount) as total FROM TelSms_BetResults $whereClause");
bindFilterParams($totalAmountStmt, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $calculatedStartDate, $calculatedEndDate);
$totalAmountStmt->execute();
$totalAmount = $totalAmountStmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

// 总输赢金额
$totalWinLossStmt = $pdo->prepare("SELECT SUM(WinGold) as total FROM TelSms_BetResults $whereClause");
bindFilterParams($totalWinLossStmt, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $calculatedStartDate, $calculatedEndDate);
$totalWinLossStmt->execute();
$totalWinLoss = $totalWinLossStmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

// 获取指定时间范围内有数据的日期（按日期倒序排列）
$dateRangeData = getDateRangeData($pdo, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $calculatedStartDate, $calculatedEndDate);

// 检查是否有数据
$hasDateRangeData = !empty($dateRangeData);

// 获取用户和账号的输赢汇总数据（用于图表）
$userSummaryData = getUserSummaryData($pdo, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $calculatedStartDate, $calculatedEndDate);

// 获取HgAdmin子账号的输赢汇总数据（用于图表）
$hgAdminSummaryData = getHgAdminSummaryData($pdo, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $calculatedStartDate, $calculatedEndDate);

// 获取WagerType投注类型的汇总数据（用于图表）
$wagerTypeSummaryData = getWagerTypeSummaryData($pdo, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $calculatedStartDate, $calculatedEndDate);

// 获取投注明细（当点击总输赢时）
$detailDate = isset($_GET['detail_date']) ? $_GET['detail_date'] : '';
$betDetails = [];
if (!empty($detailDate)) {
    $betDetails = getBetDetailsByDate($pdo, $detailDate, $accountNo, $username, $gameType, $hgAdmin, $wagerType);
}

// 获取所有投注明细（按时间倒序）
//$allBetDetails = getAllBetDetails($pdo, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $calculatedStartDate, $calculatedEndDate);
$hasAllBetDetails = !empty($allBetDetails);

// 辅助函数：计算日期范围
function calculateDateRange($type) {
    $startDate = '';
    $endDate = '';
    
    if (empty($type)) {
        return [$startDate, $endDate]; // 不限制时间范围
    }
    
    $endDate = date('Y-m-d'); // 默认结束日期为今天
    
    switch ($type) {
        case 'today':
            $startDate = date('Y-m-d');
            break;
        case 'yesterday':
            $startDate = date('Y-m-d', strtotime('-1 day'));
            $endDate = date('Y-m-d', strtotime('-1 day'));
            break;
        case 'thisweek':
            // 计算本周第一天（周一）
            $startDate = date('Y-m-d', strtotime('monday this week'));
            // 如果今天是周日，调整为本周日
            if (date('w') == 0) {
                $startDate = date('Y-m-d', strtotime('-1 week monday this week'));
            }
            break;
        case 'thismonth':
            $startDate = date('Y-m-d', strtotime('first day of this month'));
            break;
        case 'recent30':
            $startDate = date('Y-m-d', strtotime('-29 days'));
            break;
        case 'custom':
            $startDate = '';
            $endDate = '';
            break;
    }
    
    return [$startDate, $endDate];
}

// 辅助函数：构建WHERE子句
function buildWhereClause($accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate) {
    $conditions = [];
    if (!empty($accountNo)) $conditions[] = "AccountNo = :account_no";
    if (!empty($username)) $conditions[] = "Username = :username";
    if (!empty($gameType)) $conditions[] = "GameType = :game_type";
    if (!empty($hgAdmin)) $conditions[] = "HgAdmin = :hg_admin";
    if (!empty($wagerType)) $conditions[] = "WagerType = :wager_type";
    
    // 添加时间范围条件（只有选择了时间范围类型才添加）
    if (!empty($timeRangeType) && ($timeRangeType != 'custom' || (!empty($startDate) && !empty($endDate)))) {
        $conditions[] = "DATE(GameTime) BETWEEN :start_date AND :end_date";
    }
    
    return $conditions ? "WHERE " . implode(" AND ", $conditions) : "";
}

// 辅助函数：绑定过滤参数
function bindFilterParams($stmt, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate) {
    if (!empty($accountNo)) $stmt->bindValue(':account_no', $accountNo);
    if (!empty($username)) $stmt->bindValue(':username', $username);
    if (!empty($gameType)) $stmt->bindValue(':game_type', $gameType);
    if (!empty($hgAdmin)) $stmt->bindValue(':hg_admin', $hgAdmin);
    if (!empty($wagerType)) $stmt->bindValue(':wager_type', $wagerType);
    
    // 绑定时间范围参数（只有选择了时间范围类型才绑定）
    if (!empty($timeRangeType) && ($timeRangeType != 'custom' || (!empty($startDate) && !empty($endDate)))) {
        $stmt->bindValue(':start_date', $startDate);
        $stmt->bindValue(':end_date', $endDate);
    }
}

// 辅助函数：获取指定时间范围内有数据的日期（按日期倒序）
function getDateRangeData($pdo, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate) {
    $data = [];
    
    // 如果是默认情况（未选择时间范围），查询所有有数据的日期
    if (empty($timeRangeType)) {
        // 构建不包含时间范围的WHERE子句
        $whereClause = buildWhereClause($accountNo, $username, $gameType, $hgAdmin, $wagerType, '', '', '');
        
        $stmt = $pdo->prepare("
            SELECT 
                DATE(GameTime) as date,
                COUNT(Id) as bet_count,
                SUM(BetAmount) as total_amount,
                SUM(WinGold) as total_win_loss
            FROM TelSms_BetResults
            $whereClause
            GROUP BY DATE(GameTime)
            HAVING bet_count > 0 OR total_amount > 0 OR total_win_loss != 0
            ORDER BY date DESC
        ");
        bindFilterParams($stmt, $accountNo, $username, $gameType, $hgAdmin, $wagerType, '', '', '');
        $stmt->execute();
        
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($results as $result) {
            $data[] = [
                'date' => $result['date'],
                'day_of_week' => getDayOfWeek(date('w', strtotime($result['date']))),
                'bet_count' => $result['bet_count'],
                'total_amount' => $result['total_amount'],
                'total_win_loss' => $result['total_win_loss']
            ];
        }
        
        return $data;
    }
    
    // 如果没有有效的日期范围，返回空数组
    if (($timeRangeType == 'custom' && (empty($startDate) || empty($endDate))) || 
        ($timeRangeType != 'custom' && (empty($startDate) || empty($endDate)))) {
        return $data;
    }
    
    // 遍历日期范围内的每一天
    $currentDate = $startDate;
    $tempData = [];
    
    while (strtotime($currentDate) <= strtotime($endDate)) {
        $dayOfWeek = getDayOfWeek(date('w', strtotime($currentDate)));
        
        // 构建不包含时间范围的WHERE子句，因为我们要精确查询这一天
        $dayWhereClause = buildWhereClause($accountNo, $username, $gameType, $hgAdmin, $wagerType, '', '', '');
        $additionalWhere = $dayWhereClause ? "AND " . substr($dayWhereClause, 6) : "";
        
        // 统计该日数据
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(Id) as bet_count,
                SUM(BetAmount) as total_amount,
                SUM(WinGold) as total_win_loss
            FROM TelSms_BetResults
            WHERE DATE(GameTime) = :date $additionalWhere
        ");
        $stmt->bindValue(':date', $currentDate);
        bindFilterParams($stmt, $accountNo, $username, $gameType, $hgAdmin, $wagerType, '', '', '');
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // 只添加有数据的日期
        if ($result['bet_count'] > 0 || $result['total_amount'] > 0 || $result['total_win_loss'] != 0) {
            $tempData[] = [
                'date' => $currentDate,
                'day_of_week' => $dayOfWeek,
                'bet_count' => $result['bet_count'] ?? 0,
                'total_amount' => $result['total_amount'] ?? 0,
                'total_win_loss' => $result['total_win_loss'] ?? 0
            ];
        }
        
        $currentDate = date('Y-m-d', strtotime($currentDate . ' +1 day'));
    }
    
    // 按日期倒序排列
    usort($tempData, function($a, $b) {
        return strtotime($b['date']) - strtotime($a['date']);
    });
    
    return $tempData;
}

// 辅助函数：获取所有投注明细（按时间倒序）
function getAllBetDetails($pdo, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate) {
    // 确保即使没有过滤条件也能获取数据
    $whereClause = buildWhereClause($accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate);
    
    // 调试用：输出SQL查询
    // error_log("SQL: SELECT * FROM TelSms_BetResults $whereClause ORDER BY GameTime DESC");
    
    $stmt = $pdo->prepare("
        SELECT 
            Id, WagerId, AccountNo, Username, TeamH, TeamC, GameType,
            HgAdmin, WagerType, BetAmount, Odds, WinGold, ResultWl, GameTime
        FROM TelSms_BetResults
        $whereClause
        ORDER BY GameTime DESC
    ");
    bindFilterParams($stmt, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate);
    $stmt->execute();
    
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// 辅助函数：获取星期几
function getDayOfWeek($num) {
    $days = ['周日', '周一', '周二', '周三', '周四', '周五', '周六'];
    return $days[$num];
}

// 辅助函数：获取用户汇总数据
function getUserSummaryData($pdo, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate) {
    $whereClause = buildWhereClause($accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate);
    
    $stmt = $pdo->prepare("
        SELECT 
            AccountNo,
            Username,
            SUM(WinGold) as total_win_loss
        FROM TelSms_BetResults
        $whereClause
        GROUP BY AccountNo, Username
        HAVING AccountNo IS NOT NULL AND Username IS NOT NULL
        ORDER BY total_win_loss DESC
    ");
    bindFilterParams($stmt, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate);
    $stmt->execute();
    
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// 辅助函数：获取HgAdmin子账号汇总数据
function getHgAdminSummaryData($pdo, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate) {
    $whereClause = buildWhereClause($accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate);
    
    $stmt = $pdo->prepare("
        SELECT 
            HgAdmin,
            SUM(BetAmount) as total_amount,
            SUM(WinGold) as total_win_loss
        FROM TelSms_BetResults
        $whereClause
        GROUP BY HgAdmin
        HAVING HgAdmin IS NOT NULL
        ORDER BY total_amount DESC
    ");
    bindFilterParams($stmt, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate);
    $stmt->execute();
    
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// 辅助函数：获取WagerType投注类型汇总数据
function getWagerTypeSummaryData($pdo, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate) {
    $whereClause = buildWhereClause($accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate);
    
    $stmt = $pdo->prepare("
        SELECT 
            WagerType,
            COUNT(Id) as bet_count,
            SUM(BetAmount) as total_amount,
            SUM(WinGold) as total_win_loss
        FROM TelSms_BetResults
        $whereClause
        GROUP BY WagerType
        HAVING WagerType IS NOT NULL
        ORDER BY bet_count DESC
    ");
    bindFilterParams($stmt, $accountNo, $username, $gameType, $hgAdmin, $wagerType, $timeRangeType, $startDate, $endDate);
    $stmt->execute();
    
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// 辅助函数：获取所有比赛类型
function getGameTypes($pdo) {
    $stmt = $pdo->query("SELECT DISTINCT GameType FROM TelSms_BetResults WHERE GameType IS NOT NULL ORDER BY GameType");
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

// 辅助函数：获取所有HgAdmin子账号
function getHgAdmins($pdo) {
    $stmt = $pdo->query("SELECT DISTINCT HgAdmin FROM TelSms_BetResults WHERE HgAdmin IS NOT NULL ORDER BY HgAdmin");
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

// 辅助函数：获取所有投注类型
function getWagerTypes($pdo) {
    $stmt = $pdo->query("SELECT DISTINCT WagerType FROM TelSms_BetResults WHERE WagerType IS NOT NULL ORDER BY WagerType");
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

// 辅助函数：获取所有账户编号
function getAccountNumbers($pdo) {
    $stmt = $pdo->query("SELECT DISTINCT AccountNo FROM TelSms_BetResults WHERE AccountNo IS NOT NULL ORDER BY AccountNo");
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

// 辅助函数：获取所有用户名
function getUsernames($pdo) {
    $stmt = $pdo->query("SELECT DISTINCT Username FROM TelSms_BetResults WHERE Username IS NOT NULL ORDER BY Username");
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

// 辅助函数：根据日期获取投注明细（按时间倒序）
function getBetDetailsByDate($pdo, $date, $accountNo, $username, $gameType, $hgAdmin, $wagerType) {
    $whereClause = buildWhereClause($accountNo, $username, $gameType, $hgAdmin, $wagerType, '', '', '');
    $additionalWhere = $whereClause ? "AND " . substr($whereClause, 6) : "";
    
    // 确保按时间倒序排列
    $stmt = $pdo->prepare("
        SELECT 
            Id, WagerId, AccountNo, Username, TeamH, TeamC, GameType,
            HgAdmin, WagerType, BetAmount, Odds, WinGold, ResultWl, GameTime
        FROM TelSms_BetResults
        WHERE DATE(GameTime) = :date $additionalWhere
        ORDER BY GameTime DESC
    ");
    $stmt->bindValue(':date', $date);
    bindFilterParams($stmt, $accountNo, $username, $gameType, $hgAdmin, $wagerType, '', '', '');
    $stmt->execute();
    
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>投注统计系统 - Telegram风格</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdn.jsdelivr.net/npm/font-awesome@4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.8/dist/chart.umd.min.js"></script>
    
    <!-- 配置Tailwind自定义颜色（Telegram风格） -->
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        telegram: {
                            bg: '#1E293B',
                            bgLight: '#273449',
                            primary: '#3B82F6',
                            text: '#E2E8F0',
                            textLight: '#94A3B8',
                            success: '#10B981',
                            danger: '#EF4444',
                            neutral: '#64748B'
                        }
                    },
                    fontFamily: {
                        sans: ['Inter', 'system-ui', 'sans-serif'],
                    },
                }
            }
        }
    </script>
    
    <style type="text/tailwindcss">
        @layer utilities {
            .content-auto {
                content-visibility: auto;
            }
            .telegram-shadow {
                box-shadow: 0 1px 3px rgba(0, 0, 0, 0.12), 0 1px 2px rgba(0, 0, 0, 0.24);
            }
            .telegram-hover {
                transition: all 0.2s ease;
            }
            .telegram-hover:hover {
                background-color: rgba(255, 255, 255, 0.05);
            }
        }
    </style>
</head>
<body class="bg-telegram-bg text-telegram-text min-h-screen">
    <!-- 顶部导航 -->
    <header class="bg-telegram-bgLight border-b border-telegram-neutral/30 sticky top-0 z-50">
        <div class="container mx-auto px-4 py-4 flex justify-between items-center">
            <div class="flex items-center space-x-2">
                <i class="fa fa-bar-chart text-telegram-primary text-2xl"></i>
                <h1 class="text-xl font-bold"><a href="index1.php">投注统计系统</a></h1>
            </div>
            <div>
                <button id="themeToggle" class="p-2 rounded-full hover:bg-telegram-neutral/20 transition">
                    <i class="fa fa-moon-o"></i>
                </button>
            </div>
        </div>
    </header>

    <main class="container mx-auto px-4 py-6">
        <!-- 成功/错误消息 -->
        <?php if (isset($success_msg)): ?>
            <div class="bg-telegram-success/20 border border-telegram-success/40 rounded-lg p-4 mb-6 flex items-center">
                <i class="fa fa-check-circle text-telegram-success mr-2"></i>
                <span><?php echo $success_msg; ?></span>
            </div>
        <?php endif; ?>
        
        <?php if (isset($error_msg)): ?>
            <div class="bg-telegram-danger/20 border border-telegram-danger/40 rounded-lg p-4 mb-6 flex items-center">
                <i class="fa fa-exclamation-circle text-telegram-danger mr-2"></i>
                <span><?php echo $error_msg; ?></span>
            </div>
        <?php endif; ?>

        <!-- 头部统计 -->
        <section class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-8">
            <div class="bg-telegram-bgLight rounded-xl p-6 telegram-shadow">
                <div class="flex justify-between items-start mb-4">
                    <div>
                        <p class="text-telegram-textLight text-sm">总投注数</p>
                        <h2 class="text-3xl font-bold mt-1"><?php echo number_format($totalBets); ?></h2>
                    </div>
                    <div class="bg-telegram-primary/20 p-3 rounded-full">
                        <i class="fa fa-ticket text-telegram-primary"></i>
                    </div>
                </div>
                <div class="text-xs text-telegram-textLight">
                    <i class="fa fa-info-circle mr-1"></i> 所选范围内的投注总数
                </div>
            </div>
            
            <div class="bg-telegram-bgLight rounded-xl p-6 telegram-shadow">
                <div class="flex justify-between items-start mb-4">
                    <div>
                        <p class="text-telegram-textLight text-sm">总投注额</p>
                        <h2 class="text-3xl font-bold mt-1"><?php echo number_format($totalAmount, 2); ?></h2>
                    </div>
                    <div class="bg-telegram-primary/20 p-3 rounded-full">
                        <i class="fa fa-money text-telegram-primary"></i>
                    </div>
                </div>
                <div class="text-xs text-telegram-textLight">
                    <i class="fa fa-info-circle mr-1"></i> 所选范围内的投注金额总和
                </div>
            </div>
            
            <div class="bg-telegram-bgLight rounded-xl p-6 telegram-shadow">
                <div class="flex justify-between items-start mb-4">
                    <div>
                        <p class="text-telegram-textLight text-sm">总输赢金额</p>
                        <h2 class="text-3xl font-bold mt-1 <?php echo $totalWinLoss >= 0 ? 'text-telegram-success' : 'text-telegram-danger'; ?>">
                            <?php echo number_format($totalWinLoss, 2); ?>
                        </h2>
                    </div>
                    <div class="bg-telegram-primary/20 p-3 rounded-full">
                        <i class="fa fa-balance-scale text-telegram-primary"></i>
                    </div>
                </div>
                <div class="text-xs text-telegram-textLight">
                    <i class="fa fa-info-circle mr-1"></i> 所选范围内的输赢总和
                </div>
            </div>
        </section>

        <!-- 过滤区域 -->
        <section class="bg-telegram-bgLight rounded-xl p-6 mb-8 telegram-shadow">
            <h3 class="text-lg font-semibold mb-4 flex items-center">
                <i class="fa fa-filter text-telegram-primary mr-2"></i> 数据过滤
            </h3>
            
            <form method="get" class="space-y-4">
                <div class="grid grid-cols-1 md:grid-cols-3 lg:grid-cols-5 gap-4">
                    <div>
                        <label for="account_no" class="block text-sm text-telegram-textLight mb-1">账户编号</label>
                        <select id="account_no" name="account_no" 
                               class="w-full bg-telegram-bg border border-telegram-neutral/30 rounded-lg px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-telegram-primary/50">
                            <option value="">所有账户</option>
                            <?php foreach ($accountNumbers as $accNo): ?>
                                <option value="<?php echo htmlspecialchars($accNo); ?>" 
                                    <?php echo $accountNo == $accNo ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($accNo); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div>
                        <label for="username" class="block text-sm text-telegram-textLight mb-1">用户名</label>
                        <select id="username" name="username" 
                               class="w-full bg-telegram-bg border border-telegram-neutral/30 rounded-lg px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-telegram-primary/50">
                            <option value="">所有用户</option>
                            <?php foreach ($userNames as $user): ?>
                                <option value="<?php echo htmlspecialchars($user); ?>" 
                                    <?php echo $username == $user ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($user); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div>
                        <label for="game_type" class="block text-sm text-telegram-textLight mb-1">比赛类型</label>
                        <select id="game_type" name="game_type" 
                                class="w-full bg-telegram-bg border border-telegram-neutral/30 rounded-lg px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-telegram-primary/50">
                            <option value="">所有类型</option>
                            <?php foreach ($gameTypes as $type): ?>
                                <option value="<?php echo htmlspecialchars($type); ?>" 
                                    <?php echo $gameType == $type ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($type); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div>
                        <label for="hg_admin" class="block text-sm text-telegram-textLight mb-1">子账号</label>
                        <select id="hg_admin" name="hg_admin" 
                                class="w-full bg-telegram-bg border border-telegram-neutral/30 rounded-lg px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-telegram-primary/50">
                            <option value="">所有子账号</option>
                            <?php foreach ($hgAdmins as $admin): ?>
                                <option value="<?php echo htmlspecialchars($admin); ?>" 
                                    <?php echo $hgAdmin == $admin ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($admin); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div>
                        <label for="wager_type" class="block text-sm text-telegram-textLight mb-1">投注类型</label>
                        <select id="wager_type" name="wager_type" 
                                class="w-full bg-telegram-bg border border-telegram-neutral/30 rounded-lg px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-telegram-primary/50">
                            <option value="">所有投注类型</option>
                            <?php foreach ($wagerTypes as $type): ?>
                                <option value="<?php echo htmlspecialchars($type); ?>" 
                                    <?php echo $wagerType == $type ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($type); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                
                <!-- 时间范围过滤 -->
                <div class="mt-6">
                    <label class="block text-sm text-telegram-textLight mb-2">时间范围</label>
                    <div class="flex flex-wrap gap-2 mb-3">
                        <button type="button" class="time-range-btn px-3 py-1 rounded text-sm border 
                            <?php echo $timeRangeType == 'today' ? 'border-telegram-primary bg-telegram-primary/10' : 'border-telegram-neutral/30 hover:border-telegram-primary/50'; ?>"
                            data-type="today">今日</button>
                            
                        <button type="button" class="time-range-btn px-3 py-1 rounded text-sm border 
                            <?php echo $timeRangeType == 'yesterday' ? 'border-telegram-primary bg-telegram-primary/10' : 'border-telegram-neutral/30 hover:border-telegram-primary/50'; ?>"
                            data-type="yesterday">昨日</button>
                            
                        <button type="button" class="time-range-btn px-3 py-1 rounded text-sm border 
                            <?php echo $timeRangeType == 'thisweek' ? 'border-telegram-primary bg-telegram-primary/10' : 'border-telegram-neutral/30 hover:border-telegram-primary/50'; ?>"
                            data-type="thisweek">本周</button>
                            
                        <button type="button" class="time-range-btn px-3 py-1 rounded text-sm border 
                            <?php echo $timeRangeType == 'thismonth' ? 'border-telegram-primary bg-telegram-primary/10' : 'border-telegram-neutral/30 hover:border-telegram-primary/50'; ?>"
                            data-type="thismonth">本月</button>
                            
                        <button type="button" class="time-range-btn px-3 py-1 rounded text-sm border 
                            <?php echo $timeRangeType == 'recent30' ? 'border-telegram-primary bg-telegram-primary/10' : 'border-telegram-neutral/30 hover:border-telegram-primary/50'; ?>"
                            data-type="recent30">最近30天</button>
                            
                        <button type="button" class="time-range-btn px-3 py-1 rounded text-sm border 
                            <?php echo $timeRangeType == 'custom' ? 'border-telegram-primary bg-telegram-primary/10' : 'border-telegram-neutral/30 hover:border-telegram-primary/50'; ?>"
                            data-type="custom">自定义</button>
                            
                        <button type="button" class="time-range-btn px-3 py-1 rounded text-sm border 
                            <?php echo empty($timeRangeType) ? 'border-telegram-primary bg-telegram-primary/10' : 'border-telegram-neutral/30 hover:border-telegram-primary/50'; ?>"
                            data-type="">所有时间</button>
                    </div>
                    
                    <div id="customDateRange" class="grid grid-cols-1 md:grid-cols-2 gap-4 <?php echo $timeRangeType != 'custom' ? 'hidden' : ''; ?>">
                        <div>
                            <label for="start_date" class="block text-sm text-telegram-textLight mb-1">开始日期</label>
                            <input type="date" id="start_date" name="start_date" 
                                   value="<?php echo htmlspecialchars($startDate); ?>"
                                   class="w-full bg-telegram-bg border border-telegram-neutral/30 rounded-lg px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-telegram-primary/50">
                        </div>
                        
                        <div>
                            <label for="end_date" class="block text-sm text-telegram-textLight mb-1">结束日期</label>
                            <input type="date" id="end_date" name="end_date" 
                                   value="<?php echo htmlspecialchars($endDate); ?>"
                                   class="w-full bg-telegram-bg border border-telegram-neutral/30 rounded-lg px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-telegram-primary/50">
                        </div>
                    </div>
                    
                    <input type="hidden" id="time_range_type" name="time_range_type" value="<?php echo htmlspecialchars($timeRangeType); ?>">
                </div>
                
                <div class="flex justify-end space-x-3 pt-4">
                    <button type="submit" class="bg-telegram-primary hover:bg-telegram-primary/90 text-white px-4 py-2 rounded-lg text-sm transition flex items-center">
                        <i class="fa fa-search mr-1"></i> 搜索
                    </button>
                    <a href="?<?php echo http_build_query(array_filter($_GET, function($key) {
                        return !in_array($key, ['account_no', 'username', 'game_type', 'hg_admin', 'wager_type', 'detail_date', 'time_range_type', 'start_date', 'end_date']);
                    }, ARRAY_FILTER_USE_KEY)); ?>" 
                       class="bg-telegram-neutral/30 hover:bg-telegram-neutral/40 text-telegram-text px-4 py-2 rounded-lg text-sm transition flex items-center">
                        <i class="fa fa-refresh mr-1"></i> 重置
                    </a>
                </div>
            </form>
        </section>

        <!-- 日期范围统计报表 - 只有有数据时才显示，且只显示有数据的日期 -->
        <?php if ($hasDateRangeData): ?>
        <section class="bg-telegram-bgLight rounded-xl p-6 mb-8 telegram-shadow">
            <h3 class="text-lg font-semibold mb-4 flex items-center">
                <i class="fa fa-calendar text-telegram-primary mr-2"></i> 
                <?php 
                if (empty($timeRangeType)) {
                    echo '所有时间统计（仅显示有数据的日期）';
                } elseif ($timeRangeType == 'custom' && !empty($startDate) && !empty($endDate)) {
                    echo $startDate . ' 至 ' . $endDate . ' 统计（仅显示有数据的日期）';
                } else {
                    $rangeLabels = [
                        'today' => '今日统计',
                        'yesterday' => '昨日统计',
                        'thisweek' => '本周统计（仅显示有数据的日期）',
                        'thismonth' => '本月统计（仅显示有数据的日期）',
                        'recent30' => '最近30天统计（仅显示有数据的日期）'
                    ];
                    echo $rangeLabels[$timeRangeType] ?? '统计数据（仅显示有数据的日期）';
                }
                ?>
            </h3>
            
            <div class="overflow-x-auto">
                <table class="min-w-full">
                    <thead>
                        <tr class="border-b border-telegram-neutral/30">
                            <th class="py-3 px-4 text-left text-sm font-medium text-telegram-textLight">日期</th>
                            <th class="py-3 px-4 text-left text-sm font-medium text-telegram-textLight">星期</th>
                            <th class="py-3 px-4 text-right text-sm font-medium text-telegram-textLight">投注数</th>
                            <th class="py-3 px-4 text-right text-sm font-medium text-telegram-textLight">投注额</th>
                            <th class="py-3 px-4 text-right text-sm font-medium text-telegram-textLight">总输赢</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($dateRangeData as $day): ?>
                            <tr class="border-b border-telegram-neutral/10 telegram-hover">
                                <td class="py-3 px-4 text-sm"><?php echo $day['date']; ?></td>
                                <td class="py-3 px-4 text-sm"><?php echo $day['day_of_week']; ?></td>
                                <td class="py-3 px-4 text-right text-sm"><?php echo number_format($day['bet_count']); ?></td>
                                <td class="py-3 px-4 text-right text-sm"><?php echo number_format($day['total_amount'], 2); ?></td>
                                <td class="py-3 px-4 text-right text-sm">
                                    <button onclick="showDetails('<?php echo $day['date']; ?>')" 
                                            class="<?php echo $day['total_win_loss'] >= 0 ? 'text-telegram-success' : 'text-telegram-danger'; ?> hover:underline flex items-center justify-end">
                                        <?php echo number_format($day['total_win_loss'], 2); ?>
                                        <i class="fa fa-chevron-down ml-1 text-xs"></i>
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </section>
        <?php endif; ?>

       

        <!-- 投注明细弹窗 -->
        <?php if (!empty($betDetails)): ?>
            <div id="detailsModal" class="fixed inset-0 bg-black/50 z-50 flex items-center justify-center p-4">
                <div class="bg-telegram-bgLight rounded-xl w-full max-w-3xl max-h-[80vh] overflow-hidden flex flex-col">
                    <div class="p-4 border-b border-telegram-neutral/30 flex justify-between items-center">
                        <h3 class="font-semibold">
                            <?php echo $detailDate; ?> 投注明细（按时间倒序）
                        </h3>
                        <button onclick="closeDetails()" class="p-2 hover:bg-telegram-neutral/20 rounded-full">
                            <i class="fa fa-times"></i>
                        </button>
                    </div>
                    
                    <div class="overflow-y-auto flex-grow p-4">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <?php foreach ($betDetails as $detail): ?>
                                <div class="bg-telegram-bg rounded-lg p-3 border border-telegram-neutral/20">
                                    <div class="flex justify-between items-start mb-2">
                                        <div>
                                            <p class="text-sm font-medium"><?php echo htmlspecialchars($detail['TeamH']); ?> vs <?php echo htmlspecialchars($detail['TeamC']); ?></p>
                                            <p class="text-xs text-telegram-textLight mt-1">
                                                <?php echo date('H:i', strtotime($detail['GameTime'])); ?>
                                            </p>
                                        </div>
                                        <span class="text-xs px-2 py-1 rounded-full 
                                            <?php echo $detail['ResultWl'] == '赢' ? 'bg-telegram-success/20 text-telegram-success' : 
                                                     ($detail['ResultWl'] == '输' ? 'bg-telegram-danger/20 text-telegram-danger' : 
                                                     'bg-telegram-neutral/20 text-telegram-textLight'); ?>">
                                            <?php echo htmlspecialchars($detail['ResultWl']); ?>
                                        </span>
                                    </div>
                                    
                                    <div class="grid grid-cols-2 gap-2 text-sm mt-3">
                                        <div>
                                            <p class="text-telegram-textLight text-xs">账户</p>
                                            <p><?php echo htmlspecialchars($detail['AccountNo'] ?? 'N/A'); ?></p>
                                        </div>
                                        <div>
                                            <p class="text-telegram-textLight text-xs">用户</p>
                                            <p><?php echo htmlspecialchars($detail['Username'] ?? 'N/A'); ?></p>
                                        </div>
                                        <div>
                                            <p class="text-telegram-textLight text-xs">子账号</p>
                                            <p><?php echo htmlspecialchars($detail['HgAdmin'] ?? 'N/A'); ?></p>
                                        </div>
                                        <div>
                                            <p class="text-telegram-textLight text-xs">投注类型</p>
                                            <p><?php echo htmlspecialchars($detail['WagerType'] ?? 'N/A'); ?></p>
                                        </div>
                                        <div>
                                            <p class="text-telegram-textLight text-xs">投注额</p>
                                            <p><?php echo number_format($detail['BetAmount'], 2); ?></p>
                                        </div>
                                        <div>
                                            <p class="text-telegram-textLight text-xs">赔率</p>
                                            <p><?php echo number_format($detail['Odds'], 2); ?></p>
                                        </div>
                                        <div class="col-span-2 mt-2">
                                            <p class="text-telegram-textLight text-xs">输赢</p>
                                            <p class="<?php echo $detail['WinGold'] >= 0 ? 'text-telegram-success' : 'text-telegram-danger'; ?> font-medium">
                                                <?php echo number_format($detail['WinGold'], 2); ?>
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        
                        <?php if (empty($betDetails)): ?>
                            <div class="text-center py-8 text-telegram-textLight">
                                <i class="fa fa-info-circle text-2xl mb-2"></i>
                                <p>该日期没有投注记录</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- 图表区域 - 用户输赢统计 -->
        <section class="bg-telegram-bgLight rounded-xl p-6 mb-8 telegram-shadow">
            <h3 class="text-lg font-semibold mb-4 flex items-center">
                <i class="fa fa-pie-chart text-telegram-primary mr-2"></i> 用户输赢统计
            </h3>
            
            <div class="h-80">
                <?php if (!empty($userSummaryData)): ?>
                    <canvas id="userSummaryChart"></canvas>
                <?php else: ?>
                    <div class="flex items-center justify-center h-full text-telegram-textLight">
                        <div class="text-center">
                            <i class="fa fa-bar-chart text-4xl mb-3"></i>
                            <p>没有可用的用户统计数据</p>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </section>

        <!-- 图表区域 - HgAdmin子账号统计 -->
        <section class="bg-telegram-bgLight rounded-xl p-6 mb-8 telegram-shadow">
            <h3 class="text-lg font-semibold mb-4 flex items-center">
                <i class="fa fa-users text-telegram-primary mr-2"></i> 子账号统计
            </h3>
            
            <div class="h-80">
                <?php if (!empty($hgAdminSummaryData)): ?>
                    <canvas id="hgAdminSummaryChart"></canvas>
                <?php else: ?>
                    <div class="flex items-center justify-center h-full text-telegram-textLight">
                        <div class="text-center">
                            <i class="fa fa-bar-chart text-4xl mb-3"></i>
                            <p>没有可用的子账号统计数据</p>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </section>

        <!-- 图表区域 - WagerType投注类型统计 -->
        <section class="bg-telegram-bgLight rounded-xl p-6 mb-8 telegram-shadow">
            <h3 class="text-lg font-semibold mb-4 flex items-center">
                <i class="fa fa-list-alt text-telegram-primary mr-2"></i> 投注类型统计
            </h3>
            
            <div class="h-80">
                <?php if (!empty($wagerTypeSummaryData)): ?>
                    <canvas id="wagerTypeSummaryChart"></canvas>
                <?php else: ?>
                    <div class="flex items-center justify-center h-full text-telegram-textLight">
                        <div class="text-center">
                            <i class="fa fa-bar-chart text-4xl mb-3"></i>
                            <p>没有可用的投注类型统计数据</p>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </section>

        <!-- 清空数据功能 -->
        <section class="bg-telegram-bgLight rounded-xl p-6 mb-8 telegram-shadow">
            <h3 class="text-lg font-semibold mb-4 flex items-center">
                <i class="fa fa-trash text-telegram-danger mr-2"></i> 数据管理
            </h3>
            
            <div class="bg-telegram-danger/10 border border-telegram-danger/30 rounded-lg p-4">
                <p class="text-sm mb-4">
                    <i class="fa fa-exclamation-triangle text-telegram-danger mr-2"></i>
                    警告：此操作将删除所有投注记录数据，且无法恢复。请谨慎操作。
                </p>
                
                <form method="post" onsubmit="return confirm('确定要清空所有数据吗？此操作不可恢复！');">
                    <button type="submit" name="clear_all" 
                            class="bg-telegram-danger hover:bg-telegram-danger/90 text-white px-4 py-2 rounded-lg text-sm transition flex items-center">
                        <i class="fa fa-trash mr-1"></i> 一键清空所有数据
                    </button>
                </form>
            </div>
        </section>
    </main>

    <footer class="bg-telegram-bgLight border-t border-telegram-neutral/30 py-4">
        <div class="container mx-auto px-4 text-center text-telegram-textLight text-sm">
            <p>投注统计系统 &copy; <?php echo date('Y'); ?></p>
        </div>
    </footer>

    <script>
        // 显示明细
        function showDetails(date) {
            // 构建当前URL参数
            const params = new URLSearchParams(window.location.search);
            params.set('detail_date', date);
            window.location.search = params.toString();
        }
        
        // 关闭明细
        function closeDetails() {
            const params = new URLSearchParams(window.location.search);
            params.delete('detail_date');
            window.location.search = params.toString();
        }
        
        // 时间范围选择交互
        document.querySelectorAll('.time-range-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                const type = this.getAttribute('data-type');
                document.getElementById('time_range_type').value = type;
                
                // 显示或隐藏自定义日期范围
                const customDateRange = document.getElementById('customDateRange');
                if (type === 'custom') {
                    customDateRange.classList.remove('hidden');
                } else {
                    customDateRange.classList.add('hidden');
                    // 如果不是自定义范围，自动提交表单
                    document.querySelector('form').submit();
                }
                
                // 更新按钮样式
                document.querySelectorAll('.time-range-btn').forEach(b => {
                    if (b.getAttribute('data-type') === type) {
                        b.classList.remove('border-telegram-neutral/30', 'hover:border-telegram-primary/50');
                        b.classList.add('border-telegram-primary', 'bg-telegram-primary/10');
                    } else {
                        b.classList.remove('border-telegram-primary', 'bg-telegram-primary/10');
                        b.classList.add('border-telegram-neutral/30', 'hover:border-telegram-primary/50');
                    }
                });
            });
        });
        
        // 初始化所有图表
        document.addEventListener('DOMContentLoaded', function() {
            // 1. 用户输赢统计图表
            initUserSummaryChart();
            
            // 2. HgAdmin子账号统计图表
            initHgAdminSummaryChart();
            
            // 3. WagerType投注类型统计图表
            initWagerTypeSummaryChart();
        });
        
        // 初始化用户输赢统计图表
        function initUserSummaryChart() {
            const ctx = document.getElementById('userSummaryChart');
            if (!ctx) return;
            
            // 准备图表数据
            const labels = <?php echo json_encode(array_map(function($item) {
                return ($item['AccountNo'] ?? 'N/A') . ' - ' . ($item['Username'] ?? 'N/A');
            }, $userSummaryData)); ?>;
            
            const data = <?php echo json_encode(array_column($userSummaryData, 'total_win_loss')); ?>;
            
            if (labels.length === 0 || data.length === 0) return;
            
            // 设置颜色 - 正数为绿色，负数为红色
            const backgroundColors = data.map(value => 
                value >= 0 ? 'rgba(16, 185, 129, 0.7)' : 'rgba(239, 68, 68, 0.7)'
            );
            
            const borderColors = data.map(value => 
                value >= 0 ? 'rgba(16, 185, 129, 1)' : 'rgba(239, 68, 68, 1)'
            );
            
            // 创建图表
            new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: labels,
                    datasets: [{
                        label: '输赢金额',
                        data: data,
                        backgroundColor: backgroundColors,
                        borderColor: borderColors,
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                color: 'rgba(100, 116, 139, 0.1)'
                            },
                            ticks: {
                                color: '#94A3B8'
                            }
                        },
                        x: {
                            grid: {
                                display: false
                            },
                            ticks: {
                                color: '#94A3B8'
                            }
                        }
                    },
                    plugins: {
                        legend: {
                            display: false
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return `金额: ${context.raw.toFixed(2)}`;
                                }
                            }
                        }
                    }
                }
            });
        }
        
        // 初始化HgAdmin子账号统计图表
        function initHgAdminSummaryChart() {
            const ctx = document.getElementById('hgAdminSummaryChart');
            if (!ctx) return;
            
            // 准备图表数据
            const labels = <?php echo json_encode(array_column($hgAdminSummaryData, 'HgAdmin')); ?>;
            const totalAmountData = <?php echo json_encode(array_column($hgAdminSummaryData, 'total_amount')); ?>;
            const totalWinLossData = <?php echo json_encode(array_column($hgAdminSummaryData, 'total_win_loss')); ?>;
            
            if (labels.length === 0 || totalAmountData.length === 0) return;
            
            // 创建图表
            new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: labels,
                    datasets: [{
                        label: '总投注额',
                        data: totalAmountData,
                        backgroundColor: 'rgba(59, 130, 246, 0.7)',
                        borderColor: 'rgba(59, 130, 246, 1)',
                        borderWidth: 1,
                        order: 2
                    }, {
                        label: '总输赢',
                        data: totalWinLossData,
                        backgroundColor: totalWinLossData.map(value => 
                            value >= 0 ? 'rgba(16, 185, 129, 0.7)' : 'rgba(239, 68, 68, 0.7)'
                        ),
                        borderColor: totalWinLossData.map(value => 
                            value >= 0 ? 'rgba(16, 185, 129, 1)' : 'rgba(239, 68, 68, 1)'
                        ),
                        borderWidth: 1,
                        order: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                color: 'rgba(100, 116, 139, 0.1)'
                            },
                            ticks: {
                                color: '#94A3B8'
                            }
                        },
                        x: {
                            grid: {
                                display: false
                            },
                            ticks: {
                                color: '#94A3B8'
                            }
                        }
                    },
                    plugins: {
                        legend: {
                            position: 'top',
                            labels: {
                                color: '#94A3B8'
                            }
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return `${context.dataset.label}: ${context.raw.toFixed(2)}`;
                                }
                            }
                        }
                    }
                }
            });
        }
        
        // 初始化WagerType投注类型统计图表
        function initWagerTypeSummaryChart() {
            const ctx = document.getElementById('wagerTypeSummaryChart');
            if (!ctx) return;
            
            // 准备图表数据
            const labels = <?php echo json_encode(array_column($wagerTypeSummaryData, 'WagerType')); ?>;
            const betCountData = <?php echo json_encode(array_column($wagerTypeSummaryData, 'bet_count')); ?>;
            const totalAmountData = <?php echo json_encode(array_column($wagerTypeSummaryData, 'total_amount')); ?>;
            
            if (labels.length === 0 || betCountData.length === 0) return;
            
            // 创建图表
            new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: labels,
                    datasets: [{
                        label: '投注数',
                        data: betCountData,
                        backgroundColor: 'rgba(148, 163, 184, 0.7)',
                        borderColor: 'rgba(148, 163, 184, 1)',
                        borderWidth: 1,
                        order: 2,
                        yAxisID: 'y'
                    }, {
                        label: '总投注额',
                        data: totalAmountData,
                        backgroundColor: 'rgba(16, 185, 129, 0.7)',
                        borderColor: 'rgba(16, 185, 129, 1)',
                        borderWidth: 1,
                        order: 1,
                        yAxisID: 'y1'
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        y: {
                            type: 'linear',
                            display: true,
                            position: 'left',
                            beginAtZero: true,
                            grid: {
                                color: 'rgba(100, 116, 139, 0.1)'
                            },
                            ticks: {
                                color: '#94A3B8'
                            }
                        },
                        y1: {
                            type: 'linear',
                            display: true,
                            position: 'right',
                            beginAtZero: true,
                            grid: {
                                drawOnChartArea: false
                            },
                            ticks: {
                                color: '#94A3B8'
                            }
                        },
                        x: {
                            grid: {
                                display: false
                            },
                            ticks: {
                                color: '#94A3B8'
                            }
                        }
                    },
                    plugins: {
                        legend: {
                            position: 'top',
                            labels: {
                                color: '#94A3B8'
                            }
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return `${context.dataset.label}: ${context.raw.toFixed(2)}`;
                                }
                            }
                        }
                    }
                }
            });
        }
        
        // 主题切换功能
        document.getElementById('themeToggle').addEventListener('click', function() {
            const icon = this.querySelector('i');
            if (icon.classList.contains('fa-moon-o')) {
                icon.classList.remove('fa-moon-o');
                icon.classList.add('fa-sun-o');
                alert('亮色模式功能未实现');
            } else {
                icon.classList.remove('fa-sun-o');
                icon.classList.add('fa-moon-o');
                alert('已切换到深色模式');
            }
        });
    </script>
</body>
</html>
