<?php
// 放在页面最顶部，确保在任何输出之前执行
require_once 'session_check.php';
require_once 'db.php';
 try {
        $pdo = new PDO("mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    } catch (PDOException $e) {
        die(json_encode(['error' => '数据库连接失败: ' . $e->getMessage()]));
    }

?>


<?php
// 处理明细数据请求
if (isset($_GET['action']) && $_GET['action'] === 'get_details') {
    // 数据库连接
   

    // 获取参数
    $accountNo = isset($_GET['AccountNo']) ? trim($_GET['AccountNo']) : '';
    $date = isset($_GET['date']) ? $_GET['date'] : '';
    $usernameFilter = isset($_GET['username']) ? $_GET['username'] : '';
    $gameTypeFilter = isset($_GET['gameType']) ? $_GET['gameType'] : '';
    $dateFilter = isset($_GET['dateFilter']) ? $_GET['dateFilter'] : 'all';
    $startDate = isset($_GET['startDate']) ? $_GET['startDate'] : '';
    $endDate = isset($_GET['endDate']) ? $_GET['endDate'] : '';

    if (empty($accountNo)) {
         $loginUrl = 'login.php';
    header('Location: ' . $loginUrl);
    exit;
    }

    // 构建查询条件
    $conditions = ["AccountNo = :accountNo"];
    $params = [':accountNo' => $accountNo];

    // 日期条件
    if ($date !== 'all') {
        $conditions[] = "DATE(GameTime) = :date";
        $params[':date'] = $date;
    } elseif ($dateFilter === 'today') {
        $conditions[] = "DATE(GameTime) = CURDATE()";
    } elseif ($dateFilter === 'yesterday') {
        $conditions[] = "DATE(GameTime) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)";
    } elseif ($dateFilter === 'this_week') {
        $conditions[] = "YEARWEEK(GameTime, 1) = YEARWEEK(CURDATE(), 1)";
    } elseif ($dateFilter === 'this_month') {
        $conditions[] = "MONTH(GameTime) = MONTH(CURDATE()) AND YEAR(GameTime) = YEAR(CURDATE())";
    } elseif ($dateFilter === 'custom' && !empty($startDate) && !empty($endDate)) {
        $conditions[] = "GameTime BETWEEN :startDate AND :endDate";
        $params[':startDate'] = $startDate . ' 00:00:00';
        $params[':endDate'] = $endDate . ' 23:59:59';
    }

    // 其他筛选条件
    if (!empty($usernameFilter)) {
        $conditions[] = "Username = :username";
        $params[':username'] = $usernameFilter;
    }

    if (!empty($gameTypeFilter)) {
        $conditions[] = "GameType = :gameType";
        $params[':gameType'] = $gameTypeFilter;
    }

    // 查询明细数据，按投注时间(DateTime)降序排序
    try {
        $sql = "SELECT * FROM TelSms_BetResults WHERE " . implode(" AND ", $conditions) . " ORDER BY DateTime DESC";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $details = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (empty($details)) {
            echo '<div class="text-center py-10 text-neutral-lighter">
                    <i class="fa fa-inbox text-4xl mb-3"></i>
                    <p>没有找到对应的投注明细</p>
                  </div>';
            exit;
        }

        // 输出明细HTML（简化后的布局）
        echo '<div class="grid grid-cols-1 gap-4">';
        foreach ($details as $detail) {
            $winGold = floatval($detail['WinGold']);
            $betAmount = floatval($detail['BetAmount']);
            
            // 状态颜色定义
            $winColor = $winGold >= 0 ? 'text-success' : 'text-secondary';
            switch ($detail['ResultStatus']) {
                case 1: 
                    $resultBadge = '<span class="bg-success/10 text-success px-2 py-1 rounded-full text-xs">赢</span>';
                    break;
                case 2: 
                    $resultBadge = '<span class="bg-secondary/10 text-secondary px-2 py-1 rounded-full text-xs">输</span>';
                    break;
                case 3: 
                    $resultBadge = '<span class="bg-primary/10 text-primary px-2 py-1 rounded-full text-xs">和</span>';
                    break;
                default: 
                    $resultBadge = '<span class="bg-neutral-lightest text-neutral-lighter px-2 py-1 rounded-full text-xs">未知</span>';
            }
            
            // 格式化时间
            $betTime = $detail['ResultTime'];
            
            // 投注卡片
            echo '<div class="bg-white rounded-xl shadow-md overflow-hidden transition-all duration-300 hover:shadow-lg border border-gray-50">';
            // 1. 投注编号-投注时间 + 状态
            echo '<div class="bg-gray-50 px-4 py-3 border-b border-gray-100 flex flex-wrap justify-between items-center gap-2">
                    <div class="text-sm text-neutral">
                        <span class="font-medium">' .  $betTime . '</span>
                    </div>
                    <div class="flex items-center gap-3">
                        <span class="text-xs text-neutral-lighter">' . htmlspecialchars($detail['WagerId']). '</span>
                        ' . $resultBadge . '
                    </div>
                  </div>';
            
            // 卡片主体
            echo '<div class="p-4">';
            // 2. 投注内容（完整）
            echo '<div class="mb-4 p-3 bg-neutral-lightest/50 rounded-lg">
                    <div class="text-xs text-neutral-lighter mb-1">投注内容</div>
                    <div class="text-xs text-neutral whitespace-pre-line break-all">' . htmlspecialchars($detail['BetContent'] ?? '无投注内容') . '</div>
                  </div>';
            
            // 3. 投注金额 和 4. 输赢结果
            echo '<div class="grid grid-cols-2 gap-4">
                    <div class="p-3 bg-primary/5 rounded-lg">
                        <div class="text-xs text-neutral-lighter mb-1">投注金额</div>
                        <div class="text-2xs font-bold text-neutral">¥' . number_format($betAmount, 2) . '</div>
                    </div>
                    <div class="p-3 bg-primary/5 rounded-lg">
                        <div class="text-xs text-neutral-lighter mb-1">输赢结果</div>
                        <div class="text-xs font-bold ' . $winColor . '">'.$detail['ResultWl']. number_format($winGold, 2) . '</div>
                    </div>
                  </div>';
            
            echo '</div>'; // 主体结束
            echo '</div>'; // 卡片结束
        }
        echo '</div>'; // 网格容器结束
    } catch (PDOException $e) {
        echo '<div class="bg-red-50 border-l-4 border-red-400 p-4 text-center">获取明细失败: ' . htmlspecialchars($e->getMessage()) . '</div>';
    }
    exit;
}

// 数据库连接函数 - 提取为公共函数，避免重复代码
function getDbConnection() {
    
   require_once 'db.php';
    try {
        
       $pdo = new PDO("mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
       $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
       
        return $pdo;
    } catch (PDOException $e) {
        die("数据库连接失败: " . $e->getMessage());
    }
}

// 主页面逻辑


// 获取筛选参数 - 强制验证AccountNo
$accountNo = isset($_GET['AccountNo']) ? trim($_GET['AccountNo']) : '';
if (empty($accountNo)) {
     $loginUrl = 'login.php';
    header('Location: ' . $loginUrl);
    exit;
}

// 初始化筛选参数
$dateFilter = isset($_GET['dateFilter']) ? $_GET['dateFilter'] : 'all';
$startDate = isset($_GET['startDate']) ? $_GET['startDate'] : '';
$endDate = isset($_GET['endDate']) ? $_GET['endDate'] : '';
$usernameFilter = isset($_GET['username']) ? $_GET['username'] : '';
$gameTypeFilter = isset($_GET['gameType']) ? $_GET['gameType'] : '';
$pageSize = isset($_GET['pageSize']) ? max(1, intval($_GET['pageSize'])) : 30;
$currentPage = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$refreshInterval = isset($_GET['refreshInterval']) ? max(2, intval($_GET['refreshInterval'])) : 30;

// 初始化错误信息数组
$errors = [];

// 验证日期
if ($dateFilter === 'custom') {
    if (empty($startDate) || empty($endDate)) {
        $errors[] = '自定义日期范围时，开始日期和结束日期不能为空';
        $dateFilter = 'all';
    } elseif ($startDate > $endDate) {
        $errors[] = '开始日期不能大于结束日期';
        $dateFilter = 'all';
    } elseif (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $startDate) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $endDate)) {
        $errors[] = '日期格式不正确，请使用YYYY-MM-DD格式';
        $dateFilter = 'all';
    }
}

// 获取所有可用的用户名和比赛类型用于下拉选项
$usernames = [];
$gameTypes = [];

try {
    // 获取用户名列表
    $userStmt = $pdo->prepare("SELECT DISTINCT Username FROM TelSms_BetResults WHERE AccountNo = :accountNo AND Username IS NOT NULL AND Username != '' ORDER BY Username");
    $userStmt->bindValue(':accountNo', $accountNo, PDO::PARAM_STR);
    $userStmt->execute();
    $usernames = $userStmt->fetchAll(PDO::FETCH_COLUMN, 0);
    
    // 获取比赛类型列表
    $gameTypeStmt = $pdo->prepare("SELECT DISTINCT GameType FROM TelSms_BetResults WHERE AccountNo = :accountNo AND GameType IS NOT NULL AND GameType != '' ORDER BY GameType");
    $gameTypeStmt->bindValue(':accountNo', $accountNo, PDO::PARAM_STR);
    $gameTypeStmt->execute();
    $gameTypes = $gameTypeStmt->fetchAll(PDO::FETCH_COLUMN, 0);
} catch (PDOException $e) {
    $errors[] = '获取筛选选项失败: ' . $e->getMessage();
}

// 构建日期筛选条件
$dateCondition = '';
$dateParams = [];

try {
    switch ($dateFilter) {
        case 'today':
            $dateCondition = "AND DATE(GameTime) = CURDATE()";
            break;
        case 'yesterday':
            $dateCondition = "AND DATE(GameTime) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)";
            break;
        case 'this_week':
            $dateCondition = "AND YEARWEEK(GameTime, 1) = YEARWEEK(CURDATE(), 1)";
            break;
        case 'this_month':
            $dateCondition = "AND MONTH(GameTime) = MONTH(CURDATE()) AND YEAR(GameTime) = YEAR(CURDATE())";
            break;
        case 'custom':
            if (!empty($startDate) && !empty($endDate)) {
                $dateCondition = "AND GameTime BETWEEN :startDate AND :endDate";
                $dateParams[':startDate'] = $startDate . ' 00:00:00';
                $dateParams[':endDate'] = $endDate . ' 23:59:59';
            }
            break;
    }
} catch (Exception $e) {
    $errors[] = '日期筛选条件构建失败: ' . $e->getMessage();
    $dateCondition = '';
    $dateParams = [];
}

// 构建其他筛选条件
$otherConditions = [];
$otherParams = [];

// 强制加入AccountNo条件，作为基础查询条件
$otherConditions[] = "AccountNo = :accountNo";
$otherParams[':accountNo'] = $accountNo;

try {
    if (!empty($usernameFilter)) {
        $otherConditions[] = "Username = :username";
        $otherParams[':username'] = $usernameFilter;
    }

    if (!empty($gameTypeFilter)) {
        $otherConditions[] = "GameType = :gameType";
        $otherParams[':gameType'] = $gameTypeFilter;
    }
} catch (Exception $e) {
    $errors[] = '筛选条件构建失败: ' . $e->getMessage();
    $otherConditions = ["AccountNo = :accountNo"]; // 确保AccountNo条件始终存在
    $otherParams[':accountNo'] = $accountNo;
}

$otherCondition = '';
if (!empty($otherConditions)) {
    $otherCondition = "AND " . implode(" AND ", $otherConditions);
}

// 总体统计数据
$totalStats = [
    'total_bets' => 0,
    'total_amount' => 0,
    'total_win_gold' => 0
];

// 每日统计数据
$dailyStats = [];

// 用户名输赢统计（用于图表）
$userWinStats = [];

try {
    // 获取总体统计
    $totalStmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_bets,
            SUM(BetAmount) as total_amount,
            SUM(WinGold) as total_win_gold
        FROM TelSms_BetResults 
        WHERE 1=1 $dateCondition $otherCondition
    ");
    $totalStmt->execute(array_merge($dateParams, $otherParams));
    $totalStats = $totalStmt->fetch(PDO::FETCH_ASSOC);
    
    // 获取每日统计（按GameTime日期分组）
    // 优化：添加DISTINCT确保日期唯一，并使用明确的分组字段
    $dailyStmt = $pdo->prepare("
        SELECT 
            DATE(GameTime) as game_date,
            DAYNAME(GameTime) as weekday,
            COUNT(DISTINCT WagerId) as bet_count,  -- 使用DISTINCT避免重复计算
            SUM(BetAmount) as total_amount,
            SUM(WinGold) as total_win_gold
        FROM TelSms_BetResults 
        WHERE 1=1 $dateCondition $otherCondition
        GROUP BY game_date, weekday  -- 明确按两个字段分组
        ORDER BY game_date DESC
    ");
    $dailyStmt->execute(array_merge($dateParams, $otherParams));
    $dailyStats = $dailyStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // 去重处理：添加PHP层面的去重逻辑作为双重保障
    $uniqueDailyStats = [];
    foreach ($dailyStats as $day) {
        // 使用game_date作为唯一键，确保不会有重复日期
        if (!isset($uniqueDailyStats[$day['game_date']])) {
            $uniqueDailyStats[$day['game_date']] = $day;
        }
    }
    // 重新索引数组并保持原有排序
    $dailyStats = array_values($uniqueDailyStats);
    
    // 转换星期为中文
    $weekdayMap = [
        'Sunday' => '周日',
        'Monday' => '周一',
        'Tuesday' => '周二',
        'Wednesday' => '周三',
        'Thursday' => '周四',
        'Friday' => '周五',
        'Saturday' => '周六'
    ];
    
    foreach ($dailyStats as &$day) {
        $day['weekday'] = $weekdayMap[$day['weekday']] ?? $day['weekday'];
    }
    unset($day); // 解除引用
    
    // 获取用户名输赢统计（用于图表）
    $userStmt = $pdo->prepare("
        SELECT 
            Username,
            SUM(WinGold) as total_win_gold
        FROM TelSms_BetResults 
        WHERE 1=1 $dateCondition $otherCondition AND Username IS NOT NULL AND Username != ''
        GROUP BY Username
        ORDER BY total_win_gold DESC
    ");
    $userStmt->execute(array_merge($dateParams, $otherParams));
    $userWinStats = $userStmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    $errors[] = '统计数据查询失败: ' . $e->getMessage();
}

// 准备用于图表的数据
$chartLabels = [];
$chartData = [];
foreach ($userWinStats as $user) {
    $chartLabels[] = $user['Username'];
    $chartData[] = floatval($user['total_win_gold']);
}

// 准备图表数据集
$chartDataset = [
    'labels' => $chartLabels,
    'datasets' => [
        [
            'label' => '用户输赢金额',
            'data' => $chartData,
            'backgroundColor' => array_map(function($value) {
                return $value >= 0 ? 'rgba(82, 196, 26, 0.7)' : 'rgba(255, 77, 79, 0.7)';
            }, $chartData),
            'borderColor' => array_map(function($value) {
                return $value >= 0 ? 'rgba(82, 196, 26, 1)' : 'rgba(255, 77, 79, 1)';
            }, $chartData),
            'borderWidth' => 1
        ]
    ]
];
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>投注结果统计</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdn.jsdelivr.net/npm/font-awesome@4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.8/dist/chart.umd.min.js"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#165DFF',
                        secondary: '#FF4D4F',
                        success: '#52C41A',
                        neutral: '#1D2129',
                        'neutral-light': '#4E5969',
                        'neutral-lighter': '#86909C',
                        'neutral-lightest': '#F2F3F5'
                    },
                    fontFamily: {
                        sans: ['Inter', 'system-ui', 'sans-serif'],
                    },
                }
            }
        }
    </script>
    <style type="text/tailwindcss">
        @layer utilities {
            .content-auto {
                content-visibility: auto;
            }
            .scrollbar-hide {
                -ms-overflow-style: none;
                scrollbar-width: none;
            }
            .scrollbar-hide::-webkit-scrollbar {
                display: none;
            }
            .card-shadow {
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            }
            .bg-gradient-blue {
                background: linear-gradient(135deg, #165DFF 0%, #4080FF 100%);
            }
            .bg-gradient-red {
                background: linear-gradient(135deg, #FF4D4F 0%, #FF7A45 100%);
            }
            .bg-gradient-green {
                background: linear-gradient(135deg, #52C41A 0%, #73D13D 100%);
            }
        }
    </style>
</head>
<body class="bg-neutral-lightest min-h-screen font-sans">
    <!-- 顶部导航栏 -->
    <header class="bg-white shadow-md fixed top-0 left-0 right-0 z-10 transition-all duration-300">
        <div class="container mx-auto px-4 py-3 flex items-center justify-between">
            <div class="flex items-center space-x-2">
                <i class="fa fa-bar-chart text-primary text-xl"></i>
                <h1 class="text-lg font-bold text-neutral">结果统计</h1>
                <h1 class="text-lg font-bold text-neutral" style="display: none;">账号：<?php echo htmlspecialchars($accountNo); ?></h1>
            </div>
            <div class="flex items-center space-x-4">
                <button id="refreshBtn" class="text-primary hover:text-primary/80 transition-colors">
                    <i class="fa fa-refresh"></i>
                </button>
                <button id="settingsBtn" class="text-neutral-light hover:text-neutral transition-colors">
                    <i class="fa fa-cog"></i>
                </button>
            </div>
        </div>
    </header>

    <!-- 错误提示 -->
    <?php if (!empty($errors)): ?>
        <div class="fixed top-16 left-0 right-0 z-50 bg-secondary text-white py-2 px-4">
            <div class="container mx-auto">
                <div class="flex items-center">
                    <i class="fa fa-exclamation-circle mr-2"></i>
                    <div>
                        <?php foreach ($errors as $error): ?>
                            <div><?php echo htmlspecialchars($error); ?></div>
                        <?php endforeach; ?>
                    </div>
                    <button id="closeError" class="ml-auto text-white hover:text-white/80">
                        <i class="fa fa-times"></i>
                    </button>
                </div>
            </div>
        </div>
        <script>
            document.getElementById('closeError').addEventListener('click', function() {
                this.closest('div').style.display = 'none';
            });
        </script>
    <?php endif; ?>

    <!-- 设置面板 -->
    <div id="settingsPanel" class="fixed inset-0 bg-black/50 z-50 hidden flex items-center justify-center">
        <div class="bg-white rounded-lg w-full max-w-md mx-4 overflow-hidden card-shadow transform transition-all duration-300 scale-95 opacity-0" id="settingsModal">
            <div class="bg-primary text-white px-6 py-4 flex justify-between items-center">
                <h3 class="font-bold">设置</h3>
                <button id="closeSettings" class="text-white hover:text-white/80">
                    <i class="fa fa-times"></i>
                </button>
            </div>
            <div class="p-6 space-y-4">
                <div>
                    <label class="block text-sm font-medium text-neutral-light mb-1">刷新间隔 (秒)</label>
                    <div class="flex items-center space-x-2">
                        <input type="number" id="refreshIntervalInput" min="2" value="<?php echo $refreshInterval; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary">
                        <span class="text-sm text-neutral-lighter">最小值: 2秒</span>
                    </div>
                </div>
                <div>
                    <label class="block text-sm font-medium text-neutral-light mb-1">每页显示数量</label>
                    <input type="number" id="pageSizeInput" min="1" value="<?php echo $pageSize; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary">
                </div>
                <div class="pt-4">
                    <button id="saveSettings" class="w-full bg-primary hover:bg-primary/90 text-white font-medium py-2 px-4 rounded-md transition-colors">
                        保存设置
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- 筛选面板 -->
    <div id="filterPanel" class="fixed bottom-0 left-0 right-0 bg-white rounded-t-xl p-4 card-shadow transform translate-y-full transition-transform duration-300 z-30">
        <div class="flex justify-between items-center mb-4">
            <h3 class="font-bold text-neutral">筛选条件</h3>
            <button id="closeFilter" class="text-neutral-light hover:text-neutral transition-colors">
                <i class="fa fa-times"></i>
            </button>
        </div>
        
        <form id="filterForm" class="space-y-4">
            <div>
                <label class="block text-sm font-medium text-neutral-light mb-1">日期筛选</label>
                <div class="grid grid-cols-5 gap-2 mb-2">
                    <button type="button" class="date-filter-btn py-1 px-2 text-xs rounded-full <?php echo $dateFilter == 'today' ? 'bg-primary text-white' : 'bg-neutral-lightest text-neutral-light'; ?>" data-filter="today">今天</button>
                    <button type="button" class="date-filter-btn py-1 px-2 text-xs rounded-full <?php echo $dateFilter == 'yesterday' ? 'bg-primary text-white' : 'bg-neutral-lightest text-neutral-light'; ?>" data-filter="yesterday">昨天</button>
                    <button type="button" class="date-filter-btn py-1 px-2 text-xs rounded-full <?php echo $dateFilter == 'this_week' ? 'bg-primary text-white' : 'bg-neutral-lightest text-neutral-light'; ?>" data-filter="this_week">本周</button>
                    <button type="button" class="date-filter-btn py-1 px-2 text-xs rounded-full <?php echo $dateFilter == 'this_month' ? 'bg-primary text-white' : 'bg-neutral-lightest text-neutral-light'; ?>" data-filter="this_month">本月</button>
                    <button type="button" class="date-filter-btn py-1 px-2 text-xs rounded-full <?php echo $dateFilter == 'custom' ? 'bg-primary text-white' : 'bg-neutral-lightest text-neutral-light'; ?>" data-filter="custom">自定义</button>
                </div>
                <div id="customDateRange" class="grid grid-cols-2 gap-2 <?php echo $dateFilter == 'custom' ? '' : 'hidden'; ?>">
                    <div>
                        <input type="date" id="startDate" value="<?php echo $startDate; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary text-sm">
                    </div>
                    <div>
                        <input type="date" id="endDate" value="<?php echo $endDate; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary text-sm">
                    </div>
                </div>
                <input type="hidden" id="dateFilterInput" value="<?php echo $dateFilter; ?>">
            </div>
            
            <div>
                <label class="block text-sm font-medium text-neutral-light mb-1">用户名</label>
                <select id="usernameInput" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary">
                    <option value="">全部</option>
                    <?php foreach ($usernames as $user): ?>
                        <option value="<?php echo htmlspecialchars($user); ?>" <?php echo $usernameFilter == $user ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($user); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div>
                <label class="block text-sm font-medium text-neutral-light mb-1">比赛类型</label>
                <select id="gameTypeInput" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary">
                    <option value="">全部</option>
                    <?php foreach ($gameTypes as $type): ?>
                        <option value="<?php echo htmlspecialchars($type); ?>" <?php echo $gameTypeFilter == $type ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($type); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="flex space-x-2">
                <button type="button" id="applyFilter" class="flex-1 bg-primary hover:bg-primary/90 text-white font-medium py-2 px-4 rounded-md transition-colors">
                    应用筛选
                </button>
                <button type="button" id="resetFilter" class="flex-1 bg-neutral-lightest hover:bg-gray-200 text-neutral font-medium py-2 px-4 rounded-md transition-colors">
                    重置
                </button>
            </div>
        </form>
    </div>

    <!-- 明细弹窗 -->
    <div id="detailModal" class="fixed inset-0 bg-black/50 z-50 hidden flex items-center justify-center">
        <div class="bg-white rounded-lg w-full max-w-md mx-4 max-h-[80vh] overflow-hidden card-shadow transform transition-all duration-300 scale-95 opacity-0" id="detailContent">
            <div class="bg-primary text-white px-6 py-4 flex justify-between items-center">
                <h3 class="font-bold" id="detailTitle">投注明细</h3>
                <button id="closeDetail" class="text-white hover:text-white/80">
                    <i class="fa fa-times"></i>
                </button>
            </div>
            <div class="p-4 overflow-y-auto max-h-[calc(80vh-100px)]" id="detailBody">
                <!-- 明细内容将通过JavaScript动态加载 -->
                <div class="flex justify-center py-10">
                    <div class="animate-spin rounded-full h-10 w-10 border-b-2 border-primary"></div>
                </div>
            </div>
        </div>
    </div>

    <!-- 主内容区 -->
    <main class="container mx-auto px-4 pt-20 pb-32">
        <!-- 统计卡片 -->
        <div class="grid grid-cols-3 gap-4 mb-6" id="summaryStats">
            <div class="bg-gradient-blue rounded-xl p-4 text-white shadow-lg transform transition-all duration-300 hover:scale-[1.02]">
                <div class="flex justify-between items-start mb-2">
                    <h3 class="text-xs font-medium">总注数</h3>
                    <i class="fa fa-list-alt text-white/70"></i>
                </div>
                <div class="text-xs font-bold" id="totalBets"><?php echo $totalStats['total_bets'] ?? 0; ?></div>
                <div class="text-xs text-white/80 mt-1">注</div>
            </div>
            <div class="bg-gradient-red rounded-xl p-4 text-white shadow-lg transform transition-all duration-300 hover:scale-[1.02]">
                <div class="flex justify-between items-start mb-2">
                    <h3 class="text-xs font-medium">总流水</h3>
                    <i class="fa fa-money text-white/70"></i>
                </div>
                <div class="text-xs font-bold" id="totalAmount"><?php echo number_format($totalStats['total_amount'] ?? 0, 2); ?></div>
                <div class="text-xs text-white/80 mt-1">元</div>
            </div>
            <div class="bg-gradient-green rounded-xl p-4 text-white shadow-lg transform transition-all duration-300 hover:scale-[1.02]">
                <div class="flex justify-between items-start mb-2">
                    <h3 class="text-xs font-medium">总输赢</h3>
                    <i class="fa fa-balance-scale text-white/70"></i>
                </div>
                <div class="text-xs font-bold" id="totalWinGold">
                    <?php 
                    $totalWin = $totalStats['total_win_gold'] ?? 0;
                    echo number_format($totalWin, 2);
                    ?>
                </div>
                <div class="text-xs text-white/80 mt-1">元</div>
            </div>
        </div>

        <!-- 筛选按钮及已选条件区域 -->
        <div class="flex flex-wrap justify-between items-center mb-4 gap-2">
            <div class="text-sm text-neutral-lighter">
                共 <span class="text-primary font-medium" id="daysCount"><?php echo count($dailyStats); ?></span> 天有数据
            </div>
            
            <!-- 已选条件标签容器 -->
            <div id="activeFilters" class="flex flex-wrap gap-2">
                <?php
                // 日期条件标签
                if ($dateFilter != 'all') {
                    $dateText = '';
                    switch ($dateFilter) {
                        case 'today': $dateText = '今天'; break;
                        case 'yesterday': $dateText = '昨天'; break;
                        case 'this_week': $dateText = '本周'; break;
                        case 'this_month': $dateText = '本月'; break;
                        case 'custom': $dateText = "{$startDate}至{$endDate}"; break;
                    }
                    echo "<span class='inline-flex items-center bg-primary/10 text-primary text-xs px-2 py-1 rounded-full'>
                            日期: {$dateText}
                            <button class='ml-1 text-primary/70 hover:text-primary' data-filter='dateFilter'>×</button>
                          </span>";
                }
                
                // 用户名条件标签
                if (!empty($usernameFilter)) {
                    echo "<span class='inline-flex items-center bg-primary/10 text-primary text-xs px-2 py-1 rounded-full'>
                            用户: {$usernameFilter}
                            <button class='ml-1 text-primary/70 hover:text-primary' data-filter='username'>×</button>
                          </span>";
                }
                
                // 比赛类型条件标签
                if (!empty($gameTypeFilter)) {
                    echo "<span class='inline-flex items-center bg-primary/10 text-primary text-xs px-2 py-1 rounded-full'>
                            类型: {$gameTypeFilter}
                            <button class='ml-1 text-primary/70 hover:text-primary' data-filter='gameType'>×</button>
                          </span>";
                }
                ?>
            </div>
            
            <button id="filterBtn" class="bg-primary hover:bg-primary/90 text-white px-4 py-2 rounded-full text-sm font-medium transition-colors flex items-center">
                <i class="fa fa-filter mr-1"></i> 筛选
            </button>
        </div>

        <!-- 每日统计列表 -->
        <div id="dailyStatsList" class="space-y-4 mb-8">
            <?php if (empty($dailyStats)): ?>
                <div class="bg-white rounded-xl p-6 text-center">
                    <i class="fa fa-search text-4xl text-neutral-lighter mb-3"></i>
                    <h3 class="text-neutral font-medium mb-2">没有找到匹配的统计数据</h3>
                    <p class="text-sm text-neutral-lighter">请尝试调整筛选条件或扩大时间范围</p>
                </div>
            <?php else: ?>
                <!-- 表头 -->
                <div class="bg-white rounded-xl overflow-hidden shadow-md">
                    <div class="grid grid-cols-12 gap-1 p-3 text-xs font-medium text-neutral-lighter border-b border-gray-100">
                        <div class="col-span-4">日期</div>
                        <div class="col-span-3 text-center">投注数</div>
                        <div class="col-span-2 text-right">投注额</div>
                        <div class="col-span-3 text-right">总输赢</div>
                    </div>
                </div>
                
                <!-- 数据行 -->
                <?php foreach ($dailyStats as $day): ?>
                    <div class="bg-white rounded-xl overflow-hidden shadow-md transform transition-all duration-300 hover:shadow-lg">
                        <div class="grid grid-cols-12 gap-1 p-3 text-sm">
                            <div class="col-span-4 flex flex-col justify-center">
                                <div class="font-medium text-neutral"><?php echo $day['game_date']; ?></div>
                                <div class="text-xs text-neutral-lighter"><?php echo $day['weekday']; ?></div>
                            </div>
                            <div class="col-span-2 flex items-center justify-end text-neutral font-medium">
                                <?php echo $day['bet_count']; ?>
                            </div>
                            <div class="col-span-3 flex items-center justify-end text-neutral font-medium">
                                <?php echo number_format($day['total_amount'], 2); ?>
                            </div>
                            <div class="col-span-3 flex items-center justify-end">
                                <button class="view-detail-btn font-medium transition-colors" 
                                        data-date="<?php echo $day['game_date']; ?>"
                                        style="color: <?php echo floatval($day['total_win_gold']) >= 0 ? '#52C41A' : '#FF4D4F'; ?>">
                                    <?php echo number_format($day['total_win_gold'], 2); ?>
                                </button>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <!-- 图表区域 -->
        <div class="bg-white rounded-xl overflow-hidden shadow-md p-4 mb-6 relative">
            <h3 class="text-neutral font-medium mb-4">用户输赢统计</h3>
            <div class="h-64">
                <canvas id="userWinChart"></canvas>
            </div>
            <!-- 图表无数据提示 -->
            <?php if (empty($userWinStats)): ?>
                <div class="absolute inset-4 flex items-center justify-center bg-white/80 pointer-events-none">
                    <div class="text-center text-neutral-lighter">
                        <i class="fa fa-bar-chart text-3xl mb-2"></i>
                        <p>没有可用的用户输赢数据</p>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <!-- 底部导航 -->
    <nav class="fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 py-2 px-4 z-20">
        <div class="flex justify-around">
            <a href="bet_list.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-home text-lg"></i>
                <span class="text-xs mt-1">首页</span>
            </a>
            <a href="bet_good.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-line-chart text-lg"></i>
                <span class="text-xs mt-1">数据</span>
            </a>
            <a href="bet_result.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-secondary">
                <i class="fa fa-trophy text-lg"></i>
                <span class="text-xs mt-1">结果</span>
            </a>
            <a href="bet_notice.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-bell text-lg"></i>
                <span class="text-xs mt-1">通知</span>
            </a>
            <a href="bet_my.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-user text-lg"></i>
                <span class="text-xs mt-1">我的</span>
            </a>
        </div>
    </nav>

    <script>
        // 当前筛选参数
        const currentParams = {
            AccountNo: '<?php echo $accountNo; ?>',
            dateFilter: '<?php echo $dateFilter; ?>',
            startDate: '<?php echo $startDate; ?>',
            endDate: '<?php echo $endDate; ?>',
            username: '<?php echo $usernameFilter; ?>',
            gameType: '<?php echo $gameTypeFilter; ?>',
            pageSize: <?php echo $pageSize; ?>,
            page: <?php echo $currentPage; ?>,
            refreshInterval: <?php echo $refreshInterval; ?>
        };

        // 图表数据
        const chartData = <?php echo json_encode($chartDataset); ?>;

        // 全局变量
        let refreshTimer = null;
        let isSettingInProgress = false; // 标记是否正在进行设置操作
        let userWinChart = null;

        // 初始化页面
        document.addEventListener('DOMContentLoaded', function() {
            // 初始化图表
            initChart();
            
            // 设置面板动画
            const settingsPanel = document.getElementById('settingsPanel');
            const settingsModal = document.getElementById('settingsModal');
            
            document.getElementById('settingsBtn').addEventListener('click', function() {
                // 打开设置面板时暂停自动刷新
                pauseAutoRefresh();
                isSettingInProgress = true;
                
                settingsPanel.classList.remove('hidden');
                setTimeout(() => {
                    settingsModal.classList.remove('scale-95', 'opacity-0');
                    settingsModal.classList.add('scale-100', 'opacity-100');
                }, 10);
            });
            
            document.getElementById('closeSettings').addEventListener('click', closeSettingsPanel);
            
            function closeSettingsPanel() {
                settingsModal.classList.remove('scale-100', 'opacity-100');
                settingsModal.classList.add('scale-95', 'opacity-0');
                setTimeout(() => {
                    settingsPanel.classList.add('hidden');
                    // 关闭设置面板后恢复自动刷新
                    isSettingInProgress = false;
                    resumeAutoRefresh();
                }, 300);
            }
            
            // 筛选面板动画
            const filterPanel = document.getElementById('filterPanel');
            
            document.getElementById('filterBtn').addEventListener('click', function() {
                // 打开筛选面板时暂停自动刷新
                pauseAutoRefresh();
                isSettingInProgress = true;
                
                filterPanel.classList.remove('translate-y-full');
            });
            
            document.getElementById('closeFilter').addEventListener('click', function() {
                filterPanel.classList.add('translate-y-full');
                // 关闭筛选面板后恢复自动刷新
                setTimeout(() => {
                    isSettingInProgress = false;
                    resumeAutoRefresh();
                }, 300);
            });
            
            // 日期筛选按钮
            document.querySelectorAll('.date-filter-btn').forEach(btn => {
                btn.addEventListener('click', function() {
                    document.querySelectorAll('.date-filter-btn').forEach(b => {
                        b.classList.remove('bg-primary', 'text-white');
                        b.classList.add('bg-neutral-lightest', 'text-neutral-light');
                    });
                    
                    this.classList.remove('bg-neutral-lightest', 'text-neutral-light');
                    this.classList.add('bg-primary', 'text-white');
                    
                    const filter = this.getAttribute('data-filter');
                    document.getElementById('dateFilterInput').value = filter;
                    
                    if (filter === 'custom') {
                        document.getElementById('customDateRange').classList.remove('hidden');
                    } else {
                        document.getElementById('customDateRange').classList.add('hidden');
                    }
                });
            });
            
            // 条件标签删除功能
            document.querySelectorAll('#activeFilters button').forEach(button => {
                button.addEventListener('click', function(e) {
                    e.preventDefault();
                    const filterType = this.getAttribute('data-filter');
                    
                    // 清除对应筛选条件
                    switch(filterType) {
                        case 'dateFilter':
                            currentParams.dateFilter = 'all';
                            currentParams.startDate = '';
                            currentParams.endDate = '';
                            break;
                        case 'username':
                            currentParams.username = '';
                            break;
                        case 'gameType':
                            currentParams.gameType = '';
                            break;
                    }
                    
                    currentParams.page = 1; // 重置到第一页
                    updateUrlAndReload();
                });
            });
            
            // 应用筛选
            document.getElementById('applyFilter').addEventListener('click', function() {
                try {
                    currentParams.dateFilter = document.getElementById('dateFilterInput').value;
                    currentParams.startDate = document.getElementById('startDate').value;
                    currentParams.endDate = document.getElementById('endDate').value;
                    currentParams.username = document.getElementById('usernameInput').value;
                    currentParams.gameType = document.getElementById('gameTypeInput').value;
                    currentParams.page = 1; // 重置到第一页
                    
                    // 验证日期
                    if (currentParams.dateFilter === 'custom') {
                        if (!currentParams.startDate || !currentParams.endDate) {
                            alert('自定义日期范围时，开始日期和结束日期不能为空');
                            return;
                        }
                        
                        if (new Date(currentParams.startDate) > new Date(currentParams.endDate)) {
                            alert('开始日期不能大于结束日期');
                            return;
                        }
                        
                        // 验证日期格式
                        const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
                        if (!dateRegex.test(currentParams.startDate) || !dateRegex.test(currentParams.endDate)) {
                            alert('日期格式不正确，请使用YYYY-MM-DD格式');
                            return;
                        }
                    }
                    
                    updateUrlAndReload();
                    filterPanel.classList.add('translate-y-full');
                    
                    // 应用筛选后恢复自动刷新
                    setTimeout(() => {
                        isSettingInProgress = false;
                        resumeAutoRefresh();
                    }, 300);
                } catch (error) {
                    console.error('应用筛选出错:', error);
                    alert('应用筛选时发生错误: ' + error.message);
                }
            });
            
            // 重置筛选
            document.getElementById('resetFilter').addEventListener('click', function() {
                try {
                    document.getElementById('dateFilterInput').value = 'all';
                    document.getElementById('startDate').value = '';
                    document.getElementById('endDate').value = '';
                    document.getElementById('usernameInput').value = '';
                    document.getElementById('gameTypeInput').value = '';
                    
                    document.querySelectorAll('.date-filter-btn').forEach(b => {
                        b.classList.remove('bg-primary', 'text-white');
                        b.classList.add('bg-neutral-lightest', 'text-neutral-light');
                    });
                    
                    currentParams.dateFilter = 'all';
                    currentParams.startDate = '';
                    currentParams.endDate = '';
                    currentParams.username = '';
                    currentParams.gameType = '';
                    currentParams.page = 1; // 重置到第一页
                    
                    updateUrlAndReload();
                    filterPanel.classList.add('translate-y-full');
                    
                    // 重置筛选后恢复自动刷新
                    setTimeout(() => {
                        isSettingInProgress = false;
                        resumeAutoRefresh();
                    }, 300);
                } catch (error) {
                    console.error('重置筛选出错:', error);
                    alert('重置筛选时发生错误: ' + error.message);
                }
            });
            
            // 保存设置
            document.getElementById('saveSettings').addEventListener('click', function() {
                try {
                    const refreshInterval = Math.max(2, parseInt(document.getElementById('refreshIntervalInput').value));
                    const pageSize = parseInt(document.getElementById('pageSizeInput').value);
                    
                    if (isNaN(refreshInterval) || refreshInterval < 2) {
                        alert('刷新间隔必须至少为2秒');
                        return;
                    }
                    
                    if (isNaN(pageSize) || pageSize < 1) {
                        alert('每页显示数量必须至少为1条');
                        return;
                    }
                    
                    currentParams.refreshInterval = refreshInterval;
                    currentParams.pageSize = pageSize;
                    currentParams.page = 1; // 重置到第一页
                    
                    updateUrlAndReload();
                    closeSettingsPanel();
                    
                    // 保存设置后恢复自动刷新
                    isSettingInProgress = false;
                    resumeAutoRefresh();
                } catch (error) {
                    console.error('保存设置出错:', error);
                    alert('保存设置时发生错误: ' + error.message);
                }
            });
            
            // 查看明细按钮
            document.querySelectorAll('.view-detail-btn').forEach(btn => {
                btn.addEventListener('click', function() {
                    const date = this.getAttribute('data-date');
                    loadDetailData(date);
                });
            });
            
            // 总输赢查看明细
            document.getElementById('totalWinGold').addEventListener('click', function() {
                loadDetailData('all');
            });
            
            // 关闭明细弹窗
            document.getElementById('closeDetail').addEventListener('click', function() {
                const detailContent = document.getElementById('detailContent');
                detailContent.classList.remove('scale-100', 'opacity-100');
                detailContent.classList.add('scale-95', 'opacity-0');
                setTimeout(() => {
                    document.getElementById('detailModal').classList.add('hidden');
                    // 关闭弹窗后恢复自动刷新
                    resumeAutoRefresh();
                }, 300);
            });
            
            // 手动刷新按钮
            document.getElementById('refreshBtn').addEventListener('click', function() {
                try {
                    refreshData();
                } catch (error) {
                    console.error('手动刷新出错:', error);
                    alert('刷新数据时发生错误: ' + error.message);
                }
            });
            
            // 开始自动刷新
            startAutoRefresh();
        });
        
        // 初始化图表
        function initChart() {
            const ctx = document.getElementById('userWinChart').getContext('2d');
            
            // 如果图表已存在，先销毁
            if (userWinChart) {
                userWinChart.destroy();
            }
            
            // 检查是否有数据
            if (chartData.datasets[0].data.length === 0) {
                return; // 无数据时不初始化图表，使用静态提示
            }
            
            userWinChart = new Chart(ctx, {
                type: 'bar',
                data: chartData,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        y: {
                            beginAtZero: true,
                            title: {
                                display: true,
                                text: '金额 (元)'
                            }
                        },
                        x: {
                            title: {
                                display: true,
                                text: '用户名'
                            }
                        }
                    },
                    plugins: {
                        legend: {
                            display: false
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return `输赢金额: ${context.raw.toFixed(2)} 元`;
                                }
                            }
                        }
                    }
                }
            });
        }
        
        // 更新URL并重新加载数据
        function updateUrlAndReload() {
            try {
                const queryParams = new URLSearchParams();
                for (const key in currentParams) {
                    if (currentParams[key] !== '' && currentParams[key] !== null && currentParams[key] !== undefined) {
                        queryParams.set(key, currentParams[key]);
                    }
                }
                
                const newUrl = window.location.pathname + '?' + queryParams.toString();
                window.history.replaceState({}, '', newUrl);
                
                refreshData();
            } catch (error) {
                console.error('更新URL出错:', error);
                alert('更新页面时发生错误: ' + error.message);
            }
        }
        
        // 开始自动刷新
        function startAutoRefresh() {
            // 如果正在进行设置操作，则不启动自动刷新
            if (isSettingInProgress) {
                return;
            }
            
            if (refreshTimer) {
                clearTimeout(refreshTimer);
            }
            
            refreshTimer = setTimeout(refreshData, currentParams.refreshInterval * 1000);
        }
        
        // 暂停自动刷新
        function pauseAutoRefresh() {
            if (refreshTimer) {
                clearTimeout(refreshTimer);
                refreshTimer = null;
            }
        }
        
        // 恢复自动刷新
        function resumeAutoRefresh() {
            // 如果不在设置操作中，则恢复自动刷新
            if (!isSettingInProgress) {
                startAutoRefresh();
            }
        }
        
        // 刷新数据
        function refreshData() {
            try {
                // 显示加载动画
                const statsList = document.getElementById('dailyStatsList');
                const originalContent = statsList.innerHTML;
                statsList.innerHTML = '<div class="flex justify-center items-center py-10"><div class="animate-spin rounded-full h-10 w-10 border-b-2 border-primary"></div></div>';
                
                // 使用fetch获取新数据
                fetch(window.location.href)
                    .then(response => {
                        if (!response.ok) {
                            throw new Error('网络响应不正常');
                        }
                        return response.text();
                    })
                    .then(html => {
                        // 创建临时DOM解析器
                        const parser = new DOMParser();
                        const doc = parser.parseFromString(html, 'text/html');
                        
                        // 更新统计列表
                        const newStatsList = doc.getElementById('dailyStatsList');
                        if (newStatsList) {
                            statsList.innerHTML = newStatsList.innerHTML;
                            
                            // 重新绑定查看明细事件
                            document.querySelectorAll('.view-detail-btn').forEach(btn => {
                                btn.addEventListener('click', function() {
                                    const date = this.getAttribute('data-date');
                                    loadDetailData(date);
                                });
                            });
                        }
                        
                        // 更新顶部统计数据
                        const newTotalBets = doc.getElementById('totalBets');
                        const newTotalAmount = doc.getElementById('totalAmount');
                        const newTotalWinGold = doc.getElementById('totalWinGold');
                        
                        if (newTotalBets && newTotalAmount && newTotalWinGold) {
                            document.getElementById('totalBets').textContent = newTotalBets.textContent;
                            document.getElementById('totalAmount').textContent = newTotalAmount.textContent;
                            document.getElementById('totalWinGold').textContent = newTotalWinGold.textContent;
                        }
                        
                        // 更新天数统计
                        const newDaysCount = doc.getElementById('daysCount');
                        if (newDaysCount) {
                            document.getElementById('daysCount').textContent = newDaysCount.textContent;
                        }
                        
                        // 更新活跃筛选条件
                        const newActiveFilters = doc.getElementById('activeFilters');
                        if (newActiveFilters) {
                            document.getElementById('activeFilters').innerHTML = newActiveFilters.innerHTML;
                            
                            // 重新绑定筛选条件删除事件
                            document.querySelectorAll('#activeFilters button').forEach(button => {
                                button.addEventListener('click', function(e) {
                                    e.preventDefault();
                                    const filterType = this.getAttribute('data-filter');
                                    
                                    // 清除对应筛选条件
                                    switch(filterType) {
                                        case 'dateFilter':
                                            currentParams.dateFilter = 'all';
                                            currentParams.startDate = '';
                                            currentParams.endDate = '';
                                            break;
                                        case 'username':
                                            currentParams.username = '';
                                            break;
                                        case 'gameType':
                                            currentParams.gameType = '';
                                            break;
                                    }
                                    
                                    currentParams.page = 1; // 重置到第一页
                                    updateUrlAndReload();
                                });
                            });
                        }
                        
                        // 更新图表数据
                        try {
                            // 从新HTML中提取图表数据
                            const scriptTag = doc.querySelector('script#chartData');
                            if (scriptTag) {
                                const newChartData = JSON.parse(scriptTag.textContent);
                                if (newChartData) {
                                    // 销毁现有图表
                                    if (userWinChart) {
                                        userWinChart.destroy();
                                    }
                                    
                                    // 清除可能存在的无数据提示
                                    const chartContainer = document.querySelector('#userWinChart').closest('div');
                                    const noDataElements = chartContainer.querySelectorAll('.absolute');
                                    noDataElements.forEach(el => el.remove());
                                    
                                    // 检查是否有新数据
                                    if (newChartData.datasets[0].data.length > 0) {
                                        // 创建新图表
                                        const ctx = document.getElementById('userWinChart').getContext('2d');
                                        userWinChart = new Chart(ctx, {
                                            type: 'bar',
                                            data: newChartData,
                                            options: {
                                                responsive: true,
                                                maintainAspectRatio: false,
                                                scales: {
                                                    y: {
                                                        beginAtZero: true,
                                                        title: {
                                                            display: true,
                                                            text: '金额 (元)'
                                                        }
                                                    },
                                                    x: {
                                                        title: {
                                                            display: true,
                                                            text: '用户名'
                                                        }
                                                    }
                                                },
                                                plugins: {
                                                    legend: {
                                                        display: false
                                                    },
                                                    tooltip: {
                                                        callbacks: {
                                                            label: function(context) {
                                                                return `输赢金额: ${context.raw.toFixed(2)} 元`;
                                                            }
                                                        }
                                                    }
                                                }
                                            }
                                        });
                                    } else {
                                        // 无数据时显示提示
                                        chartContainer.innerHTML += `
                                            <div class="absolute inset-4 flex items-center justify-center bg-white/80 pointer-events-none">
                                                <div class="text-center text-neutral-lighter">
                                                    <i class="fa fa-bar-chart text-3xl mb-2"></i>
                                                    <p>没有可用的用户输赢数据</p>
                                                </div>
                                            </div>
                                        `;
                                    }
                                }
                            }
                        } catch (chartError) {
                            console.error('更新图表时出错:', chartError);
                        }
                        
                        // 继续自动刷新
                        startAutoRefresh();
                    })
                    .catch(error => {
                        console.error('刷新数据出错:', error);
                        statsList.innerHTML = '<div class="bg-red-50 border-l-4 border-red-400 p-4 mb-4">刷新数据失败，请稍后重试</div>';
                        
                        // 继续自动刷新
                        startAutoRefresh();
                    });
            } catch (error) {
                console.error('刷新数据出错:', error);
                alert('刷新数据时发生错误: ' + error.message);
                
                // 继续自动刷新
                startAutoRefresh();
            }
        }
        
        // 加载明细数据（调用本页面接口）
        function loadDetailData(date) {
            try {
                // 暂停自动刷新
                pauseAutoRefresh();
                
                // 显示弹窗
                const detailModal = document.getElementById('detailModal');
                const detailContent = document.getElementById('detailContent');
                const detailTitle = document.getElementById('detailTitle');
                
                // 设置标题
                detailTitle.textContent = date === 'all' ? '所有投注明细' : `${date} 投注明细`;
                
                // 显示加载状态
                document.getElementById('detailBody').innerHTML = 
                    '<div class="flex justify-center py-10"><div class="animate-spin rounded-full h-10 w-10 border-b-2 border-primary"></div></div>';
                
                // 显示弹窗
                detailModal.classList.remove('hidden');
                setTimeout(() => {
                    detailContent.classList.remove('scale-95', 'opacity-0');
                    detailContent.classList.add('scale-100', 'opacity-100');
                }, 10);
                
                // 构建查询参数
                const params = new URLSearchParams(currentParams);
                params.set('action', 'get_details');
                params.set('date', date);
                
                // 使用fetch获取明细数据（调用本页面）
                fetch(window.location.pathname + '?' + params.toString())
                    .then(response => {
                        if (!response.ok) {
                            throw new Error('获取明细数据失败');
                        }
                        return response.text();
                    })
                    .then(html => {
                        document.getElementById('detailBody').innerHTML = html;
                    })
                    .catch(error => {
                        console.error('获取明细数据出错:', error);
                        document.getElementById('detailBody').innerHTML = 
                            '<div class="bg-red-50 border-l-4 border-red-400 p-4 text-center">获取明细数据失败，请稍后重试</div>';
                    });
            } catch (error) {
                console.error('加载明细数据出错:', error);
                alert('加载明细数据时发生错误: ' + error.message);
            }
        }
    </script>
    <!-- 图表数据存储，用于刷新时更新图表 -->
    <script id="chartData" type="application/json">
        <?php echo json_encode($chartDataset); ?>
    </script>
</body>
</html>
