
<?php
// 放在页面最顶部，确保在任何输出之前执行
require_once 'session_check.php';
require_once 'db.php';
?>


<?php
// 数据库连接

try {
    $pdo = new PDO("mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die("数据库连接失败: " . $e->getMessage());
}

// 处理清空通知请求
if (isset($_GET['action']) && $_GET['action'] === 'clear' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // 获取参数
        $accountNo = isset($_GET['AccountNo']) ? trim($_GET['AccountNo']) : '';
        // 验证账号参数
        if (empty($accountNo)) {
            throw new Exception('账号参数不能为空');
        }
        
        // 执行删除操作
        $deleteSql = "DELETE FROM TelSms_Logs WHERE AccountNo = :accountNo";
        $deleteStmt = $pdo->prepare($deleteSql);
        $deleteStmt->bindValue(':accountNo', $accountNo, PDO::PARAM_STR);
        $deleteStmt->execute();
        
        // 返回成功响应
        header('Content-Type: application/json');
        echo json_encode([
            'success' => true,
            'message' => '所有通知已清空'
        ]);
        exit;
    } catch (Exception $e) {
        // 返回错误响应
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => $e->getMessage()
        ]);
        exit;
    }
}

// 获取参数
$accountNo = isset($_GET['AccountNo']) ? trim($_GET['AccountNo']) : '';
if (empty($accountNo)) {
    $loginUrl = 'login.php';
    header('Location: ' . $loginUrl);
    exit;
}

$pageSize = isset($_GET['pageSize']) ? max(1, intval($_GET['pageSize'])) : 10;
$currentPage = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$refreshInterval = isset($_GET['refreshInterval']) ? max(2, intval($_GET['refreshInterval'])) : 10;

// 初始化错误信息数组
$errors = [];

// 计算偏移量
$offset = ($currentPage - 1) * $pageSize;

// 查询数据
$notices = [];
$totalRecords = 0;
$totalPages = 1;

try {
    // 查询总记录数
    $countSql = "SELECT COUNT(*) as total FROM TelSms_Logs WHERE AccountNo = :accountNo";
    $countStmt = $pdo->prepare($countSql);
    $countStmt->bindValue(':accountNo', $accountNo, PDO::PARAM_STR);
    $countStmt->execute();
    $totalRecords = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
    $totalPages = ceil($totalRecords / $pageSize);

    // 确保当前页在有效范围内
    if ($currentPage > $totalPages && $totalPages > 0) {
        $currentPage = $totalPages;
        $offset = ($currentPage - 1) * $pageSize;
    }

    // 查询通知数据
    $sql = "SELECT Contents, CreateDate FROM TelSms_Logs WHERE AccountNo = :accountNo ORDER BY CreateDate DESC LIMIT :offset, :pageSize";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':accountNo', $accountNo, PDO::PARAM_STR);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->bindValue(':pageSize', $pageSize, PDO::PARAM_INT);
    $stmt->execute();

    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
          
        $row['Contents'] = $row['Contents'];
        $row['CreateDate'] = date('Y-m-d H:i:s', strtotime($row['CreateDate']));
        $notices[] = $row;
    }
    
    // 添加调试输出
    echo "<!-- 总记录数: $totalRecords, 当前页: $currentPage, 每页显示: $pageSize, 当前页数据数量: " . count($notices) . " -->\n";
    
} catch (Exception $e) {
    $errors[] = '数据库查询失败: ' . $e->getMessage();
  //   throw new Exception("数据库查询失败: ". $e->getMessage());

}
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>消息通知</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdn.jsdelivr.net/npm/font-awesome@4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        "primary": '#165DFF',
                        "secondary": '#FF4D4F',
                        "neutral": '#1D2129',
                        'neutral-light': '#4E5969',
                        'neutral-lighter': '#86909C',
                        'neutral-lightest': '#F2F3F5'
                    },
                    fontFamily: {
                        sans: ['Inter', 'system-ui', 'sans-serif'],
                    },
                }
            }
        }
    </script>
    <style type="text/tailwindcss">
        @layer utilities {
            .content-auto {
                content-visibility: auto;
            }
            .scrollbar-hide {
                -ms-overflow-style: none;
                scrollbar-width: none;
            }
            .scrollbar-hide::-webkit-scrollbar {
                display: none;
            }
            .card-shadow {
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            }
            .bg-gradient-blue {
                background: linear-gradient(135deg, #165DFF 0%, #4080FF 100%);
            }
            .bg-gradient-purple {
                background: linear-gradient(135deg, #722ED1 0%, #EB0AA4 100%);
            }
        }
    </style>
</head>
<body class="bg-neutral-lightest min-h-screen font-sans">
    <!-- 顶部导航栏 -->
    <header class="bg-white shadow-md fixed top-0 left-0 right-0 z-10 transition-all duration-300">
        <div class="container mx-auto px-4 py-3 flex items-center justify-between">
            <div class="flex items-center space-x-2">
                <i class="fa fa-bell text-primary text-xl"></i>
                <h1 class="text-lg font-bold text-neutral">消息中心</h1>
                <h1 class="text-lg font-bold text-neutral" style="display: none;">账号：<?php  echo htmlspecialchars($accountNo); ?></h1>
            </div>
            <div class="flex items-center space-x-4">
                <button id="refreshBtn" class="text-primary hover:text-primary/80 transition-colors">
                    <i class="fa fa-refresh"></i>
                </button>
                <button id="clearBtn" class="text-secondary hover:text-secondary/80 transition-colors">
                    <i class="fa fa-trash"></i>
                </button>
                <button id="settingsBtn" class="text-neutral-light hover:text-neutral transition-colors">
                    <i class="fa fa-cog"></i>
                </button>
            </div>
        </div>
    </header>

    <!-- 错误提示 -->
    <?php if (!empty($errors)): ?>
        <div class="fixed top-16 left-0 right-0 z-50 bg-secondary text-white py-2 px-4">
            <div class="container mx-auto">
                <div class="flex items-center">
                    <i class="fa fa-exclamation-circle mr-2"></i>
                    <div>
                        <?php foreach ($errors as $error): ?>
                            <div><?php echo htmlspecialchars($error); ?></div>
                        <?php endforeach; ?>
                    </div>
                    <button id="closeError" class="ml-auto text-white hover:text-white/80">
                        <i class="fa fa-times"></i>
                    </button>
                </div>
            </div>
        </div>
        <script>
            document.getElementById('closeError').addEventListener('click', function() {
                this.closest('div').style.display = 'none';
            });
        </script>
    <?php endif; ?>

    <!-- 设置面板 -->
    <div id="settingsPanel" class="fixed inset-0 bg-black/50 z-50 hidden flex items-center justify-center">
        <div class="bg-white rounded-lg w-full max-w-md mx-4 overflow-hidden card-shadow transform transition-all duration-300 scale-95 opacity-0" id="settingsModal">
            <div class="bg-primary text-white px-6 py-4 flex justify-between items-center">
                <h3 class="font-bold">设置</h3>
                <button id="closeSettings" class="text-white hover:text-white/80">
                    <i class="fa fa-times"></i>
                </button>
            </div>
            <div class="p-6 space-y-4">
                <div>
                    <label class="block text-sm font-medium text-neutral-light mb-1">刷新间隔 (秒)</label>
                    <div class="flex items-center space-x-2">
                        <input type="number" id="refreshIntervalInput" min="2" value="<?php echo $refreshInterval; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary">
                        <span class="text-sm text-neutral-lighter">最小值: 2秒</span>
                    </div>
                </div>
                <div>
                    <label class="block text-sm font-medium text-neutral-light mb-1">每页显示数量</label>
                    <input type="number" id="pageSizeInput" min="1" value="<?php echo $pageSize; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary">
                </div>
                <div class="pt-4">
                    <button id="saveSettings" class="w-full bg-primary hover:bg-primary/90 text-white font-medium py-2 px-4 rounded-md transition-colors">
                        保存设置
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- 清空确认弹窗 -->
    <div id="clearConfirmPanel" class="fixed inset-0 bg-black/50 z-50 hidden flex items-center justify-center">
        <div class="bg-white rounded-lg w-full max-w-md mx-4 overflow-hidden card-shadow transform transition-all duration-300 scale-95 opacity-0" id="clearConfirmModal">
            <div class="bg-secondary text-white px-6 py-4">
                <h3 class="font-bold">确认清空</h3>
            </div>
            <div class="p-6 space-y-4">
                <p class="text-neutral">您确定要清空当前账号的所有通知吗？此操作不可恢复。</p>
                <div class="flex space-x-3 pt-2">
                    <button id="cancelClear" class="flex-1 px-4 py-2 border border-gray-200 rounded-md text-neutral hover:bg-gray-50 transition-colors">
                        取消
                    </button>
                    <button id="confirmClear" class="flex-1 px-4 py-2 bg-secondary text-white rounded-md hover:bg-secondary/90 transition-colors">
                        确认清空
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- 主内容区 -->
    <main class="container mx-auto px-4 pt-20 pb-32">
        <!-- 统计卡片 -->
        <div class="grid grid-cols-1 mb-6">
            <div class="bg-gradient-purple rounded-xl p-4 text-white shadow-lg transform transition-all duration-300 hover:scale-[1.02]">
                <div class="flex justify-between items-start mb-2">
                    <h3 class="text-sm font-medium">消息总数</h3>
                    <i class="fa fa-envelope text-white/70"></i>
                </div>
                <div class="text-2xl font-bold"><?php echo $totalRecords; ?></div>
                <div class="text-xs text-white/80 mt-1">条</div>
            </div>
        </div>

        <!-- 通知列表 -->
        <div id="noticeList" class="grid grid-cols-1 gap-4">
            <?php if (empty($notices)): ?>
                <div class="bg-white rounded-xl p-6 text-center">
                    <i class="fa fa-inbox text-4xl text-neutral-lighter mb-3"></i>
                    <h3 class="text-neutral font-medium mb-2">没有消息通知</h3>
                    <p class="text-sm text-neutral-lighter">您当前没有任何消息通知</p>
                </div>
            <?php else: ?>
                <?php foreach ($notices as $notice): ?>
                    <?php
                    $content = json_decode($notice['Contents'], true); // 只解析一次

                    
                    $createDate = $notice['CreateDate'];
                    
                    // 解析标题信息
                    $titleParts = [];
                    if (!empty($content['title'])) {
                        $titleParts = explode(';', $content['title']);
                    }
                    
                    // 解析内容信息
                    $contentItems = [];
                    if (!empty($content['content'])) {
                        $lines = explode('\n', $content['content']);
                        foreach ($lines as $line) {
                            $line = trim($line, ';');
                            if (!empty($line)) {
                                if (strpos($line, '-') !== false) {
                                    list($key, $value) = explode('-', $line, 2);
                                    $contentItems[trim($key)] = trim($value);
                                } elseif (strpos($line, ';') !== false) {
                                    list($key, $value) = explode(';', $line, 2);
                                    $contentItems[trim($key)] = trim($value);
                                } else {
                                    // 处理单行内容的情况，如 "账号状态报告"
                                    $contentItems['内容'] = trim($line);
                                }
                            }
                        }
                    }
                    
                    // 确定卡片样式
                    $statusClass = 'bg-white';
                    $statusIcon = 'fa-info-circle text-primary';
                    
                    if (!empty($content['status']) && $content['status'] === '成功') {
                        $statusClass = 'bg-green-50 border-l-4 border-green-500';
                        $statusIcon = 'fa-check-circle text-green-500';
                    } elseif (!empty($content['status']) && $content['status'] === '失败') {
                        $statusClass = 'bg-red-50 border-l-4 border-red-500';
                        $statusIcon = 'fa-exclamation-circle text-red-500';
                    }
                    
                    // 处理 message 字段
                    $message = !empty($content['message']) ? $content['message'] : '';
                    ?>  
                    
                    <div class="notice-card <?php echo $statusClass; ?> rounded-xl overflow-hidden shadow-md transform transition-all duration-300 hover:shadow-lg">
                        <div class="p-4">
                            <div class="flex justify-between items-start mb-3">
                                <div class="flex items-center">
                                    <i class="fa <?php echo $statusIcon; ?> mr-2"></i>
                                    <h3 class="text-sm font-medium text-neutral">
                                        <?php echo !empty($content['title']) ? htmlspecialchars($content['title']) : '系统通知'; ?>
                                    </h3>
                                </div>
                                <div class="text-xs text-neutral-lighter">
                                    <?php echo htmlspecialchars($createDate); ?>
                                </div>
                            </div>
                            
                            <div class="space-y-2 mb-3">
                                <?php if (!empty($contentItems)): ?>
                                    <?php foreach ($contentItems as $key => $value): ?>
                                        <div class="flex">
                                            <span class="text-xs text-neutral-lighter w-24"><?php echo htmlspecialchars($key); ?>:</span>
                                            <span class="text-xs text-neutral flex-1"><?php echo htmlspecialchars($value); ?></span>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <div class="text-sm text-neutral">
                                        <?php echo !empty($content['content']) ? htmlspecialchars($content['content']) : '无详细内容'; ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <?php if (!empty($message)): ?>
                                <div class="mt-2 text-sm text-neutral-lighter">
                                    <?php echo htmlspecialchars($message); ?>
                                </div>
                            <?php endif; ?>
                            
                            <?php if (!empty($content['status'])): ?>
                                <div class="flex justify-end">
                                    <span class="inline-block px-2 py-1 text-xs rounded-full <?php echo $content['status'] === '成功' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                                        <?php echo htmlspecialchars($content['status']); ?>
                                    </span>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <!-- 分页 -->
        <?php if ($totalPages > 1): ?>
            <div class="flex justify-between items-center mt-6">
                <button id="prevPage" class="px-4 py-2 rounded-md bg-white hover:bg-gray-50 text-neutral-light border border-gray-200 disabled:opacity-50 disabled:pointer-events-none transition-colors" <?php echo $currentPage == 1 ? 'disabled' : ''; ?>>
                    <i class="fa fa-angle-left mr-1"></i> 上一页
                </button>
                
                <div class="flex items-center space-x-1">
                    <?php
                    $visiblePages = 5;
                    $startPage = max(1, $currentPage - floor($visiblePages / 2));
                    $endPage = min($startPage + $visiblePages - 1, $totalPages);
                    
                    if ($startPage > 1) {
                        echo '<span class="px-2 py-1 text-sm text-neutral-light">...</span>';
                    }
                    
                    for ($i = $startPage; $i <= $endPage; $i++) {
                        if ($i == $currentPage) {
                            echo '<span class="px-3 py-1 bg-primary text-white rounded-md text-sm font-medium">' . $i . '</span>';
                        } else {
                            echo '<button class="page-btn px-3 py-1 hover:bg-gray-100 rounded-md text-sm text-neutral-light transition-colors">' . $i . '</button>';
                        }
                    }
                    
                    if ($endPage < $totalPages) {
                        echo '<span class="px-2 py-1 text-sm text-neutral-light">...</span>';
                    }
                    ?>
                </div>
                
                <button id="nextPage" class="px-4 py-2 rounded-md bg-white hover:bg-gray-50 text-neutral-light border border-gray-200 disabled:opacity-50 disabled:pointer-events-none transition-colors" <?php echo $currentPage >= $totalPages ? 'disabled' : ''; ?>>
                    下一页 <i class="fa fa-angle-right ml-1"></i>
                </button>
            </div>
        <?php endif; ?>
    </main>
    <script>
        // 当前账号参数
        const currentAccount = '<?php echo htmlspecialchars($accountNo); ?>';

        // 初始化页面
        document.addEventListener('DOMContentLoaded', function() {
            // 刷新按钮
            document.getElementById('refreshBtn').addEventListener('click', function() {
                try {
                    window.location.reload();
                } catch (error) {
                    console.error('刷新出错:', error);
                    alert('刷新时发生错误: ' + error.message);
                }
            });

            // 设置按钮 - 暂时使用相同的设置面板
            document.getElementById('settingsBtn').addEventListener('click', function() {
                alert('设置功能将在后续版本中实现');
            });

            // 获取当前页面文件名
            const currentPage = window.location.pathname.split('/').pop();
            const navLinks = document.querySelectorAll('nav a');

            navLinks.forEach(link => {
                const linkHref = link.getAttribute('href').split('?')[0];
                if (linkHref === currentPage) {
                    link.classList.add('text-secondary');
                }
            });
        });
    </script>
    <!-- 底部导航 -->
     <nav class="fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 py-2 px-4 z-20">
        <div class="flex justify-around">
            <a href="bet_list.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-home text-lg"></i>
                <span class="text-xs mt-1">首页</span>
            </a>
            <a href="bet_good.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-line-chart text-lg"></i>
                <span class="text-xs mt-1">数据</span>
            </a>
            <a href="bet_result.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-trophy text-lg"></i>
                <span class="text-xs mt-1">结果</span>
            </a>
            <a href="bet_notice.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-bell text-lg"></i>
                <span class="text-xs mt-1">通知</span>
            </a>
            <a href="bet_my.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-user text-lg"></i>
                <span class="text-xs mt-1">我的</span>
            </a>
        </div>
    </nav>

    <script>
        // 当前筛选参数
        const currentParams = {
            AccountNo: '<?php echo $accountNo; ?>',
            pageSize: <?php echo $pageSize; ?>,
            page: <?php echo $currentPage; ?>,
            refreshInterval: <?php echo $refreshInterval; ?>
        };

        // 自动刷新相关变量
        let refreshTimer = null;
        let isSettingInProgress = false;

        // 初始化页面
        document.addEventListener('DOMContentLoaded', function() {
            // 设置面板动画
            const settingsPanel = document.getElementById('settingsPanel');
            const settingsModal = document.getElementById('settingsModal');

            document.getElementById('settingsBtn').addEventListener('click', function() {
                // 打开设置面板时暂停自动刷新
                pauseAutoRefresh();
                isSettingInProgress = true;
                
                settingsPanel.classList.remove('hidden');
                setTimeout(() => {
                    settingsModal.classList.remove('scale-95', 'opacity-0');
                    settingsModal.classList.add('scale-100', 'opacity-100');
                }, 10);
            });
            
            document.getElementById('closeSettings').addEventListener('click', closeSettingsPanel);
            
            function closeSettingsPanel() {
                settingsModal.classList.remove('scale-100', 'opacity-100');
                settingsModal.classList.add('scale-95', 'opacity-0');
                setTimeout(() => {
                    settingsPanel.classList.add('hidden');
                    // 关闭设置面板后恢复自动刷新
                    isSettingInProgress = false;
                    resumeAutoRefresh();
                }, 300);
            }
            
            // 保存设置
            document.getElementById('saveSettings').addEventListener('click', function() {
                try {
                    const refreshInterval = Math.max(2, parseInt(document.getElementById('refreshIntervalInput').value));
                    const pageSize = parseInt(document.getElementById('pageSizeInput').value);
                    
                    if (isNaN(refreshInterval) || refreshInterval < 2) {
                        alert('刷新间隔必须至少为2秒');
                        return;
                    }
                    
                    if (isNaN(pageSize) || pageSize < 1) {
                        alert('每页显示数量必须至少为1条');
                        return;
                    }
                    
                    currentParams.refreshInterval = refreshInterval;
                    currentParams.pageSize = pageSize;
                    currentParams.page = 1; // 重置到第一页
                    
                    updateUrlAndReload();
                    closeSettingsPanel();
                } catch (error) {
                    console.error('保存设置出错:', error);
                    alert('保存设置时发生错误: ' + error.message);
                }
            });
            
            // 分页按钮
            document.getElementById('prevPage').addEventListener('click', function() {
                if (currentParams.page > 1) {
                    currentParams.page--;
                    updateUrlAndReload();
                }
            });
            
            document.getElementById('nextPage').addEventListener('click', function() {
                const totalPages = <?php echo $totalPages; ?>;
                if (currentParams.page < totalPages) {
                    currentParams.page++;
                    updateUrlAndReload();
                }
            });
            
            // 页码按钮
            document.querySelectorAll('.page-btn').forEach(btn => {
                btn.addEventListener('click', function() {
                    const page = parseInt(this.textContent);
                    currentParams.page = page;
                    updateUrlAndReload();
                });
            });
            
            // 手动刷新按钮
            document.getElementById('refreshBtn').addEventListener('click', function() {
                try {
                    refreshData();
                } catch (error) {
                    console.error('手动刷新出错:', error);
                    alert('刷新数据时发生错误: ' + error.message);
                }
            });
            
            // 清空通知相关逻辑
            const clearPanel = document.getElementById('clearConfirmPanel');
            const clearModal = document.getElementById('clearConfirmModal');
            
            // 打开清空确认弹窗
            document.getElementById('clearBtn').addEventListener('click', function() {
                pauseAutoRefresh();
                clearPanel.classList.remove('hidden');
                setTimeout(() => {
                    clearModal.classList.remove('scale-95', 'opacity-0');
                    clearModal.classList.add('scale-100', 'opacity-100');
                }, 10);
            });
            
            // 关闭清空确认弹窗
            document.getElementById('cancelClear').addEventListener('click', function() {
                closeClearPanel();
            });
            
            function closeClearPanel() {
                clearModal.classList.remove('scale-100', 'opacity-100');
                clearModal.classList.add('scale-95', 'opacity-0');
                setTimeout(() => {
                    clearPanel.classList.add('hidden');
                    resumeAutoRefresh();
                }, 300);
            }
            
            // 确认清空通知
            document.getElementById('confirmClear').addEventListener('click', function() {
                // 显示加载状态
                this.innerHTML = '<i class="fa fa-spinner fa-spin mr-1"></i> 处理中...';
                this.disabled = true;
                
                // 发送清空请求
                fetch('?AccountNo=<?php echo urlencode($accountNo); ?>&action=clear', { method: 'POST' })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            // 清空成功后刷新页面
                            window.location.reload();
                        } else {
                            alert('清空失败: ' + (data.message || '未知错误'));
                            closeClearPanel();
                        }
                    })
                    .catch(error => {
                        console.error('清空请求失败:', error);
                        alert('操作失败，请重试');
                        closeClearPanel();
                    });
            });
            
            // 开始自动刷新
            startAutoRefresh();
        });
        
        // 更新URL并重新加载数据
        function updateUrlAndReload() {
            try {
                const queryParams = new URLSearchParams();
                for (const key in currentParams) {
                    if (currentParams[key] !== '' && currentParams[key] !== null && currentParams[key] !== undefined) {
                        queryParams.set(key, currentParams[key]);
                    }
                }
                
                const newUrl = window.location.pathname + '?' + queryParams.toString();
                window.history.replaceState({}, '', newUrl);
                
                refreshData();
            } catch (error) {
                console.error('更新URL出错:', error);
                alert('更新页面时发生错误: ' + error.message);
            }
        }
        
        // 开始自动刷新
        function startAutoRefresh() {
            // 如果正在进行设置操作，则不启动自动刷新
            if (isSettingInProgress) {
                return;
            }
            
            if (refreshTimer) {
                clearTimeout(refreshTimer);
            }
            
            refreshTimer = setTimeout(refreshData, currentParams.refreshInterval * 1000);
        }
        
        // 暂停自动刷新
        function pauseAutoRefresh() {
            if (refreshTimer) {
                clearTimeout(refreshTimer);
                refreshTimer = null;
            }
        }
        
        // 恢复自动刷新
        function resumeAutoRefresh() {
            // 如果不在设置操作中，则恢复自动刷新
            if (!isSettingInProgress) {
                startAutoRefresh();
            }
        }
        
        // 刷新数据
        function refreshData() {
            try {
                // 显示加载动画
                const noticeList = document.getElementById('noticeList');
                const originalContent = noticeList.innerHTML;
                noticeList.innerHTML = '<div class="flex justify-center items-center py-10"><div class="animate-spin rounded-full h-10 w-10 border-b-2 border-primary"></div></div>';
                
                // 模拟AJAX请求
                setTimeout(() => {
                    window.location.reload();
                    
                    // 继续自动刷新
                    startAutoRefresh();
                }, 500);
            } catch (error) {
                console.error('刷新数据出错:', error);
                alert('刷新数据时发生错误: ' + error.message);
                
                // 恢复原始内容
                const noticeList = document.getElementById('noticeList');
                noticeList.innerHTML = '<div class="bg-red-50 border-l-4 border-red-400 p-4 mb-4">刷新数据失败，请稍后重试</div>';
                
                // 继续自动刷新
                startAutoRefresh();
            }
        }
    </script>
</body>
</html>