<?php
// 放在页面最顶部，确保在任何输出之前执行
require_once 'session_check.php';
require_once 'db.php';
?>


<?php
// 数据库连接


// 初始化变量
$adminInfo = [];
$planeGroups = [];
$robots = [];
$errors = [];
$validityPeriod = '';

try {
    // 连接数据库
    $pdo = new PDO("mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // 从会话获取当前登录用户名（由session_check.php确保已登录）
    $currentUserName = $_SESSION['admin_username'];

    // 查询管理员信息
    try {
        $sql = "SELECT UserName, Password, StartDate, EndDate, Status, ServerIP, `From` 
                FROM TelSms_Admin 
                WHERE UserName = :userName";
        
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':userName', $currentUserName, PDO::PARAM_STR);
        $stmt->execute();
        
        $adminInfo = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$adminInfo) {
            $errors[] = "未找到当前用户的信息，请联系管理员";
        } else {
            // 格式化有效期
            if (!empty($adminInfo['StartDate']) && !empty($adminInfo['EndDate'])) {
                $startDate = strtotime($adminInfo['StartDate']);
                $endDate = strtotime($adminInfo['EndDate']);
                
                if ($startDate !== false && $endDate !== false) {
                    $validityPeriod = date('Y-m-d', $startDate) . ' - ' . date('Y-m-d', $endDate);
                } else {
                    $validityPeriod = "日期格式错误";
                }
            }
        }
    } catch (PDOException $e) {
        $errors[] = "管理员信息查询失败: " . $e->getMessage();
    }

    // 查询飞机群列表
    try {
        $sql = "SELECT AccountNo, ChatId 
                FROM TelSms_Account 
                WHERE UserName = :userName
                ORDER BY AccountNo";
        
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':userName', $currentUserName, PDO::PARAM_STR);
        $stmt->execute();
        
        $planeGroups = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        $errors[] = "飞机群列表查询失败: " . $e->getMessage();
    }

    // 查询HG机器人列表
    try {
        $sql = "SELECT BotId, HgAdminList, AccountNoList, Status 
                FROM TelSms_BotConfig 
                WHERE AdminUserName = :userName
                ORDER BY BotId";
        
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':userName', $currentUserName, PDO::PARAM_STR);
        $stmt->execute();
        
        $robots = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        $errors[] = "HG机器人列表查询失败: " . $e->getMessage();
    }
} catch (PDOException $e) {
    $errors[] = "数据库连接失败: " . $e->getMessage();
}

// 机器人状态文本映射
$botStatusMap = [
    1 => ['text' => '空闲', 'class' => 'bg-blue-100 text-blue-600'],
    2 => ['text' => '运行中', 'class' => 'bg-green-100 text-green-600'],
    3 => ['text' => '机器人有异常请检查', 'class' => 'bg-red-100 text-red-600']
];
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>飞信球探 - 个人中心</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdn.jsdelivr.net/npm/font-awesome@4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#165DFF',
                        secondary: '#FF4D4F',
                        success: '#52C41A',
                        neutral: '#1D2129',
                        'neutral-light': '#4E5969',
                        'neutral-lighter': '#86909C',
                        'neutral-lightest': '#F2F3F5'
                    },
                    fontFamily: {
                        sans: ['Inter', 'system-ui', 'sans-serif'],
                    },
                }
            }
        }
    </script>
    <style type="text/tailwindcss">
        @layer utilities {
            .content-auto {
                content-visibility: auto;
            }
            .card-shadow {
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            }
            .bg-gradient-blue {
                background: linear-gradient(135deg, #165DFF 0%, #4080FF 100%);
            }
            .card-hover {
                transition: transform 0.2s, box-shadow 0.2s;
            }
            .card-hover:hover {
                transform: translateY(-2px);
                box-shadow: 0 6px 24px rgba(0, 0, 0, 0.12);
            }
        }
    </style>
</head>
<body class="bg-neutral-lightest min-h-screen font-sans">
    <!-- 顶部导航栏 -->
    <header class="bg-white shadow-md fixed top-0 left-0 right-0 z-10 transition-all duration-300">
        <div class="container mx-auto px-4 py-3 flex items-center justify-between">
            <div class="flex items-center space-x-2">
                <i class="fa fa-bolt text-primary text-xl"></i>
                <h1 class="text-lg font-bold text-neutral">个人中心</h1>
            </div>
            <div class="flex items-center space-x-4">
                <button id="refreshBtn" class="text-primary hover:text-primary/80 transition-colors">
                    <i class="fa fa-refresh"></i>
                </button>
                <button id="settingsBtn" class="text-neutral-light hover:text-neutral transition-colors">
                    <i class="fa fa-cog"></i>
                </button>
                <button id="logoutBtn" class="text-neutral-light hover:text-neutral transition-colors">
                    <i class="fa fa-sign-out"></i>
                </button>
            </div>
        </div>
    </header>

    <!-- 错误提示 -->
    <?php if (!empty($errors)): ?>
        <div class="fixed top-16 left-0 right-0 z-50 bg-secondary text-white py-2 px-4">
            <div class="container mx-auto">
                <div class="flex items-center">
                    <i class="fa fa-exclamation-circle mr-2"></i>
                    <div>
                        <?php foreach ($errors as $error): ?>
                            <div><?php echo htmlspecialchars($error); ?></div>
                        <?php endforeach; ?>
                    </div>
                    <button id="closeError" class="ml-auto text-white hover:text-white/80">
                        <i class="fa fa-times"></i>
                    </button>
                </div>
            </div>
        </div>
    <?php endif; ?>

    <!-- 主内容区 -->
    <main class="container mx-auto px-4 pt-20 pb-32">
        <?php if (empty($errors)): ?>
            <!-- 个人信息卡片（缩小版） -->
            <div class="bg-white rounded-xl overflow-hidden card-shadow mb-6 card-hover">
                <div class="bg-gradient-blue p-4 text-white">
                    <div class="flex items-center">
                        <div class="w-12 h-12 rounded-full bg-white/20 flex items-center justify-center mr-3">
                            <i class="fa fa-user-circle text-xl"></i>
                        </div>
                        <div>
                            <h2 class="text-base font-bold">个人账户信息</h2>
                            <p class="text-white/80 text-sm mt-0.5"><?php echo htmlspecialchars($adminInfo['UserName']); ?></p>
                        </div>
                    </div>
                </div>
                
                <div class="p-4 space-y-3">
                    <div class="grid grid-cols-2 gap-3">
                        <!-- 用户名 -->
                        <div class="flex items-center p-2.5 bg-neutral-lightest rounded-lg">
                            <i class="fa fa-id-card-o text-primary mr-2 text-sm"></i>
                            <span class="text-neutral-light text-sm">用户名</span>
                            <span class="ml-auto font-medium text-neutral text-sm"><?php echo htmlspecialchars($adminInfo['UserName']); ?></span>
                        </div>
                        
                        <!-- 账户状态 -->
                        <div class="flex items-center p-2.5 bg-neutral-lightest rounded-lg">
                            <i class="fa fa-toggle-on text-primary mr-2 text-sm"></i>
                            <span class="text-neutral-light text-sm">账户状态</span>
                            <span class="ml-auto">
                                <?php 
                                $statusText = $adminInfo['Status'] == 1 ? '启用' : '禁用';
                                $statusClass = $adminInfo['Status'] == 1 ? 'bg-success/10 text-success' : 'bg-secondary/10 text-secondary';
                                echo "<span class='px-2 py-0.5 rounded-full text-xs {$statusClass}'>{$statusText}</span>";
                                ?>
                            </span>
                        </div>
                        
                        <!-- 有效期 -->
                        <div class="flex items-center p-2.5 bg-neutral-lightest rounded-lg">
                            <i class="fa fa-calendar text-primary mr-2 text-sm"></i>
                            <span class="text-neutral-light text-sm">有效期</span>
                            <span class="ml-auto font-medium text-neutral text-sm">
                                <?php echo $validityPeriod ?: '未设置'; ?>
                                <?php 
                                if (!empty($adminInfo['EndDate']) && strtotime($adminInfo['EndDate']) !== false && strtotime($adminInfo['EndDate']) < time()) {
                                    echo '<span class="ml-1 text-xs bg-secondary/10 text-secondary px-2 py-0.5 rounded-full">已过期</span>';
                                }
                                ?>
                            </span>
                        </div>
                        
                        <!-- 服务器IP -->
                        <div class="flex items-center p-2.5 bg-neutral-lightest rounded-lg">
                            <i class="fa fa-server text-primary mr-2 text-sm"></i>
                            <span class="text-neutral-light text-sm">服务器IP</span>
                            <span class="ml-auto font-medium text-neutral text-sm"><?php echo htmlspecialchars($adminInfo['ServerIP'] ?: '未设置'); ?></span>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- 飞机群列表卡片 -->
            <div class="bg-white rounded-xl overflow-hidden card-shadow mb-6 card-hover">
                <div class="p-4 border-b border-gray-100">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center">
                            <i class="fa fa-users text-primary mr-2"></i>
                            <h2 class="font-bold text-neutral">我的飞机群列表</h2>
                        </div>
                        <span class="text-xs text-neutral-lighter bg-neutral-lightest px-2 py-1 rounded-full">
                            共 <?php echo count($planeGroups); ?> 个
                        </span>
                    </div>
                </div>
                
                <div class="p-4">
                    <?php if (!empty($planeGroups)): ?>
                        <div class="space-y-3">
                            <?php foreach ($planeGroups as $group): ?>
                                <div class="p-3 bg-neutral-lightest rounded-lg flex flex-col sm:flex-row sm:items-center justify-between">
                                    <div class="space-y-1 sm:space-y-0">
                                        <div class="flex items-center">
                                            <span class="text-neutral-light text-sm mr-2">飞机群编号:</span>
                                            <span class="font-medium text-neutral"><?php echo htmlspecialchars($group['AccountNo']); ?></span>
                                        </div>
                                        <div class="flex items-center">
                                            <span class="text-neutral-light text-sm mr-2">会话ID:</span>
                                            <span class="font-medium text-neutral"><?php echo htmlspecialchars($group['ChatId'] ?: '未设置'); ?></span>
                                        </div>
                                    </div>
                                    <a href="bet_list.php?AccountNo=<?php echo urlencode($group['AccountNo']); ?>" 
                                       class="mt-2 sm:mt-0 text-primary hover:text-primary/80 text-sm flex items-center transition-colors">
                                        <span>查看投注数据</span>
                                        <i class="fa fa-angle-right ml-1"></i>
                                    </a>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <div class="text-center py-6 text-neutral-lighter">
                            <i class="fa fa-users text-2xl mb-2"></i>
                            <p>暂无飞机群数据</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- HG机器人列表卡片 -->
            <div class="bg-white rounded-xl overflow-hidden card-shadow mb-6 card-hover">
                <div class="p-4 border-b border-gray-100">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center">
                            <i class="fa fa-robot text-primary mr-2"></i>
                            <h2 class="font-bold text-neutral">我的HG机器人列表</h2>
                        </div>
                        <span class="text-xs text-neutral-lighter bg-neutral-lightest px-2 py-1 rounded-full">
                            共 <?php echo count($robots); ?> 个
                        </span>
                    </div>
                </div>
                
                <div class="p-4">
                    <?php if (!empty($robots)): ?>
                        <div class="space-y-3">
                            <?php foreach ($robots as $robot): ?>
                                <div class="p-3 bg-neutral-lightest rounded-lg">
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-2">
                                        <div class="flex items-center">
                                            <span class="text-neutral-light text-sm mr-2">机器人编号:</span>
                                            <span class="font-medium text-neutral"><?php echo htmlspecialchars($robot['BotId']); ?></span>
                                        </div>
                                        
                                        <div class="flex items-center">
                                            <span class="text-neutral-light text-sm mr-2">当前状态:</span>
                                            <?php
                                            $status = isset($robot['Status']) ? $robot['Status'] : 3;
                                            $statusInfo = $botStatusMap[$status] ?? $botStatusMap[3];
                                            echo "<span class='px-2 py-0.5 rounded-full text-xs {$statusInfo['class']}'>{$statusInfo['text']}</span>";
                                            ?>
                                        </div>
                                        
                                        <div class="flex items-start">
                                            <span class="text-neutral-light text-sm mr-2 mt-0.5">监听HG账号:</span>
                                            <span class="font-medium text-neutral text-sm break-all"><?php echo htmlspecialchars($robot['HgAdminList'] ?: '未设置'); ?></span>
                                        </div>
                                        
                                        <div class="flex items-start">
                                            <span class="text-neutral-light text-sm mr-2 mt-0.5">发送到飞机群:</span>
                                            <span class="font-medium text-neutral text-sm break-all"><?php echo htmlspecialchars($robot['AccountNoList'] ?: '未设置'); ?></span>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <div class="text-center py-6 text-neutral-lighter">
                            <i class="fa fa-robot text-2xl mb-2"></i>
                            <p>暂无HG机器人数据</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- 账号操作按钮 -->
            <div class="grid grid-cols-2 gap-4">
                <a href="change_password.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" 
                   class="bg-primary hover:bg-primary/90 text-white py-3 rounded-xl transition-colors flex items-center justify-center">
                    <i class="fa fa-key mr-2"></i> 修改密码
                </a>
                <a href="my_config.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" 
                   class="bg-white border border-primary text-primary hover:bg-primary/5 py-3 rounded-xl transition-colors flex items-center justify-center">
                    <i class="fa fa-cogs mr-2"></i> 系统配置
                </a>
            </div>
        <?php else: ?>
            <!-- 无数据时显示 -->
            <div class="bg-white rounded-xl p-8 text-center card-shadow mt-8">
                <i class="fa fa-user-o text-4xl text-neutral-lighter mb-4"></i>
                <h3 class="text-neutral font-medium mb-2">无法加载账户信息</h3>
                <p class="text-sm text-neutral-lighter mb-6">请刷新页面或联系管理员</p>
                <button id="retryBtn" class="px-4 py-2 bg-primary text-white rounded-lg">
                    <i class="fa fa-refresh mr-1"></i> 重试
                </button>
            </div>
        <?php endif; ?>
    </main>

    <!-- 底部导航 -->
    <nav class="fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 py-2 px-4 z-20">
        <div class="flex justify-around">
            <a href="bet_list.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-home text-lg"></i>
                <span class="text-xs mt-1">首页</span>
            </a>
            <a href="bet_good.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-line-chart text-lg"></i>
                <span class="text-xs mt-1">数据</span>
            </a>
            <a href="bet_result.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-trophy text-lg"></i>
                <span class="text-xs mt-1">结果</span>
            </a>
            <a href="bet_notice.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-bell text-lg"></i>
                <span class="text-xs mt-1">通知</span>
            </a>
            <a href="bet_my.php?AccountNo=<?php echo isset($_GET['AccountNo']) ? urlencode($_GET['AccountNo']) : ''; ?>" class="flex flex-col items-center text-secondary">
                <i class="fa fa-user text-lg"></i>
                <span class="text-xs mt-1">我的</span>
            </a>
        </div>
    </nav>

    <script>
        // 初始化页面
        document.addEventListener('DOMContentLoaded', function() {
            // 关闭错误提示
            const closeErrorBtn = document.getElementById('closeError');
            if (closeErrorBtn) {
                closeErrorBtn.addEventListener('click', function() {
                    this.closest('div').style.display = 'none';
                });
            }

            // 刷新按钮
            document.getElementById('refreshBtn').addEventListener('click', function() {
                window.location.reload();
            });

            // 重试按钮
            const retryBtn = document.getElementById('retryBtn');
            if (retryBtn) {
                retryBtn.addEventListener('click', function() {
                    window.location.reload();
                });
            }

            // 设置按钮
            document.getElementById('settingsBtn').addEventListener('click', function() {
                alert('设置功能即将上线，敬请期待！');
            });

            // 退出登录按钮
            document.getElementById('logoutBtn').addEventListener('click', function() {
                if (confirm('确定要退出登录吗？')) {
                    // 发送退出登录请求
                    fetch('logout.php', { method: 'POST' })
                        .then(() => {
                            // 保留AccountNo参数跳转登录页
                            const accountNo = new URLSearchParams(window.location.search).get('AccountNo') || '';
                            const loginUrl = 'login.php' + (accountNo ? '?AccountNo=' + encodeURIComponent(accountNo) : '');
                            window.location.href = loginUrl;
                        })
                        .catch(err => {
                            console.error('退出登录失败:', err);
                            alert('退出登录失败，请重试');
                        });
                }
            });
        });
    </script>
</body>
</html>