<?php
// 放在页面最顶部，确保在任何输出之前执行
require_once 'session_check.php';
require_once 'db.php';
?>



<?php
// 数据库连接

try {
    $pdo = new PDO("mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die("数据库连接失败: " . $e->getMessage());
}

// 获取筛选参数 - 强制验证AccountNo
$accountNo = isset($_GET['AccountNo']) ? trim($_GET['AccountNo']) : '';
if (empty($accountNo)) {
     $loginUrl = 'login.php';
    header('Location: ' . $loginUrl);
    exit;
}

$dateFilter = isset($_GET['dateFilter']) ? $_GET['dateFilter'] : 'today';
$startDate = isset($_GET['startDate']) ? $_GET['startDate'] : '';
$endDate = isset($_GET['endDate']) ? $_GET['endDate'] : '';
$betAccount = isset($_GET['betAccount']) ? $_GET['betAccount'] : '';
$betType = isset($_GET['betType']) ? $_GET['betType'] : '';
$betHandType = isset($_GET['betHandType']) ? $_GET['betHandType'] : '';
$minAmount = isset($_GET['minAmount']) ? $_GET['minAmount'] : '';
$maxAmount = isset($_GET['maxAmount']) ? $_GET['maxAmount'] : '';
$pageSize = isset($_GET['pageSize']) ? max(1, intval($_GET['pageSize'])) : 30;
$currentPage = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$refreshInterval = isset($_GET['refreshInterval']) ? max(2, intval($_GET['refreshInterval'])) : 30;

// 初始化错误信息数组
$errors = [];

// 验证日期
if ($dateFilter === 'custom') {
    if (empty($startDate) || empty($endDate)) {
        $errors[] = '自定义日期范围时，开始日期和结束日期不能为空';
        $dateFilter = 'all';
    } elseif ($startDate > $endDate) {
        $errors[] = '开始日期不能大于结束日期';
        $dateFilter = 'all';
    }
}

// 验证金额范围
if (!empty($minAmount) && !empty($maxAmount) && $minAmount > $maxAmount) {
    $errors[] = '最小金额不能大于最大金额';
    $minAmount = '';
    $maxAmount = '';
}

// 构建日期筛选条件
$dateCondition = '';
$dateParams = [];

try {
    switch ($dateFilter) {
        case 'today':
            // 如果当前时间在12点之前，则今日范围为昨天12点到今天11:59:59
            // 如果当前时间在12点或之后，则今日范围为今天12点到明天11:59:59
            $dateCondition = "AND CreateDate BETWEEN " . 
                "CASE WHEN HOUR(NOW()) < 12 THEN DATE_SUB(CURDATE(), INTERVAL 1 DAY) ELSE CURDATE() END + INTERVAL 12 HOUR " . 
                "AND " . 
                "CASE WHEN HOUR(NOW()) < 12 THEN CURDATE() ELSE DATE_ADD(CURDATE(), INTERVAL 1 DAY) END + INTERVAL 11 HOUR + INTERVAL 59 MINUTE + INTERVAL 59 SECOND";
            break;
        case 'yesterday':
            // 如果当前时间在12点之前，则昨天范围为前天12点到昨天11:59:59
            // 如果当前时间在12点或之后，则昨天范围为昨天12点到今天11:59:59
            $dateCondition = "AND CreateDate BETWEEN " . 
                "CASE WHEN HOUR(NOW()) < 12 THEN DATE_SUB(CURDATE(), INTERVAL 2 DAY) ELSE DATE_SUB(CURDATE(), INTERVAL 1 DAY) END + INTERVAL 12 HOUR " . 
                "AND " . 
                "CASE WHEN HOUR(NOW()) < 12 THEN DATE_SUB(CURDATE(), INTERVAL 1 DAY) ELSE CURDATE() END + INTERVAL 11 HOUR + INTERVAL 59 MINUTE + INTERVAL 59 SECOND";
            break;
        case 'this_week':
            $dateCondition = "AND YEARWEEK(CreateDate, 1) = YEARWEEK(CURDATE(), 1)";
            break;
        case 'this_month':
            $dateCondition = "AND MONTH(CreateDate) = MONTH(CURDATE()) AND YEAR(CreateDate) = YEAR(CURDATE())";
            break;
        case 'last_month':
            $dateCondition = "AND MONTH(CreateDate) = MONTH(CURDATE() - INTERVAL 1 MONTH) AND YEAR(CreateDate) = YEAR(CURDATE() - INTERVAL 1 MONTH)";
            break;
        case 'custom':
            if (!empty($startDate) && !empty($endDate)) {
                $dateCondition = "AND CreateDate BETWEEN :startDate AND :endDate";
                $dateParams[':startDate'] = $startDate . ' 00:00:00';
                $dateParams[':endDate'] = $endDate . ' 23:59:59';
            }
            break;
    }
} catch (Exception $e) {
    $errors[] = '日期筛选条件构建失败: ' . $e->getMessage();
    $dateCondition = '';
    $dateParams = [];
}

// 构建其他筛选条件
$otherConditions = [];
$otherParams = [];

// 强制加入AccountNo条件，作为基础查询条件
$otherConditions[] = "AccountNo = :accountNo";
$otherParams[':accountNo'] = $accountNo;

try {
    if (!empty($betAccount)) {
        $otherConditions[] = "BetAccount LIKE :betAccount";
        $otherParams[':betAccount'] = "%$betAccount%";
    }

    if (!empty($betType)) {
        $otherConditions[] = "BetType LIKE :betType";
        $otherParams[':betType'] = "%$betType%";
    }

    if (!empty($betHandType)) {
        $otherConditions[] = "BetHandType LIKE :betHandType";
        $otherParams[':betHandType'] = "%$betHandType%";
    }

    if (!empty($minAmount)) {
        $otherConditions[] = "BetAmount >= :minAmount";
        $otherParams[':minAmount'] = $minAmount;
    }

    if (!empty($maxAmount)) {
        $otherConditions[] = "BetAmount <= :maxAmount";
        $otherParams[':maxAmount'] = $maxAmount;
    }
} catch (Exception $e) {
    $errors[] = '筛选条件构建失败: ' . $e->getMessage();
    $otherConditions = ["AccountNo = :accountNo"]; // 确保AccountNo条件始终存在
    $otherParams[':accountNo'] = $accountNo;
}

$otherCondition = '';
if (!empty($otherConditions)) {
    $otherCondition = "AND " . implode(" AND ", $otherConditions);
}

// 计算偏移量
$offset = ($currentPage - 1) * $pageSize;

// 查询总记录数（用于分页）
$bets = [];
$stats = ['total_bets' => 0, 'total_amount' => 0];
$totalRecords = 0;
$totalPages = 1;

try {
    $countSql = "SELECT COUNT(*) as total FROM TelSms_BetList WHERE 1=1 $dateCondition $otherCondition";
    $countStmt = $pdo->prepare($countSql);
    $countStmt->execute(array_merge($dateParams, $otherParams));
    $totalRecords = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
    $totalPages = ceil($totalRecords / $pageSize);

    // 确保当前页在有效范围内
    if ($currentPage > $totalPages && $totalPages > 0) {
        $currentPage = $totalPages;
        $offset = ($currentPage - 1) * $pageSize;
    }

   // 查询数据
$sql = "SELECT * FROM TelSms_BetList WHERE 1=1 $dateCondition $otherCondition";
$sql .= " ORDER BY CreateDate DESC LIMIT :offset, :pageSize";

$stmt = $pdo->prepare($sql);

// 绑定所有参数
$allParams = array_merge($dateParams, $otherParams);
$allParams[':offset'] = $offset;
$allParams[':pageSize'] = $pageSize;

foreach ($allParams as $key => $value) {
    $type = is_int($value) ? PDO::PARAM_INT : PDO::PARAM_STR;
    $stmt->bindValue($key, $value, $type);
}

$stmt->execute();
$bets = $stmt->fetchAll(PDO::FETCH_ASSOC);
    // 计算统计数据
    $statsSql = "SELECT COUNT(*) as total_bets, SUM(BetAmount) as total_amount FROM TelSms_BetList WHERE 1=1 $dateCondition $otherCondition";
    $statsStmt = $pdo->prepare($statsSql);
    $statsStmt->execute(array_merge($dateParams, $otherParams));
    $stats = $statsStmt->fetch(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $errors[] = '数据库查询失败: ' . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>订单跟踪</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdn.jsdelivr.net/npm/font-awesome@4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#165DFF',
                        secondary: '#FF4D4F',
                        neutral: '#1D2129',
                        'neutral-light': '#4E5969',
                        'neutral-lighter': '#86909C',
                        'neutral-lightest': '#F2F3F5'
                    },
                    fontFamily: {
                        sans: ['Inter', 'system-ui', 'sans-serif'],
                    },
                }
            }
        }
    </script>
    <style type="text/tailwindcss">
        @layer utilities {
            .content-auto {
                content-visibility: auto;
            }
            .scrollbar-hide {
                -ms-overflow-style: none;
                scrollbar-width: none;
            }
            .scrollbar-hide::-webkit-scrollbar {
                display: none;
            }
            .card-shadow {
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            }
            .bg-gradient-blue {
                background: linear-gradient(135deg, #165DFF 0%, #4080FF 100%);
            }
            .bg-gradient-red {
                background: linear-gradient(135deg, #FF4D4F 0%, #FF7A45 100%);
            }
        }
    </style>
</head>
<body class="bg-neutral-lightest min-h-screen font-sans">
    <!-- 顶部导航栏 -->
    <header class="bg-white shadow-md fixed top-0 left-0 right-0 z-10 transition-all duration-300">
        <div class="container mx-auto px-4 py-3 flex items-center justify-between">
            <div class="flex items-center space-x-2">
                <i class="fa fa-bolt text-primary text-xl"></i>
                <h1 class="text-lg font-bold text-neutral">比赛跟踪</h1>
                <h1 class="text-lg font-bold text-neutral" style="display: none;">账号：<?php  echo htmlspecialchars($accountNo); ?></h1>
            </div>
            <div class="flex items-center space-x-4">
                <button id="refreshBtn" class="text-primary hover:text-primary/80 transition-colors">
                    <i class="fa fa-refresh"></i>
                </button>
                <button id="settingsBtn" class="text-neutral-light hover:text-neutral transition-colors">
                    <i class="fa fa-cog"></i>
                </button>
            </div>
        </div>
    </header>

    <!-- 错误提示 -->
    <?php if (!empty($errors)): ?>
        <div class="fixed top-16 left-0 right-0 z-50 bg-secondary text-white py-2 px-4">
            <div class="container mx-auto">
                <div class="flex items-center">
                    <i class="fa fa-exclamation-circle mr-2"></i>
                    <div>
                        <?php foreach ($errors as $error): ?>
                            <div><?php echo htmlspecialchars($error); ?></div>
                        <?php endforeach; ?>
                    </div>
                    <button id="closeError" class="ml-auto text-white hover:text-white/80">
                        <i class="fa fa-times"></i>
                    </button>
                </div>
            </div>
        </div>
        <script>
            document.getElementById('closeError').addEventListener('click', function() {
                this.closest('div').style.display = 'none';
            });
        </script>
    <?php endif; ?>

    <!-- 设置面板 -->
    <div id="settingsPanel" class="fixed inset-0 bg-black/50 z-50 hidden flex items-center justify-center">
        <div class="bg-white rounded-lg w-full max-w-md mx-4 overflow-hidden card-shadow transform transition-all duration-300 scale-95 opacity-0" id="settingsModal">
            <div class="bg-primary text-white px-6 py-4 flex justify-between items-center">
                <h3 class="font-bold">设置</h3>
                <button id="closeSettings" class="text-white hover:text-white/80">
                    <i class="fa fa-times"></i>
                </button>
            </div>
            <div class="p-6 space-y-4">
                <div>
                    <label class="block text-sm font-medium text-neutral-light mb-1">刷新间隔 (秒)</label>
                    <div class="flex items-center space-x-2">
                        <input type="number" id="refreshIntervalInput" min="2" value="<?php echo $refreshInterval; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary">
                        <span class="text-sm text-neutral-lighter">最小值: 2秒</span>
                    </div>
                </div>
                <div>
                    <label class="block text-sm font-medium text-neutral-light mb-1">每页显示数量</label>
                    <input type="number" id="pageSizeInput" min="1" value="<?php echo $pageSize; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary">
                </div>
                <div class="pt-4">
                    <button id="saveSettings" class="w-full bg-primary hover:bg-primary/90 text-white font-medium py-2 px-4 rounded-md transition-colors">
                        保存设置
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- 筛选面板 -->
    <div id="filterPanel" class="fixed bottom-0 left-0 right-0 bg-white rounded-t-xl p-4 card-shadow transform translate-y-full transition-transform duration-300 z-30">
        <div class="flex justify-between items-center mb-4">
            <h3 class="font-bold text-neutral">筛选条件</h3>
            <button id="closeFilter" class="text-neutral-light hover:text-neutral transition-colors">
                <i class="fa fa-times"></i>
            </button>
        </div>
        
        <form id="filterForm" class="space-y-4">
            <div>
                <label class="block text-sm font-medium text-neutral-light mb-1">日期筛选</label>
                <div class="grid grid-cols-5 gap-2 mb-2">
                    <button type="button" class="date-filter-btn py-1 px-2 text-xs rounded-full <?php echo $dateFilter == 'today' ? 'bg-primary text-white' : 'bg-neutral-lightest text-neutral-light'; ?>" data-filter="today">今天</button>
                    <button type="button" class="date-filter-btn py-1 px-2 text-xs rounded-full <?php echo $dateFilter == 'yesterday' ? 'bg-primary text-white' : 'bg-neutral-lightest text-neutral-light'; ?>" data-filter="yesterday">昨天</button>
                    <button type="button" class="date-filter-btn py-1 px-2 text-xs rounded-full <?php echo $dateFilter == 'this_week' ? 'bg-primary text-white' : 'bg-neutral-lightest text-neutral-light'; ?>" data-filter="this_week">本周</button>
                    <button type="button" class="date-filter-btn py-1 px-2 text-xs rounded-full <?php echo $dateFilter == 'this_month' ? 'bg-primary text-white' : 'bg-neutral-lightest text-neutral-light'; ?>" data-filter="this_month">本月</button>
                    <button type="button" class="date-filter-btn py-1 px-2 text-xs rounded-full <?php echo $dateFilter == 'last_month' ? 'bg-primary text-white' : 'bg-neutral-lightest text-neutral-light'; ?>" data-filter="last_month">上月</button>
                </div>
                <div id="customDateRange" class="grid grid-cols-2 gap-2 <?php echo $dateFilter == 'custom' ? '' : 'hidden'; ?>">
                    <div>
                        <input type="date" id="startDate" value="<?php echo $startDate; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary text-sm">
                    </div>
                    <div>
                        <input type="date" id="endDate" value="<?php echo $endDate; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary text-sm">
                    </div>
                </div>
                <input type="hidden" id="dateFilterInput" value="<?php echo $dateFilter; ?>">
            </div>
            
            <div>
                <label class="block text-sm font-medium text-neutral-light mb-1">投注会员</label>
                <input type="text" id="betAccountInput" value="<?php echo $betAccount; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary">
            </div>
            
            <div>
                <label class="block text-sm font-medium text-neutral-light mb-1">比赛类型</label>
                <select id="betTypeInput" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary">
                    <option value="">全部</option>
                    <option value="足球" <?php echo $betType == '足球' ? 'selected' : ''; ?>>足球</option>
                    <option value="篮球" <?php echo $betType == '篮球' ? 'selected' : ''; ?>>篮球</option>
                    <option value="其他" <?php echo $betType == '其他' ? 'selected' : ''; ?>>其他</option>
                </select>
            </div>
            
            <div>
                <label class="block text-sm font-medium text-neutral-light mb-1">盘口类型</label>
                <select id="betHandTypeInput" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary">
                    <option value="">全部</option>
                    <option value="滚球" <?php echo $betHandType == '滚球' ? 'selected' : ''; ?>>滚球</option>
                    <option value="今日" <?php echo $betHandType == '今日' ? 'selected' : ''; ?>>今日</option>
                    <option value="早盘" <?php echo $betHandType == '早盘' ? 'selected' : ''; ?>>早盘</option>
                </select>
            </div>
            
            <div>
                <label class="block text-sm font-medium text-neutral-light mb-1">投注金额范围</label>
                <div class="grid grid-cols-2 gap-2">
                    <div>
                        <input type="number" id="minAmountInput" placeholder="最小金额" value="<?php echo $minAmount; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary text-sm">
                    </div>
                    <div>
                        <input type="number" id="maxAmountInput" placeholder="最大金额" value="<?php echo $maxAmount; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary text-sm">
                    </div>
                </div>
            </div>
            
            <div class="flex space-x-2">
                <button type="button" id="applyFilter" class="flex-1 bg-primary hover:bg-primary/90 text-white font-medium py-2 px-4 rounded-md transition-colors">
                    应用筛选
                </button>
                <button type="button" id="resetFilter" class="flex-1 bg-neutral-lightest hover:bg-gray-200 text-neutral font-medium py-2 px-4 rounded-md transition-colors">
                    重置
                </button>
            </div>
        </form>
    </div>

    <!-- 主内容区 -->
    <main class="container mx-auto px-4 pt-20 pb-32">
        <!-- 统计卡片 -->
        <div class="grid grid-cols-2 gap-4 mb-6">
            <div class="bg-gradient-blue rounded-xl p-4 text-white shadow-lg transform transition-all duration-300 hover:scale-[1.02]">
                <div class="flex justify-between items-start mb-2">
                    <h3 class="text-sm font-medium">总订单数量</h3>
                    <i class="fa fa-trophy text-white/70"></i>
                </div>
                <div class="text-2xl font-bold"><?php echo $stats['total_bets'] ?? 0; ?></div>
                <div class="text-xs text-white/80 mt-1">场</div>
            </div>
            <div class="bg-gradient-red rounded-xl p-4 text-white shadow-lg transform transition-all duration-300 hover:scale-[1.02]">
                <div class="flex justify-between items-start mb-2">
                    <h3 class="text-sm font-medium">总订单金额</h3>
                    <i class="fa fa-money text-white/70"></i>
                </div>
                <div class="text-2xl font-bold"><?php echo number_format($stats['total_amount'] ?? 0, 2); ?></div>
                <div class="text-xs text-white/80 mt-1">元</div>
            </div>
        </div>

        <!-- 筛选按钮 -->
       <!-- 筛选按钮及已选条件区域 -->
<div class="flex flex-wrap justify-between items-center mb-4 gap-2">
    <div class="text-sm text-neutral-lighter">
        共 <span class="text-primary font-medium"><?php echo $totalRecords; ?></span> 条记录
    </div>
    
    <!-- 已选条件标签容器 -->
    <div id="activeFilters" class="flex flex-wrap gap-2">
        <?php
        // 日期条件标签
        if ($dateFilter != 'all') {
            $dateText = '';
            switch ($dateFilter) {
                case 'today': $dateText = '今天'; break;
                case 'yesterday': $dateText = '昨天'; break;
                case 'this_week': $dateText = '本周'; break;
                case 'this_month': $dateText = '本月'; break;
                case 'last_month': $dateText = '上月'; break;
                case 'custom': $dateText = "{$startDate}至{$endDate}"; break;
            }
            echo "<span class='inline-flex items-center bg-primary/10 text-primary text-xs px-2 py-1 rounded-full'>
                    日期: {$dateText}
                    <button class='ml-1 text-primary/70 hover:text-primary' data-filter='dateFilter'>×</button>
                  </span>";
        }
        
        // 投注会员条件标签
        if (!empty($betAccount)) {
            echo "<span class='inline-flex items-center bg-primary/10 text-primary text-xs px-2 py-1 rounded-full'>
                    会员: {$betAccount}
                    <button class='ml-1 text-primary/70 hover:text-primary' data-filter='betAccount'>×</button>
                  </span>";
        }
        
        // 比赛类型条件标签
        if (!empty($betType)) {
            echo "<span class='inline-flex items-center bg-primary/10 text-primary text-xs px-2 py-1 rounded-full'>
                    类型: {$betType}
                    <button class='ml-1 text-primary/70 hover:text-primary' data-filter='betType'>×</button>
                  </span>";
        }
        
        // 盘口类型条件标签
        if (!empty($betHandType)) {
            echo "<span class='inline-flex items-center bg-primary/10 text-primary text-xs px-2 py-1 rounded-full'>
                    盘口: {$betHandType}
                    <button class='ml-1 text-primary/70 hover:text-primary' data-filter='betHandType'>×</button>
                  </span>";
        }
        
        // 金额范围条件标签
        if (!empty($minAmount) || !empty($maxAmount)) {
            $amountText = '';
            if (!empty($minAmount) && !empty($maxAmount)) {
                $amountText = "{$minAmount}-{$maxAmount}元";
            } elseif (!empty($minAmount)) {
                $amountText = "≥{$minAmount}元";
            } else {
                $amountText = "≤{$maxAmount}元";
            }
            echo "<span class='inline-flex items-center bg-primary/10 text-primary text-xs px-2 py-1 rounded-full'>
                    金额: {$amountText}
                    <button class='ml-1 text-primary/70 hover:text-primary' data-filter='amount'>×</button>
                  </span>";
        }
        ?>
    </div>
    
    <button id="filterBtn" class="bg-primary hover:bg-primary/90 text-white px-4 py-2 rounded-full text-sm font-medium transition-colors flex items-center">
        <i class="fa fa-filter mr-1"></i> 筛选
    </button>
</div>

        <!-- 投注列表 -->
        <div id="betList" class="space-y-4">
            <?php if (empty($bets)): ?>
                <div class="bg-white rounded-xl p-6 text-center">
                    <i class="fa fa-search text-4xl text-neutral-lighter mb-3"></i>
                    <h3 class="text-neutral font-medium mb-2">没有找到匹配的比赛数据</h3>
                    <p class="text-sm text-neutral-lighter">请尝试调整筛选条件或扩大时间范围</p>
                </div>
            <?php else: ?>
                <?php foreach ($bets as $bet): ?>
                    <?php
                    // 解析BetContent字段
                    $betContent = $bet['BetContent'];
                    $lines = explode("\n", $betContent);
                    $betData = [];
                    
                    foreach ($lines as $line) {
                        if (strpos($line, ':') !== false) {
                            list($key, $value) = explode(': ', $line, 2);
                            $betData[$key] = $value;
                        }
                    }
                    
                    // 获取投注金额
                    preg_match('/<code>投注金额: (.*?)<\/code>/', $betContent, $amountMatches);
                    $betAmount = isset($amountMatches[1]) ? $amountMatches[1] : '';
                    
                 
                // ... existing code ...
                 // 提取比赛比分及附加信息
                 $score = '';
                 if (isset($betData['球队对阵'])) {
                     $teams = preg_replace('/\s*\(.*?\)\s*/', '', $betData['球队对阵']);
                     // 去掉 $teams 值，保留后面的字符串
                     $score = str_replace($teams, '', $betData['球队对阵']);
                     $score = trim($score);
                 }
// ... existing code ...

                    
                    // 确定卡片颜色
                    $cardColor = 'bg-white';
                    if (isset($betData['盘口类型']) && strpos($betData['盘口类型'], '滚球') !== false) {
                        $cardColor = 'bg-primary/5';
                    }
                    ?>
                    
                    <div class="bet-card <?php echo $cardColor; ?> rounded-xl overflow-hidden shadow-md transform transition-all duration-300 hover:shadow-lg">
                        <div class="p-4">
                            <div class="flex justify-between items-start mb-2">
                                <div class="text-xs font-medium text-neutral-lighter">
                                    <?php echo isset($bet['BetAccount']) ? htmlspecialchars($bet['BetAccount']) : ''; ?>
                                </div>
                                <div class="text-xs font-medium text-primary">
                                    <?php echo isset($bet['SendTime']) ? htmlspecialchars($bet['SendTime']) : htmlspecialchars($bet['BetTime']); ?>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <div class="text-sm font-medium text-neutral">
                                    <?php echo isset($bet['BetType']) ? htmlspecialchars($bet['BetType']) : ''; ?> - <?php echo isset($bet['BetName']) ? htmlspecialchars($bet['BetName']) : ''; ?>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <div class="flex justify-between items-center">
                                    <div class="text-lg font-bold text-neutral">
                                        <?php 
                                        if (isset($bet['BetTeamV'])) {
                                           // $teams = preg_replace('/\s*\(.*?\)\s*/', '', $betData['球队对阵']);
                                            echo htmlspecialchars($bet['BetTeamV']);
                                        }
                                        ?>
                                    </div>
                                    <div class="px-2 py-1 bg-secondary/10 text-secondary rounded text-xs font-medium">
                                        <?php echo htmlspecialchars($bet['BetTeamNumberV']); ?> - <?php echo htmlspecialchars($bet['BetNo']); ?>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="grid grid-cols-2 gap-2 mb-3">
                                <div class="bg-neutral-lightest rounded p-2 text-xs">
                                    <div class="text-neutral-lighter">盘口类型</div>
                                    <div class="font-medium text-neutral"><?php echo isset($bet['BetHandType']) ? htmlspecialchars($bet['BetHandType']) : ''; ?></div>
                                </div>
                                <div class="bg-neutral-lightest rounded p-2 text-xs">
                                    <div class="text-neutral-lighter">投注项</div>
                                    <div class="font-medium text-neutral"><?php echo isset($bet['BetItem']) ? htmlspecialchars($bet['BetItem']) : ''; ?></div>
                                </div>
                            </div>
                            
                            <div class="flex justify-between items-center pt-3 border-t border-gray-100">
                                <div class="text-sm font-medium text-neutral">
                                    投注金额: <span class="text-secondary font-bold text-lg" ><?php echo htmlspecialchars($bet['BetAmount']); ?></span>
                                </div>
                                <div class="text-sm font-medium text-neutral">
                                    赔率: <span class="text-primary  text-lg"><?php echo isset($bet['BetOdds']) ? htmlspecialchars($bet['BetOdds']) : ''; ?></span>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <!-- 分页 -->
        <div class="flex justify-between items-center mt-6">
            <button id="prevPage" class="px-4 py-2 rounded-md bg-white hover:bg-gray-50 text-neutral-light border border-gray-200 disabled:opacity-50 disabled:pointer-events-none transition-colors" <?php echo $currentPage == 1 ? 'disabled' : ''; ?>>
                <i class="fa fa-angle-left mr-1"></i> 上一页
            </button>
            
            <div class="flex items-center space-x-1">
                <?php
                $visiblePages = 5;
                $startPage = max(1, $currentPage - floor($visiblePages / 2));
                $endPage = min($startPage + $visiblePages - 1, $totalPages);
                
                if ($startPage > 1) {
                    echo '<span class="px-2 py-1 text-sm text-neutral-light">...</span>';
                }
                
                for ($i = $startPage; $i <= $endPage; $i++) {
                    if ($i == $currentPage) {
                        echo '<span class="px-3 py-1 bg-primary text-white rounded-md text-sm font-medium">' . $i . '</span>';
                    } else {
                        echo '<button class="page-btn px-3 py-1 hover:bg-gray-100 rounded-md text-sm text-neutral-light transition-colors">' . $i . '</button>';
                    }
                }
                
                if ($endPage < $totalPages) {
                    echo '<span class="px-2 py-1 text-sm text-neutral-light">...</span>';
                }
                ?>
            </div>
            
            <button id="nextPage" class="px-4 py-2 rounded-md bg-white hover:bg-gray-50 text-neutral-light border border-gray-200 disabled:opacity-50 disabled:pointer-events-none transition-colors" <?php echo $currentPage >= $totalPages ? 'disabled' : ''; ?>>
                下一页 <i class="fa fa-angle-right ml-1"></i>
            </button>
        </div>
    </main>

   
    <script>
        // 当前账号参数
        const currentAccount = '<?php echo htmlspecialchars($accountNo); ?>';

        // 初始化页面
        document.addEventListener('DOMContentLoaded', function() {
            // 刷新按钮
            document.getElementById('refreshBtn').addEventListener('click', function() {
                try {
                    window.location.reload();
                } catch (error) {
                    console.error('刷新出错:', error);
                    alert('刷新时发生错误: ' + error.message);
                }
            });

           

            // 获取当前页面文件名
            const currentPage = window.location.pathname.split('/').pop();
            const navLinks = document.querySelectorAll('nav a');

            navLinks.forEach(link => {
                const linkHref = link.getAttribute('href').split('?')[0];
                if (linkHref === currentPage) {
                    link.classList.add('text-secondary');
                }
            });
        });
    </script>

    <!-- 底部导航 -->
     <nav class="fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 py-2 px-4 z-20">
        <div class="flex justify-around">
            <a href="bet_list.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-home text-lg"></i>
                <span class="text-xs mt-1">首页</span>
            </a>
            <a href="bet_good.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-line-chart text-lg"></i>
                <span class="text-xs mt-1">数据</span>
            </a>
            <a href="bet_result.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-trophy text-lg"></i>
                <span class="text-xs mt-1">结果</span>
            </a>
            <a href="bet_notice.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-bell text-lg"></i>
                <span class="text-xs mt-1">通知</span>
            </a>
            <a href="bet_my.php?AccountNo=<?php echo urlencode($accountNo); ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-user text-lg"></i>
                <span class="text-xs mt-1">我的</span>
            </a>
        </div>
    </nav>

    <script>
        // 当前筛选参数
        const currentParams = {
            AccountNo: '<?php echo $accountNo; ?>',
            dateFilter: '<?php echo $dateFilter; ?>',
            startDate: '<?php echo $startDate; ?>',
            endDate: '<?php echo $endDate; ?>',
            betAccount: '<?php echo $betAccount; ?>',
            betType: '<?php echo $betType; ?>',
            betHandType: '<?php echo $betHandType; ?>',
            minAmount: '<?php echo $minAmount; ?>',
            maxAmount: '<?php echo $maxAmount; ?>',
            pageSize: <?php echo $pageSize; ?>,
            page: <?php echo $currentPage; ?>,
            refreshInterval: <?php echo $refreshInterval; ?>
        };

        // 当前筛选参数
        let refreshTimer = null;
        let isSettingInProgress = false; // 标记是否正在进行设置操作

        // 初始化页面
        document.addEventListener('DOMContentLoaded', function() {
            // 设置面板动画
            const settingsPanel = document.getElementById('settingsPanel');
            const settingsModal = document.getElementById('settingsModal');

            // 条件标签删除功能
        document.querySelectorAll('#activeFilters button').forEach(button => {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                const filterType = this.getAttribute('data-filter');
                
                // 清除对应筛选条件
                switch(filterType) {
                    case 'dateFilter':
                        currentParams.dateFilter = 'all';
                        currentParams.startDate = '';
                        currentParams.endDate = '';
                        break;
                    case 'betAccount':
                        currentParams.betAccount = '';
                        break;
                    case 'betType':
                        currentParams.betType = '';
                        break;
                    case 'betHandType':
                        currentParams.betHandType = '';
                        break;
                    case 'amount':
                        currentParams.minAmount = '';
                        currentParams.maxAmount = '';
                        break;
                }
                
                currentParams.page = 1; // 重置到第一页
                updateUrlAndReload();
            });
        });
            
            document.getElementById('settingsBtn').addEventListener('click', function() {
                // 打开设置面板时暂停自动刷新
                pauseAutoRefresh();
                isSettingInProgress = true;
                
                settingsPanel.classList.remove('hidden');
                setTimeout(() => {
                    settingsModal.classList.remove('scale-95', 'opacity-0');
                    settingsModal.classList.add('scale-100', 'opacity-100');
                }, 10);
            });
            
            document.getElementById('closeSettings').addEventListener('click', closeSettingsPanel);
            
            function closeSettingsPanel() {
                settingsModal.classList.remove('scale-100', 'opacity-100');
                settingsModal.classList.add('scale-95', 'opacity-0');
                setTimeout(() => {
                    settingsPanel.classList.add('hidden');
                    // 关闭设置面板后恢复自动刷新
                    isSettingInProgress = false;
                    resumeAutoRefresh();
                }, 300);
            }
            
            // 筛选面板动画
            const filterPanel = document.getElementById('filterPanel');
            
            document.getElementById('filterBtn').addEventListener('click', function() {
                // 打开筛选面板时暂停自动刷新
                pauseAutoRefresh();
                isSettingInProgress = true;
                
                filterPanel.classList.remove('translate-y-full');
            });
            
            document.getElementById('closeFilter').addEventListener('click', function() {
                filterPanel.classList.add('translate-y-full');
                // 关闭筛选面板后恢复自动刷新
                setTimeout(() => {
                    isSettingInProgress = false;
                    resumeAutoRefresh();
                }, 300);
            });
            
            // 日期筛选按钮
            document.querySelectorAll('.date-filter-btn').forEach(btn => {
                btn.addEventListener('click', function() {
                    document.querySelectorAll('.date-filter-btn').forEach(b => {
                        b.classList.remove('bg-primary', 'text-white');
                        b.classList.add('bg-neutral-lightest', 'text-neutral-light');
                    });
                    
                    this.classList.remove('bg-neutral-lightest', 'text-neutral-light');
                    this.classList.add('bg-primary', 'text-white');
                    
                    const filter = this.getAttribute('data-filter');
                    document.getElementById('dateFilterInput').value = filter;
                    
                    if (filter === 'custom') {
                        document.getElementById('customDateRange').classList.remove('hidden');
                    } else {
                        document.getElementById('customDateRange').classList.add('hidden');
                    }
                });
            });
            
            // 自定义日期范围
            document.getElementById('dateFilterInput').addEventListener('change', function() {
                if (this.value === 'custom') {
                    document.getElementById('customDateRange').classList.remove('hidden');
                } else {
                    document.getElementById('customDateRange').classList.add('hidden');
                }
            });
            
            // 应用筛选
            document.getElementById('applyFilter').addEventListener('click', function() {
                try {
                    currentParams.dateFilter = document.getElementById('dateFilterInput').value;
                    currentParams.startDate = document.getElementById('startDate').value;
                    currentParams.endDate = document.getElementById('endDate').value;
                    currentParams.betAccount = document.getElementById('betAccountInput').value;
                    currentParams.betType = document.getElementById('betTypeInput').value;
                    currentParams.betHandType = document.getElementById('betHandTypeInput').value;
                    currentParams.minAmount = document.getElementById('minAmountInput').value;
                    currentParams.maxAmount = document.getElementById('maxAmountInput').value;
                    currentParams.page = 1; // 重置到第一页
                    
                    // 验证日期
                    if (currentParams.dateFilter === 'custom') {
                        if (!currentParams.startDate || !currentParams.endDate) {
                            alert('自定义日期范围时，开始日期和结束日期不能为空');
                            return;
                        }
                        
                        if (new Date(currentParams.startDate) > new Date(currentParams.endDate)) {
                            alert('开始日期不能大于结束日期');
                            return;
                        }
                    }
                    
                    // 验证金额范围
                    if (currentParams.minAmount && currentParams.maxAmount && 
                        parseFloat(currentParams.minAmount) > parseFloat(currentParams.maxAmount)) {
                        alert('最小金额不能大于最大金额');
                        return;
                    }
                    
                    updateUrlAndReload();
                    filterPanel.classList.add('translate-y-full');
                    
                    // 应用筛选后恢复自动刷新
                    setTimeout(() => {
                        isSettingInProgress = false;
                        resumeAutoRefresh();
                    }, 300);
                } catch (error) {
                    console.error('应用筛选出错:', error);
                    alert('应用筛选时发生错误: ' + error.message);
                }
            });
            
            // 重置筛选
            document.getElementById('resetFilter').addEventListener('click', function() {
                try {
                    document.getElementById('dateFilterInput').value = 'all';
                    document.getElementById('startDate').value = '';
                    document.getElementById('endDate').value = '';
                    document.getElementById('betAccountInput').value = '';
                    document.getElementById('betTypeInput').value = '';
                    document.getElementById('betHandTypeInput').value = '';
                    document.getElementById('minAmountInput').value = '';
                    document.getElementById('maxAmountInput').value = '';
                    
                    document.querySelectorAll('.date-filter-btn').forEach(b => {
                        b.classList.remove('bg-primary', 'text-white');
                        b.classList.add('bg-neutral-lightest', 'text-neutral-light');
                    });
                    
                    currentParams.dateFilter = 'all';
                    currentParams.startDate = '';
                    currentParams.endDate = '';
                    currentParams.betAccount = '';
                    currentParams.betType = '';
                    currentParams.betHandType = '';
                    currentParams.minAmount = '';
                    currentParams.maxAmount = '';
                    currentParams.page = 1; // 重置到第一页
                    
                    updateUrlAndReload();
                    filterPanel.classList.add('translate-y-full');
                    
                    // 重置筛选后恢复自动刷新
                    setTimeout(() => {
                        isSettingInProgress = false;
                        resumeAutoRefresh();
                    }, 300);
                } catch (error) {
                    console.error('重置筛选出错:', error);
                    alert('重置筛选时发生错误: ' + error.message);
                }
            });
            
            // 保存设置
            document.getElementById('saveSettings').addEventListener('click', function() {
                try {
                    const refreshInterval = Math.max(2, parseInt(document.getElementById('refreshIntervalInput').value));
                    const pageSize = parseInt(document.getElementById('pageSizeInput').value);
                    
                    if (isNaN(refreshInterval) || refreshInterval < 2) {
                        alert('刷新间隔必须至少为2秒');
                        return;
                    }
                    
                    if (isNaN(pageSize) || pageSize < 1) {
                        alert('每页显示数量必须至少为1条');
                        return;
                    }
                    
                    currentParams.refreshInterval = refreshInterval;
                    currentParams.pageSize = pageSize;
                    currentParams.page = 1; // 重置到第一页
                    
                    updateUrlAndReload();
                    closeSettingsPanel();
                    
                    // 保存设置后恢复自动刷新
                    isSettingInProgress = false;
                    resumeAutoRefresh();
                } catch (error) {
                    console.error('保存设置出错:', error);
                    alert('保存设置时发生错误: ' + error.message);
                }
            });
            
            // 分页按钮
            document.getElementById('prevPage').addEventListener('click', function() {
                if (currentParams.page > 1) {
                    currentParams.page--;
                    updateUrlAndReload();
                }
            });
            
            document.getElementById('nextPage').addEventListener('click', function() {
                const totalPages = <?php echo $totalPages; ?>;
                if (currentParams.page < totalPages) {
                    currentParams.page++;
                    updateUrlAndReload();
                }
            });
            
            // 页码按钮
            document.querySelectorAll('.page-btn').forEach(btn => {
                btn.addEventListener('click', function() {
                    const page = parseInt(this.textContent);
                    currentParams.page = page;
                    updateUrlAndReload();
                });
            });
            
            // 手动刷新按钮
            document.getElementById('refreshBtn').addEventListener('click', function() {
                try {
                    refreshData();
                } catch (error) {
                    console.error('手动刷新出错:', error);
                    alert('刷新数据时发生错误: ' + error.message);
                }
            });
            
            // 开始自动刷新
            startAutoRefresh();
        });
        
        // 更新URL并重新加载数据
        function updateUrlAndReload() {
            try {
                const queryParams = new URLSearchParams();
                for (const key in currentParams) {
                    if (currentParams[key] !== '' && currentParams[key] !== null && currentParams[key] !== undefined) {
                        queryParams.set(key, currentParams[key]);
                    }
                }
                
                const newUrl = window.location.pathname + '?' + queryParams.toString();
                window.history.replaceState({}, '', newUrl);
                
                refreshData();
            } catch (error) {
                console.error('更新URL出错:', error);
                alert('更新页面时发生错误: ' + error.message);
            }
        }
        
        // 开始自动刷新
        function startAutoRefresh() {
            // 如果正在进行设置操作，则不启动自动刷新
            if (isSettingInProgress) {
                return;
            }
            
            if (refreshTimer) {
                clearTimeout(refreshTimer);
            }
            
            refreshTimer = setTimeout(refreshData, currentParams.refreshInterval * 1000);
        }
        
        // 暂停自动刷新
        function pauseAutoRefresh() {
            if (refreshTimer) {
                clearTimeout(refreshTimer);
                refreshTimer = null;
            }
        }
        
        // 恢复自动刷新
        function resumeAutoRefresh() {
            // 如果不在设置操作中，则恢复自动刷新
            if (!isSettingInProgress) {
                startAutoRefresh();
            }
        }
        
        // 刷新数据
        function refreshData() {
            try {
                // 显示加载动画
                const betList = document.getElementById('betList');
                const originalContent = betList.innerHTML;
                betList.innerHTML = '<div class="flex justify-center items-center py-10"><div class="animate-spin rounded-full h-10 w-10 border-b-2 border-primary"></div></div>';
                
                // 模拟AJAX请求
                setTimeout(() => {
                    // 这里实际项目中应该使用fetch或XMLHttpRequest来获取数据
                    window.location.reload();
                    
                    // 继续自动刷新
                    startAutoRefresh();
                }, 500);
            } catch (error) {
                console.error('刷新数据出错:', error);
                alert('刷新数据时发生错误: ' + error.message);
                
                // 恢复原始内容
                const betList = document.getElementById('betList');
                betList.innerHTML = '<div class="bg-red-50 border-l-4 border-red-400 p-4 mb-4">刷新数据失败，请稍后重试</div>';
                
                // 继续自动刷新
                startAutoRefresh();
            }
        }
    </script>
</body>
</html>