<?php
session_start();

// 获取AccountNo参数
$accountNo = isset($_GET['AccountNo']) ? trim($_GET['AccountNo']) : '';
$userName = '';

// 如果有AccountNo参数，查询对应的UserName
if (!empty($accountNo)) {
    // 数据库连接
    require_once 'db.php';
    
    try {
        $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // 查询telsms_account表获取UserName
        $stmt = $pdo->prepare("SELECT UserName FROM telsms_account WHERE AccountNo = :accountNo");
        $stmt->bindParam(':accountNo', $accountNo);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result) {
            $userName = $result['UserName'];
        }
    } catch (PDOException $e) {
        // 数据库错误处理
        error_log("Database error: " . $e->getMessage());
    }
}

// 如果用户已登录，直接跳转到个人主页并携带参数
if (isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true) {
    $redirectUrl = 'bet_my.php';
    if (!empty($accountNo)) {
        $redirectUrl .= '?AccountNo=' . urlencode($accountNo);
    }
    header('Location: ' . $redirectUrl);
    exit;
}
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>系统登录</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdn.jsdelivr.net/npm/font-awesome@4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#165DFF',
                        secondary: '#FF4D4F',
                        success: '#52C41A',
                        neutral: '#1D2129',
                        'neutral-light': '#4E5969',
                        'neutral-lighter': '#86909C',
                        'neutral-lightest': '#F2F3F5'
                    },
                    fontFamily: {
                        sans: ['Inter', 'system-ui', 'sans-serif'],
                    },
                }
            }
        }
    </script>
    <style type="text/tailwindcss">
        @layer utilities {
            .content-auto {
                content-visibility: auto;
            }
            .card-shadow {
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            }
            .bg-gradient-primary {
                background: linear-gradient(135deg, #165DFF 0%, #4080FF 100%);
            }
            .modal-backdrop {
                backdrop-filter: blur(2px);
            }
        }
    </style>
</head>
<body class="bg-neutral-lightest min-h-screen font-sans flex flex-col">
    <!-- 顶部导航栏 -->
    <header class="bg-white shadow-md fixed top-0 left-0 right-0 z-10">
        <div class="container mx-auto px-4 py-3 flex items-center justify-between">
            <div class="flex items-center space-x-2">
                <i class="fa fa-bolt text-primary text-xl"></i>
                <h1 class="text-lg font-bold text-neutral">比赛跟踪系统</h1>
            </div>
        </div>
    </header>

    <!-- 主内容区 -->
    <main class="flex-1 container mx-auto px-4 pt-20 pb-24 flex flex-col items-center justify-center">
        <div class="w-full max-w-md">
            <div class="text-center mb-8">
                <div class="inline-flex items-center justify-center w-16 h-16 rounded-full bg-gradient-primary text-white mb-4">
                    <i class="fa fa-user-circle text-3xl"></i>
                </div>
                <h2 class="text-2xl font-bold text-neutral">账户登录</h2>
                <p class="text-neutral-lighter mt-2">请输入您的账号和密码</p>
            </div>
            
            <div class="bg-white rounded-xl p-6 card-shadow">
                <form id="loginForm" class="space-y-4">
                    <!-- 隐藏字段存储AccountNo -->
                    <?php if (!empty($accountNo)): ?>
                        <input type="hidden" name="AccountNo" value="<?php echo htmlspecialchars($accountNo); ?>">
                    <?php endif; ?>
                    
                    <div>
                        <label for="userName" class="block text-sm font-medium text-neutral-light mb-1">用户名</label>
                        <div class="relative">
                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <i class="fa fa-user text-neutral-lighter"></i>
                            </div>
                            <input type="text" id="userName" name="userName" 
                                class="w-full pl-10 pr-3 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary transition-colors"
                                placeholder="请输入用户名" value="<?php echo htmlspecialchars($userName); ?>">
                        </div>
                    </div>
                    
                    <div>
                        <label for="password" class="block text-sm font-medium text-neutral-light mb-1">密码</label>
                        <div class="relative">
                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <i class="fa fa-lock text-neutral-lighter"></i>
                            </div>
                            <input type="password" id="password" name="password" 
                                class="w-full pl-10 pr-3 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary/50 focus:border-primary transition-colors"
                                placeholder="请输入密码">
                            <button type="button" id="togglePassword" class="absolute inset-y-0 right-0 pr-3 flex items-center text-neutral-lighter hover:text-neutral transition-colors">
                                <i class="fa fa-eye-slash"></i>
                            </button>
                        </div>
                    </div>
                    
                    <div class="pt-2">
                        <button type="submit" class="w-full bg-primary hover:bg-primary/90 text-white font-medium py-3 px-4 rounded-lg transition-colors flex items-center justify-center">
                            <i class="fa fa-sign-in mr-2"></i> 登录
                        </button>
                    </div>
                </form>
                
                <div class="mt-4 text-center">
                    <a href="change_password.php<?php echo !empty($accountNo) ? '?AccountNo=' . urlencode($accountNo) : ''; ?>" class="text-primary hover:text-primary/80 text-sm transition-colors">
                        修改密码
                    </a>
                </div>
            </div>
        </div>
    </main>

    <!-- 底部导航 -->
    <nav class="fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 py-2 px-4 z-20">
        <div class="flex justify-around">
            <a href="bet_list.php<?php echo !empty($accountNo) ? '?AccountNo=' . urlencode($accountNo) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-home text-lg"></i>
                <span class="text-xs mt-1">首页</span>
            </a>
            <a href="bet_good.php<?php echo !empty($accountNo) ? '?AccountNo=' . urlencode($accountNo) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-line-chart text-lg"></i>
                <span class="text-xs mt-1">数据</span>
            </a>
            <a href="bet_result.php<?php echo !empty($accountNo) ? '?AccountNo=' . urlencode($accountNo) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-trophy text-lg"></i>
                <span class="text-xs mt-1">结果</span>
            </a>
            <a href="bet_notice.php<?php echo !empty($accountNo) ? '?AccountNo=' . urlencode($accountNo) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-bell text-lg"></i>
                <span class="text-xs mt-1">通知</span>
            </a>
            <a href="bet_my.php<?php echo !empty($accountNo) ? '?AccountNo=' . urlencode($accountNo) : ''; ?>" class="flex flex-col items-center text-neutral-light">
                <i class="fa fa-user text-lg"></i>
                <span class="text-xs mt-1">我的</span>
            </a>
        </div>
    </nav>

    <!-- 提示信息模态框 -->
    <div id="messageModal" class="fixed inset-0 bg-black/50 modal-backdrop z-50 hidden flex items-center justify-center p-4">
        <div class="bg-white rounded-xl w-full max-w-md overflow-hidden transform transition-all duration-300 scale-95 opacity-0" id="modalContent">
            <div class="p-6 text-center">
                <div id="modalIcon" class="inline-flex items-center justify-center w-16 h-16 rounded-full mb-4">
                    <i class="fa fa-exclamation-circle text-3xl"></i>
                </div>
                <h3 id="modalTitle" class="text-xl font-bold text-neutral mb-2">提示信息</h3>
                <p id="modalMessage" class="text-neutral-light mb-6">这是一条提示信息</p>
                <button id="modalClose" class="px-6 py-2 bg-primary hover:bg-primary/90 text-white font-medium rounded-lg transition-colors">
                    确定
                </button>
            </div>
        </div>
    </div>

    <script>
        // 获取AccountNo参数
        const urlParams = new URLSearchParams(window.location.search);
        const accountNo = urlParams.get('AccountNo') || '';
        
        // DOM元素
        const loginForm = document.getElementById('loginForm');
        const togglePassword = document.getElementById('togglePassword');
        const passwordInput = document.getElementById('password');
        const messageModal = document.getElementById('messageModal');
        const modalContent = document.getElementById('modalContent');
        const modalIcon = document.getElementById('modalIcon');
        const modalTitle = document.getElementById('modalTitle');
        const modalMessage = document.getElementById('modalMessage');
        const modalClose = document.getElementById('modalClose');

        // 显示模态框
        function showModal(title, message, type = 'error') {
            modalTitle.textContent = title;
            modalMessage.textContent = message;
            
            // 设置图标和颜色
            if (type === 'success') {
                modalIcon.className = 'inline-flex items-center justify-center w-16 h-16 rounded-full bg-success/10 text-success mb-4';
                modalIcon.innerHTML = '<i class="fa fa-check-circle text-3xl"></i>';
            } else {
                modalIcon.className = 'inline-flex items-center justify-center w-16 h-16 rounded-full bg-secondary/10 text-secondary mb-4';
                modalIcon.innerHTML = '<i class="fa fa-exclamation-circle text-3xl"></i>';
            }
            
            // 显示模态框并添加动画
            messageModal.classList.remove('hidden');
            setTimeout(() => {
                modalContent.classList.remove('scale-95', 'opacity-0');
                modalContent.classList.add('scale-100', 'opacity-100');
            }, 10);
        }

        // 隐藏模态框
        function hideModal() {
            modalContent.classList.remove('scale-100', 'opacity-100');
            modalContent.classList.add('scale-95', 'opacity-0');
            setTimeout(() => {
                messageModal.classList.add('hidden');
            }, 300);
        }

        // 密码显示/隐藏切换
        togglePassword.addEventListener('click', function() {
            if (passwordInput.type === 'password') {
                passwordInput.type = 'text';
                this.querySelector('i').classList.remove('fa-eye-slash');
                this.querySelector('i').classList.add('fa-eye');
            } else {
                passwordInput.type = 'password';
                this.querySelector('i').classList.remove('fa-eye');
                this.querySelector('i').classList.add('fa-eye-slash');
            }
        });

        // 关闭模态框
        modalClose.addEventListener('click', hideModal);

        // 点击模态框背景关闭
        messageModal.addEventListener('click', function(e) {
            if (e.target === this) {
                hideModal();
            }
        });

        // 登录表单提交
        loginForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const userName = document.getElementById('userName').value.trim();
            const password = document.getElementById('password').value.trim();
            
            // 前端验证
            if (!userName || !password) {
                showModal('输入错误', '用户名和密码不能为空');
                return;
            }
            
            try {
                // 显示加载状态
                const submitBtn = loginForm.querySelector('button[type="submit"]');
                const originalText = submitBtn.innerHTML;
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<i class="fa fa-spinner fa-spin mr-2"></i> 登录中...';
                
                // 准备提交数据，包含AccountNo
                const formData = {
                    userName: userName,
                    password: password
                };
                
                if (accountNo) {
                    formData.accountNo = accountNo;
                }
                
                // 发送登录请求
                const response = await fetch('login_validate.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(formData)
                });
                
                const result = await response.json();
                
                // 恢复按钮状态
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
                
                if (result.success) {
                    // 登录成功，跳转到个人主页并携带AccountNo参数
                    let redirectUrl = 'bet_my.php';
                    if (accountNo) {
                        redirectUrl += '?AccountNo=' + encodeURIComponent(accountNo);
                    }
                    
                    showModal('登录成功', '正在进入系统...', 'success');
                    setTimeout(() => {
                        window.location.href = redirectUrl;
                    }, 1500);
                } else {
                    // 登录失败，显示错误信息
                    showModal('登录失败', result.message || '用户名或密码不正确');
                }
            } catch (error) {
                console.error('登录请求失败:', error);
                showModal('系统错误', '登录过程中发生错误，请稍后重试');
                
                // 恢复按钮状态
                const submitBtn = loginForm.querySelector('button[type="submit"]');
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<i class="fa fa-sign-in mr-2"></i> 登录';
            }
        });

        // 检查是否有会话过期参数
        if (urlParams.get('session_expired')) {
            showModal('会话过期', '您的登录会话已过期，请重新登录');
        }
    </script>
</body>
</html>
