<?php
// 配置数据库连接

// 全局配置
const MAX_RETRIES = 1;
const RETRY_DELAY = 2; // 秒

// 初始化会话
session_start();

// CSRF令牌处理
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function verifyCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// 错误处理
function handleError($message, $isFatal = false, $logOnly = false) {
    $errorData = [
        'error' => true,
        'message' => $message,
        'timestamp' => date('Y-m-d H:i:s')
    ];
    
    error_log("系统错误: " . $message);
    
    if ($logOnly) return null;
    
    if ($isFatal) {
        header('Content-Type: application/json');
        http_response_code(500);
        echo json_encode($errorData);
        exit;
    }
    return $errorData;
}

// 数据库连接
try {
    $pdo = new PDO("mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
    $pdo->setAttribute(PDO::ATTR_TIMEOUT, 10);
} catch(PDOException $e) {
    handleError("数据库连接失败: " . $e->getMessage(), true);
}

// 账号验证
function authenticateAccount($pdo, $accountNo) {
    try {
        $stmt = $pdo->prepare("SELECT AccountNo FROM TelSms_Account WHERE AccountNo = :accountNo");
        $stmt->execute(['accountNo' => $accountNo]);
        return $stmt->fetch() !== false;
    } catch (Exception $e) {
        handleError("账号验证失败: " . $e->getMessage(), true);
        return false;
    }
}

// 获取账号参数
$accountNo = isset($_GET['AccountNo']) ? trim($_GET['AccountNo']) : '';
if ($accountNo && !authenticateAccount($pdo, $accountNo)) {
    handleError("账号验证失败，无权访问", true);
}

// 获取账号配置
function getAccountConfig($pdo, $accountNo) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM TelSms_Account WHERE AccountNo = :accountNo");
        $stmt->execute(['accountNo' => $accountNo]);
        $result = $stmt->fetch();
        if (!$result) throw new Exception("未找到账号配置: $accountNo");
        return $result;
    } catch (Exception $e) {
        throw new Exception("获取账号配置失败: " . $e->getMessage());
    }
}

// 获取账号配置
function getAdminAccountConfig($pdo, $accountNo) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM TelSms_Account WHERE AccountNo = :accountNo");
        $stmt->execute(['accountNo' => $accountNo]);
        $result = $stmt->fetch();
        if (!$result) throw new Exception("未找到账号配置: $accountNo");

        $username = $result['UserName'];

       $stmt = $pdo->prepare("SELECT * FROM TelSms_Admin WHERE UserName = :username");
        $stmt->execute(['username' => $username]);
        $result = $stmt->fetch();
      
        if (!$result) throw new Exception("未找到管理员配置: $username");

        return $result;
    } catch (Exception $e) {
        throw new Exception("获取账号配置失败: " . $e->getMessage());
    }
}

// 获取待推送数据
function getPendingBetData($pdo, $accountNo) {
    try {
        $stmt = $pdo->prepare("
            SELECT *
            FROM TelSms_BetList 
            WHERE AccountNo = :accountNo AND smsStatus = 0 
            ORDER BY BetTime ASC 
            LIMIT 1
        ");
        $stmt->execute(['accountNo' => $accountNo]);
        return $stmt->fetch();
    } catch (Exception $e) {
        throw new Exception("获取待推送数据失败: " . $e->getMessage());
    }
}

// 记录推送日志
function logPushResult($pdo, $accountNo, $title, $content, $success, $message = '') { 
    try { 
        $logContent = json_encode([ 
            'title' => $title, 
            'content' => $content, 
            'status' => $success ? '成功' : '失败', 
            'message' => $message,
            'timestamp' => date('Y-m-d H:i:s')
        ], JSON_UNESCAPED_UNICODE); 

        $stmt = $pdo->prepare(" 
            INSERT INTO TelSms_Logs (AccountNo, Contents, CreateDate) 
            VALUES (:accountNo, :contents, NOW()) 
        "); 
        $stmt->bindParam(':accountNo', $accountNo, PDO::PARAM_STR); 
        $stmt->bindParam(':contents', $logContent, PDO::PARAM_STR); 
        return $stmt->execute(); 
    } catch (Exception $e) { 
        throw new Exception("记录推送日志失败: " . $e->getMessage()); 
    } 
}

// 更新数据状态
function updateBetStatus($pdo, $id, $status) {
    try {
        $pdo->beginTransaction();
        $stmt = $pdo->prepare("
            UPDATE TelSms_BetList 
            SET smsStatus = :status, SendTime = NOW() 
            WHERE Id = :id AND smsStatus = 0
        ");
        $result = $stmt->execute(['status' => $status, 'id' => $id]);
        if ($result && $stmt->rowCount() === 0) {
            throw new Exception("数据状态已更新或不存在，ID: $id");
        }
        $pdo->commit();
        return $result && $stmt->rowCount() > 0;
    } catch (Exception $e) {
        $pdo->rollBack();
        throw new Exception("更新数据状态失败: " . $e->getMessage());
    }
}

// 发送到Telegram
function sendToTelegram($botToken, $chatId, $content) {
    $url = "https://api.telegram.org/bot$botToken/sendMessage";
    $data = [
        'chat_id' => $chatId,
        'text' => $content,
        'parse_mode' => 'HTML'
    ];

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($data),
        CURLOPT_TIMEOUT => 10,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);

    if ($error) {
        return ['ok' => false, 'description' => "请求失败: $error"];
    }
    if ($httpCode !== 200) {
        return ['ok' => false, 'description' => "状态码: $httpCode, 响应: $response"];
    }

    $result = json_decode($response, true);
    return $result ?: ['ok' => false, 'description' => "解析响应失败: $response"];
}

// 带重试的发送
function sendToTelegramWithRetry($botToken, $chatId, $content) {
    for ($i = 0; $i < MAX_RETRIES; $i++) {
        $result = sendToTelegram($botToken, $chatId, $content);
        if ($result['ok']) return $result;
        sleep(RETRY_DELAY);
    }
    return $result;
}

// 构建推送内容
function buildPushContent($accountConfig, $betData, $adminConfig) {
    // 创建当前时间对象并设置为北京时间
    $now = new DateTime('now', new DateTimeZone('Asia/Shanghai'));
    // 添加6个小时 
   // $now->add(new DateInterval('PT6H'));
    $startDate = new DateTime($adminConfig['StartDate']);
    $endDate = new DateTime($adminConfig['EndDate']);
    // 匹配所有记录的起始位置
    $text = $betData['BetContent'];
    $pattern = '/时间:  \d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}/';
    preg_match_all($pattern, $text, $matches, PREG_OFFSET_CAPTURE);

    if (!empty($matches[0])) {
        // 获取最后一条记录的起始位置
        $lastPos = end($matches[0])[1];
        // 提取最后一条记录（从起始位置到字符串结束）
        $lastRecord = substr($text, $lastPos);
        $text = $lastRecord;
    } 
    $betno = substr($betData['BetNo'], -4);

    if($betData && $betData['BetContent']){
        if ($now < $startDate || $now > $endDate) {
             return "⚠️ 账号已过期\n群编号：{$accountConfig['AccountNo']}\软件账号：{$accountConfig['UserName']}\n有效期：{$adminConfig['StartDate']} - {$adminConfig['EndDate']}\n\n当前时间：{$now->format('Y-m-d H:i:s')}";
        }else{
            return "{$betData['HgAdmin']}监控 会员:{$betData['BetAccount']} 单号:***{$betno} \n{$text}"
        ."\n\n数据中台:<a href=\"{$accountConfig['SiteUrl']}?AccountNo={$accountConfig['AccountNo']}\">点击查看</a>";
        }
    }

     return "";
}

// 构建无数据默认推送内容
function buildNoDataContent($accountConfig) {
     $now = new DateTime('now', new DateTimeZone('Asia/Shanghai'));
    $startDate = new DateTime($accountConfig['StartDate']);
    $endDate = new DateTime($accountConfig['EndDate']);
    
    // 账号状态判断
    $status = $now < $startDate ? "未生效" : 
              ($now > $endDate ? "已过期" : "正常");
    $statusIcon = $now > $endDate ? "⚠️" : "✅";
    
    return "📊 账号状态报告\n" .
           "账号：{$accountConfig['AccountNo']}\n" .
          // "用户名：{$accountConfig['UserName']}\n" .
           "状态：{$statusIcon} {$status}\n" .
           //"有效期：{$accountConfig['StartDate']} 至 {$accountConfig['EndDate']}\n" .
          // "管理后台：<a href=\"{$accountConfig['SiteUrl']}?AccountNo={$accountConfig['AccountNo']}\">点击查看</a>" .
           "更新时间：{$now->format('Y-m-d H:i:s')}\n" ;
         //  "（当前无新数据，本消息为定时状态推送）";
}

// API请求处理
if (isset($_GET['api'])) {
    header('Content-Type: application/json');
    try {
        switch ($_GET['api']) {
            case 'switch-account':
                $accountNo = $_GET['accountNo'] ?? '';
                if (!$accountNo) throw new Exception("账号不能为空");
                if (!authenticateAccount($pdo, $accountNo)) throw new Exception("账号验证失败");
                
                $config = getAccountConfig($pdo, $accountNo);
                //$betData = [];
                $betData = getPendingBetData($pdo, $accountNo);
                $adminConfig = getAdminAccountConfig($pdo, $accountNo);
                $preview = buildPushContent($config, $betData, $adminConfig);
                
                echo json_encode([
                    'success' => true,
                    'account' => $config,
                    'betData' => $betData,
                    'preview' => $preview
                ]);
                exit;

            case 'fetch-data':
                $accountNo = $_GET['accountNo'] ?? '';
                if (!$accountNo || !authenticateAccount($pdo, $accountNo)) throw new Exception("无权访问");
                
                //$betData = [];
                $betData = getPendingBetData($pdo, $accountNo);
                $config = getAccountConfig($pdo, $accountNo);
                $adminConfig = getAdminAccountConfig($pdo, $accountNo);
                $preview = buildPushContent($config, $betData, $adminConfig);
                
                echo json_encode([
                    'success' => true,
                    'betData' => $betData,
                    'preview' => $preview,
                    'timestamp' => date('Y-m-d H:i:s')
                ]);
                exit;

            case 'push-data':
                if ($_SERVER['REQUEST_METHOD'] !== 'POST') throw new Exception("请使用POST请求");

                $accountNo = $_GET['accountNo']?? '';
                $title = $_GET['title']?? '手动推送';
                $csrfToken = $_GET['csrf_token']?? '';
                
                if (!verifyCSRFToken($csrfToken)) throw new Exception("CSRF验证失败");
                if (!$accountNo) throw new Exception("缺少账号");
                
                $config = getAccountConfig($pdo, $accountNo);
                $betData = [];
                $betData = getPendingBetData($pdo,  $accountNo);
                $adminConfig = getAdminAccountConfig($pdo, $accountNo);
                $content='';
                $content = buildPushContent($config, $betData, $adminConfig);
                if ($content!== '') {

                    $result = sendToTelegramWithRetry($config['BotToken'], $config['ChatId'], $content);
                    $success = $result['ok'];
                    $message = $success ? '推送成功' : ($result['description'] ?? '未知错误');

                    $logContents = $content;
                    if (!empty($betData)) {
                        $logContents = 'AccountNo-' . $accountNo . '\n;BetAccount' . $betData['BetAccount'] . '\n;BetNo' . $betData['BetNo']
                            . '\n;BetType-' . $betData['BetType'] . '\n;BetName-' . $betData['BetName'] . '\n;BetHandType-' . $betData['BetHandType']
                            . '\n;BetTeamV-' . $betData['BetTeamV']
                            . '\n;SendTime-' . date('Y-m-d H:i:s');

                        $title =  '' . $accountNo . ';' . $betData['BetAccount'] . ';' . $betData['BetNo']
                            . ';' . $betData['BetType'] . ';' . $betData['BetName'];
                    }

                    logPushResult($pdo, $accountNo, $title, $logContents, $success, $message);

                    if ($success && $betData) {
                        $updateResult = updateBetStatus($pdo, $betData['Id'], 1);
                        if (!$updateResult) {
                            logPushResult($pdo, $accountNo, "状态更新失败", $content, false, "更新TelSms_BetList状态失败，ID: {$betData['Id']}");
                        }
                    }

                    $newBetData = getPendingBetData($pdo, $accountNo);

                    echo json_encode([
                        'success' => $success,
                        'message' => $message,
                        'newBetData' => $newBetData,
                        'newPreview' => buildPushContent($config, $newBetData, $adminConfig)
                    ]);
                }else{
                    echo json_encode([
                       'success' => false,
                       'message' => '无最新数据推送',
                        'newBetData' => $betData,
                        'newPreview' => $content
                    ]);
                }
                exit;

            case 'push-logs':
                $accountNo = $_GET['accountNo'] ?? '';
                if (!$accountNo || !authenticateAccount($pdo, $accountNo)) throw new Exception("无权访问");
                
                $stmt = $pdo->prepare("
                    SELECT Id, AccountNo, Contents, CreateDate 
                    FROM TelSms_Logs 
                    WHERE AccountNo = :accountNo 
                    ORDER BY CreateDate DESC 
                    LIMIT 10
                ");
                $stmt->execute(['accountNo' => $accountNo]);
                echo json_encode(['success' => true, 'data' => $stmt->fetchAll()]);
                exit;

            case 'push-log-detail':
                $id = (int)($_GET['id'] ?? 0);
                if (!$id) throw new Exception("缺少id参数");
                
                $stmt = $pdo->prepare("
                    SELECT l.*, a.UserName 
                    FROM TelSms_Logs l
                    JOIN TelSms_Account a ON l.AccountNo = a.AccountNo
                    WHERE l.Id = :id
                ");
                $stmt->execute(['id' => $id]);
                $log = $stmt->fetch();
                if (!$log) throw new Exception("日志不存在");
                if ($accountNo && $log['AccountNo'] !== $accountNo) throw new Exception("无权访问");
                
                echo json_encode(['success' => true, 'data' => $log]);
                exit;

            case 'test-connection':
                if ($_SERVER['REQUEST_METHOD'] !== 'POST') throw new Exception("请使用POST请求");
                $accountNo = $_GET['accountNo'] ?? '';
                if (!$accountNo || !authenticateAccount($pdo, $accountNo)) throw new Exception("无权访问");
                
                $config = getAccountConfig($pdo, $accountNo);
                 $content = buildNoDataContent($config);
                $result = sendToTelegram($config['BotToken'], $config['ChatId'], $content);
                
                echo json_encode([
                    'success' => $result['ok'],
                    'error' => $result['ok'] ? '' : $result['description']
                ]);
                exit;

            // 新增：无数据推送接口
            case 'push-no-data':
                if ($_SERVER['REQUEST_METHOD'] !== 'POST') throw new Exception("请使用POST请求");
                $accountNo = $_GET['accountNo'] ?? '';
               // throw new Exception($accountNo);
                if (!authenticateAccount($pdo, $accountNo)) throw new Exception("无权访问");
                
                $config = getAccountConfig($pdo, $accountNo);
                $content = buildNoDataContent($config);
                
                $result = sendToTelegramWithRetry($config['BotToken'], $config['ChatId'], $content);
                $success = $result['ok'];
                $message = $success ? '无数据推送成功' : ($result['description'] ?? '未知错误');
                
                logPushResult($pdo, $accountNo, "无数据定时推送", '账号状态报告', $success, $message);
                
                echo json_encode([
                    'success' => $success,
                    'message' => $message
                ]);
                exit;

            // 新增：检查数据是否存在
            case 'check-data-exists':
                $accountNo = $_GET['accountNo'] ?? '';
                if (!$accountNo || !authenticateAccount($pdo, $accountNo)) throw new Exception("无权访问");
                
                $hasData = getPendingBetData($pdo, $accountNo) !== false;
                echo json_encode([
                    'hasData' => $hasData,
                    'timestamp' => date('Y-m-d H:i:s')
                ]);
                exit;

            default:
                throw new Exception("未知API端点");
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        exit;
    }
}

// 加载账号配置和待推送数据
$accountConfig = null;
$pendingBetData = null;
$errorMessage = '';
try {
    if ($accountNo) {
        $accountConfig = getAccountConfig($pdo, $accountNo);
        $pendingBetData = getPendingBetData($pdo, $accountNo);
    }
} catch (Exception $e) {
    $errorMessage = $e->getMessage();
}

$csrfToken = generateCSRFToken();
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Telegram定时推送系统</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdn.jsdelivr.net/npm/font-awesome@4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#3b82f6',
                        success: '#10b981',
                        danger: '#ef4444',
                        dark: '#1e293b',
                    }
                }
            }
        }
    </script>
    <style type="text/tailwindcss">
        @layer utilities {
            .card-shadow { box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1); }
            .btn-hover { 
                transition: all 0.3s; 
                @apply hover:shadow-lg hover:-translate-y-0.5;
            }
            .input-focus { 
                @apply focus:ring-2 focus:ring-primary/50 focus:border-primary transition duration-200;
            }
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen flex flex-col">
    <header class="bg-white shadow-md sticky top-0 z-50">
        <div class="container mx-auto px-4 py-3 flex justify-between items-center">
            <div class="flex items-center space-x-2">
                <i class="fa fa-paper-plane text-primary text-2xl"></i>
                <h1 class="text-xl font-bold text-dark">Telegram定时推送系统</h1>
            </div>
            <span class="text-sm text-gray-600" id="current-account">当前账号: <?php echo htmlspecialchars($accountConfig['UserName'] ?? '未选择'); ?></span>
        </div>
    </header>

    <main class="flex-grow container mx-auto px-4 py-6 grid grid-cols-1 lg:grid-cols-3 gap-6">
        <!-- 左侧配置面板 -->
        <div class="lg:col-span-1 space-y-6">
            <div class="bg-white rounded-xl p-6 card-shadow">
                <h2 class="text-lg font-bold text-dark mb-4 flex items-center">
                    <i class="fa fa-user-circle text-primary mr-2"></i>选择账号
                </h2>
                <div class="flex">
                    <input type="text" id="account-input" placeholder="输入账号ID" 
                        class="flex-1 px-4 py-2 border border-gray-300 rounded-l-lg input-focus"
                        value="<?php echo htmlspecialchars($accountNo); ?>">
                    <button id="select-account" class="bg-primary text-white px-4 py-2 rounded-r-lg btn-hover">
                        <i class="fa fa-search mr-1"></i> 查找
                    </button>
                </div>
            </div>

            <div id="account-info" class="bg-white rounded-xl p-6 card-shadow hidden">
                <h2 class="text-lg font-bold text-dark mb-4 flex items-center">
                    <i class="fa fa-user-circle text-primary mr-2"></i>账号信息
                </h2>
                <div class="space-y-3">
                    <div class="grid grid-cols-2 gap-2">
                        <div><p class="text-xs text-gray-500">用户名</p><p class="font-medium" id="account-username">-</p></div>
                        <div><p class="text-xs text-gray-500">账号ID</p><p class="font-medium" id="account-id">-</p></div>
                    </div>
                    <div>
                        <p class="text-xs text-gray-500">有效期</p>
                        <p class="font-medium" id="account-validity">-</p>
                    </div>
                    <div class="mt-2 px-3 py-1 rounded-full text-sm font-medium" id="account-status">
                        <span class="bg-gray-100 text-gray-800">未加载</span>
                    </div>
                </div>
            </div>

            <div id="telegram-config" class="bg-white rounded-xl p-6 card-shadow hidden">
                <h2 class="text-lg font-bold text-dark mb-4 flex items-center">
                    <i class="fa fa-telegram text-primary mr-2"></i>Telegram配置
                </h2>
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Bot Token</label>
                        <input type="text" id="bot-token" class="w-full px-4 py-2 border border-gray-300 rounded-lg input-focus" disabled>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">聊天ID</label>
                        <input type="text" id="chat-id" class="w-full px-4 py-2 border border-gray-300 rounded-lg input-focus" disabled>
                    </div>
                    <button id="test-telegram" class="w-full bg-primary text-white px-4 py-2 rounded-lg flex items-center justify-center btn-hover">
                        <i class="fa fa-paper-plane-o mr-2"></i>测试连接
                    </button>
                </div>
            </div>

            <div class="bg-white rounded-xl p-6 card-shadow">
                <h2 class="text-lg font-bold text-dark mb-4 flex items-center">
                    <i class="fa fa-refresh text-primary mr-2"></i>数据刷新配置
                </h2>
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">自动刷新间隔 (秒)</label>
                        <input type="number" id="refresh-interval" min="1" value="1" 
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg input-focus">
                    </div>
                    <div class="flex items-center">
                        <input type="checkbox" id="auto-refresh" class="form-checkbox h-5 w-5 text-primary rounded" checked>
                        <label for="auto-refresh" class="ml-2 text-sm text-gray-700">启用自动刷新</label>
                    </div>
                    <button id="manual-refresh" class="w-full bg-primary text-white px-4 py-2 rounded-lg flex items-center justify-center btn-hover">
                        <i class="fa fa-refresh mr-2"></i>手动刷新数据
                    </button>
                </div>
            </div>

            <!-- 新增：无数据推送配置 -->
            <div id="no-data-push-config" class="bg-white rounded-xl p-6 card-shadow hidden">
                <h2 class="text-lg font-bold text-dark mb-4 flex items-center">
                    <i class="fa fa-clock-o text-orange-500 mr-2"></i>无数据推送设置
                </h2>
                <div class="space-y-4">
                    <div class="flex items-center">
                        <input type="checkbox" id="enable-no-data-push" class="form-checkbox h-5 w-5 text-orange-500 rounded">
                        <label for="enable-no-data-push" class="ml-2 text-sm text-gray-700">启用无数据定时推送</label>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">推送类型</label>
                        <select id="no-data-push-type" class="w-full px-4 py-2 border border-gray-300 rounded-lg input-focus">
                            <option value="interval">间隔时间（秒）</option>
                            <option value="specific">指定时间（HH:MM）</option>
                        </select>
                    </div>
                    
                    <div id="no-data-interval-config">
                        <label class="block text-sm font-medium text-gray-700 mb-1">推送间隔（秒，建议≥30）</label>
                        <input type="number" id="no-data-interval" min="30" value="3600" 
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg input-focus">
                    </div>
                    
                    <div id="no-data-specific-config" class="hidden">
                        <label class="block text-sm font-medium text-gray-700 mb-1">指定推送时间（多个用逗号分隔）</label>
                        <input type="text" id="no-data-specific-time" placeholder="例如: 09:00,15:30,21:00" 
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg input-focus" value="12:00">
                    </div>
                    
                    <button id="save-no-data-config" class="w-full bg-orange-500 text-white px-4 py-2 rounded-lg flex items-center justify-center btn-hover">
                        <i class="fa fa-save mr-2"></i>保存配置
                    </button>
                </div>
            </div>

            <div id="cron-config" class="bg-white rounded-xl p-6 card-shadow hidden">
                <h2 class="text-lg font-bold text-dark mb-4 flex items-center">
                    <i class="fa fa-clock-o text-primary mr-2"></i>定时推送配置
                </h2>
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">定时类型</label>
                        <select id="cron-type" class="w-full px-4 py-2 border border-gray-300 rounded-lg input-focus">
                            <option value="interval">间隔时间</option>
                            <option value="specific">指定时间</option>
                        </select>
                    </div>
                    
                    <div id="interval-config">
                        <label class="block text-sm font-medium text-gray-700 mb-1">间隔时间 (秒)</label>
                        <input type="number" id="interval-time" min="2" value="2" 
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg input-focus">
                    </div>
                    
                    <div id="specific-config" class="hidden">
                        <label class="block text-sm font-medium text-gray-700 mb-1">指定时间 (HH:MM)</label>
                        <input type="text" id="specific-time" placeholder="例如: 09:30" 
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg input-focus">
                        <p class="text-xs text-gray-500 mt-1">支持多个时间，用逗号分隔</p>
                    </div>
                    
                    <div class="flex items-center">
                        <input type="checkbox" id="enable-cron" checked="true" class="form-checkbox h-5 w-5 text-primary rounded">
                        <label for="enable-cron" class="ml-2 text-sm text-gray-700">启用定时推送</label>
                    </div>
                    
                    <button id="save-cron" class="w-full bg-primary text-white px-4 py-2 rounded-lg flex items-center justify-center btn-hover">
                        <i class="fa fa-save mr-2"></i>保存定时设置
                    </button>
                </div>
            </div>
        </div>

        <!-- 右侧数据管理 -->
        <div class="lg:col-span-2 space-y-6">
            <div id="bet-data-container" class="bg-white rounded-xl p-6 card-shadow hidden">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-lg font-bold text-dark flex items-center">
                        <i class="fa fa-table text-primary mr-2"></i>待推送数据
                    </h2>
                    <div class="text-sm text-gray-500" id="last-refresh-time">上次刷新: --</div>
                </div>
                
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">内容</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">创建时间</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">状态</th>
                            </tr>
                        </thead>
                        <tbody id="data-table-body" class="bg-white divide-y divide-gray-200">
                            <tr><td colspan="3" class="px-6 py-12 text-center text-gray-500"><i class="fa fa-info-circle mr-2"></i>请选择账号加载数据</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <div id="preview-container" class="bg-white rounded-xl p-6 card-shadow hidden">
                <h2 class="text-lg font-bold text-dark mb-4 flex items-center">
                    <i class="fa fa-eye text-primary mr-2"></i>推送内容预览
                </h2>
                <div id="preview-content" class="bg-gray-50 p-4 rounded-lg min-h-[200px] overflow-auto font-mono text-sm">
                    请选择账号查看预览
                </div>
            </div>

            <div id="push-actions" class="bg-white rounded-xl p-6 card-shadow hidden">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-lg font-bold text-dark flex items-center">
                        <i class="fa fa-paper-plane text-primary mr-2"></i>推送操作
                    </h2>
                </div>
                <div class="flex flex-wrap gap-3">
                    <button id="manual-push" class="bg-success text-white px-6 py-3 rounded-lg flex items-center btn-hover">
                        <i class="fa fa-play mr-2"></i>立即推送
                    </button>
                    <button id="push-with-title" class="bg-primary text-white px-6 py-3 rounded-lg flex items-center btn-hover">
                        <i class="fa fa-plus-circle mr-2"></i>带标题推送
                    </button>
                </div>
                
                <!-- 带标题推送的模态框 -->
                <div id="push-modal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
                    <div class="bg-white rounded-lg p-6 max-w-md w-full">
                        <h3 class="text-lg font-bold mb-4">带标题推送</h3>
                        <div class="mb-4">
                            <label for="push-title" class="block text-sm font-medium text-gray-700 mb-1">推送标题</label>
                            <input type="text" id="push-title" placeholder="请输入推送标题" class="w-full px-4 py-2 border border-gray-300 rounded-lg input-focus">
                        </div>
                        <div class="flex justify-end space-x-3">
                            <button id="cancel-push" class="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 btn-hover">取消</button>
                            <button id="confirm-push" class="px-4 py-2 bg-primary text-white rounded-lg btn-hover">确认推送</button>
                        </div>
                    </div>
                </div>
            </div>

            <div id="logs-container" class="bg-white rounded-xl p-6 card-shadow hidden">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-lg font-bold text-dark flex items-center">
                        <i class="fa fa-history text-primary mr-2"></i>推送日志
                    </h2>
                </div>
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">推送时间</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">标题</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">状态</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">操作</th>
                            </tr>
                        </thead>
                        <tbody id="log-table-body" class="bg-white divide-y divide-gray-200">
                            <tr><td colspan="4" class="px-6 py-12 text-center text-gray-500"><i class="fa fa-info-circle mr-2"></i>请选择账号加载日志</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>

    <footer class="bg-white border-t border-gray-200 py-4">
        <div class="container mx-auto px-4 text-center text-gray-500 text-sm">
            <p>© 2025 Telegram定时推送系统 | 专业级数据推送解决方案</p>
        </div>
    </footer>

    <!-- 错误模态框 -->
    <div id="error-modal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
        <div class="bg-white rounded-lg p-6 max-w-md w-full">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-bold text-red-600 flex items-center">
                    <i class="fa fa-exclamation-circle mr-2"></i>错误提示
                </h3>
                <button id="close-error-modal" class="text-gray-400 hover:text-gray-600"><i class="fa fa-times"></i></button>
            </div>
            <div id="error-message" class="text-gray-700 mb-4"></div>
            <div class="flex justify-end">
                <button id="confirm-error" class="px-4 py-2 bg-primary text-white rounded-lg btn-hover">确认</button>
            </div>
        </div>
    </div>

    <!-- 通知提示 -->
    <div id="notification" class="fixed top-4 right-4 max-w-sm bg-white rounded-lg shadow-lg p-4 transform transition-all duration-300 translate-x-full opacity-0 z-50 flex items-start">
        <div id="notification-icon" class="mr-3 text-xl"></div>
        <div class="flex-1">
            <h3 id="notification-title" class="font-medium text-gray-900"></h3>
            <p id="notification-message" class="text-sm text-gray-600 mt-1"></p>
        </div>
        <button id="close-notification" class="ml-2 text-gray-400 hover:text-gray-600"><i class="fa fa-times"></i></button>
    </div>

    <!-- 日志详情模态框 -->
    <div id="log-detail-modal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
        <div class="bg-white rounded-lg p-6 max-w-2xl w-full max-h-[80vh] overflow-auto">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-lg font-bold">推送详情</h3>
                <button id="close-log-detail" class="text-gray-400 hover:text-gray-600"><i class="fa fa-times"></i></button>
            </div>
            <div id="log-detail-content" class="space-y-4"></div>
        </div>
    </div>

    <script>
        // 全局变量
        let refreshInterval = 3;
        let autoRefresh = true;
        let refreshTimer = null;
        let cronTimer = null;
        let dailyCronTimers = [];
        let currentAccount = '<?php echo htmlspecialchars($accountNo); ?>';
        let noDataTimer = null; // 无数据推送定时器
        const csrfToken = '<?php echo htmlspecialchars($csrfToken); ?>';

        // DOM元素
        const elements = {
            accountInput: document.getElementById('account-input'),
            selectAccount: document.getElementById('select-account'),
            currentAccountEl: document.getElementById('current-account'),
            accountInfo: document.getElementById('account-info'),
            accountUsername: document.getElementById('account-username'),
            accountId: document.getElementById('account-id'),
            accountValidity: document.getElementById('account-validity'),
            accountStatus: document.getElementById('account-status'),
            telegramConfig: document.getElementById('telegram-config'),
            botToken: document.getElementById('bot-token'),
            chatId: document.getElementById('chat-id'),
            testTelegram: document.getElementById('test-telegram'),
            refreshInterval: document.getElementById('refresh-interval'),
            autoRefresh: document.getElementById('auto-refresh'),
            manualRefresh: document.getElementById('manual-refresh'),
            lastRefreshTime: document.getElementById('last-refresh-time'),
            cronConfig: document.getElementById('cron-config'),
            cronType: document.getElementById('cron-type'),
            intervalConfig: document.getElementById('interval-config'),
            specificConfig: document.getElementById('specific-config'),
            intervalTime: document.getElementById('interval-time'),
            specificTime: document.getElementById('specific-time'),
            enableCron: document.getElementById('enable-cron'),
            saveCron: document.getElementById('save-cron'),
            betDataContainer: document.getElementById('bet-data-container'),
            dataTableBody: document.getElementById('data-table-body'),
            previewContainer: document.getElementById('preview-container'),
            previewContent: document.getElementById('preview-content'),
            pushActions: document.getElementById('push-actions'),
            manualPush: document.getElementById('manual-push'),
            pushWithTitle: document.getElementById('push-with-title'),
            pushModal: document.getElementById('push-modal'),
            cancelPush: document.getElementById('cancel-push'),
            confirmPush: document.getElementById('confirm-push'),
            pushTitle: document.getElementById('push-title'),
            logsContainer: document.getElementById('logs-container'),
            logTableBody: document.getElementById('log-table-body'),
            logDetailModal: document.getElementById('log-detail-modal'),
            closeLogDetail: document.getElementById('close-log-detail'),
            logDetailContent: document.getElementById('log-detail-content'),
            errorModal: document.getElementById('error-modal'),
            errorMessage: document.getElementById('error-message'),
            closeErrorModal: document.getElementById('close-error-modal'),
            confirmError: document.getElementById('confirm-error'),
            notification: document.getElementById('notification'),
            notificationIcon: document.getElementById('notification-icon'),
            notificationTitle: document.getElementById('notification-title'),
            notificationMessage: document.getElementById('notification-message'),
            closeNotification: document.getElementById('close-notification'),
            // 无数据推送配置元素
            noDataPushConfig: document.getElementById('no-data-push-config'),
            enableNoDataPush: document.getElementById('enable-no-data-push'),
            noDataPushType: document.getElementById('no-data-push-type'),
            noDataIntervalConfig: document.getElementById('no-data-interval-config'),
            noDataSpecificConfig: document.getElementById('no-data-specific-config'),
            noDataInterval: document.getElementById('no-data-interval'),
            noDataSpecificTime: document.getElementById('no-data-specific-time'),
            saveNoDataConfig: document.getElementById('save-no-data-config')
        };

        // 初始化
        document.addEventListener('DOMContentLoaded', () => {
            // 绑定事件
            elements.selectAccount.addEventListener('click', switchAccount);
            elements.accountInput.addEventListener('keypress', e => e.key === 'Enter' && switchAccount());
            elements.refreshInterval.addEventListener('change', updateRefreshInterval);
            elements.autoRefresh.addEventListener('change', toggleAutoRefresh);
            elements.manualRefresh.addEventListener('click', fetchData);
            elements.testTelegram.addEventListener('click', testTelegramConnection);
            elements.cronType.addEventListener('change', toggleCronType);
            elements.saveCron.addEventListener('click', saveCronSettings);
            elements.manualPush.addEventListener('click', pushData);
            elements.pushWithTitle.addEventListener('click', showPushModal);
            elements.cancelPush.addEventListener('click', hidePushModal);
            elements.confirmPush.addEventListener('click', confirmPush);
            elements.closeLogDetail.addEventListener('click', hideLogDetailModal);
            elements.closeErrorModal.addEventListener('click', hideErrorModal);
            elements.confirmError.addEventListener('click', hideErrorModal);
            elements.closeNotification.addEventListener('click', hideNotification);
            
            // 无数据推送配置事件
            elements.noDataPushType.addEventListener('change', toggleNoDataPushType);
            elements.saveNoDataConfig.addEventListener('click', saveNoDataConfig);

            // 如果已有账号，加载数据
            if (currentAccount) {
                switchAccount();
                saveCronSettings();
            }
        });

        // 切换账号
        function switchAccount() {
            const accountNo = elements.accountInput.value.trim();
            if (!accountNo) {
                showNotification('错误', '请输入账号ID', 'danger');
                return;
            }

            showNotification('信息', '切换账号中...', 'info');
            fetch(`?api=switch-account&accountNo=${encodeURIComponent(accountNo)}`)
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        currentAccount = accountNo;
                        updateAccountUI(data.account);
                        updateBetDataUI(data.betData);
                        elements.previewContent.textContent = data.preview;
                        
                        elements.accountInfo.classList.remove('hidden');
                        elements.telegramConfig.classList.remove('hidden');
                        elements.betDataContainer.classList.remove('hidden');
                        elements.previewContainer.classList.remove('hidden');
                        elements.pushActions.classList.remove('hidden');
                        elements.cronConfig.classList.remove('hidden');
                        elements.logsContainer.classList.remove('hidden');
                        elements.noDataPushConfig.classList.remove('hidden'); // 显示无数据配置
                        
                        loadPushLogs();
                        startAutoRefresh();
                        setupCronJob();
                        loadNoDataConfig(); // 加载无数据推送配置
                        showNotification('成功', '账号切换成功', 'success');
                    } else {
                        throw new Error(data.error || '切换账号失败');
                    }
                })
                .catch(err => {
                    showErrorModal(err.message);
                    console.error('切换账号失败:', err);
                });
        }

        // 更新账号信息UI
        function updateAccountUI(account) {
            elements.currentAccountEl.textContent = `当前账号: ${account.UserName}`;
            elements.accountUsername.textContent = account.UserName;
            elements.accountId.textContent = account.AccountNo;
            elements.accountValidity.textContent = `${formatDate(account.StartDate)} - ${formatDate(account.EndDate)}`;
            
            // 处理BotToken显示（部分隐藏）
            const token = account.BotToken;
            elements.botToken.value = token ? `${token.substring(0, 10)}...${token.substring(token.length - 4)}` : '';
            elements.chatId.value = account.ChatId || '';
            
            // 更新账号状态
            const now = new Date();
            const start = new Date(account.StartDate);
            const end = new Date(account.EndDate);
            
            if (now < start) {
                elements.accountStatus.innerHTML = '<span class="bg-yellow-100 text-yellow-800">账号未生效</span>';
            } else if (now > end) {
                elements.accountStatus.innerHTML = '<span class="bg-red-100 text-red-800">账号已过期</span>';
            } else {
                elements.accountStatus.innerHTML = '<span class="bg-green-100 text-green-800">账号正常</span>';
            }
        }

        // 更新数据表格UI
        function updateBetDataUI(betData) {
            if (betData) {
                elements.dataTableBody.innerHTML = `
                    <tr>
                        <td class="px-6 py-4"><pre class="whitespace-pre-wrap break-words">${escapeHtml(betData.BetContent)}</pre></td>
                        <td class="px-6 py-4 whitespace-nowrap">${formatDateTime(betData.CreateDate)}</td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="px-2 py-1 text-xs font-medium bg-blue-100 text-blue-800 rounded-full">待推送</span>
                        </td>
                    </tr>
                `;
            } else {
                elements.dataTableBody.innerHTML = `
                    <tr><td colspan="3" class="px-6 py-12 text-center text-gray-500"><i class="fa fa-info-circle mr-2"></i>暂无待推送数据</td></tr>
                `;
            }
        }

        // 加载推送日志
        function loadPushLogs() {
            if (!currentAccount) return;

            fetch(`?api=push-logs&accountNo=${currentAccount}`)
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        elements.logTableBody.innerHTML = '';
                        
                        data.data.forEach(log => {
                            const content = JSON.parse(log.Contents || '{}');

                            const row = document.createElement('tr');
                            row.innerHTML = `
                                <td class="px-6 py-4 whitespace-nowrap">${formatDateTime(log.CreateDate)}</td>
                                <td class="px-6 py-4">${content.title || '无标题'}</td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="px-2 py-1 text-xs font-medium ${content.status === '成功' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'} rounded-full">
                                        ${content.status}
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <button onclick="showLogDetail(${log.Id})" class="text-primary hover:text-primary/80">查看详情</button>
                                </td>
                            `;
                            elements.logTableBody.appendChild(row);
                        });

                    } else {
                        throw new Error(data.error || '加载日志失败');
                    }
                })
                .catch(err => {
                    showNotification('错误', `加载日志失败: ${err.message}`, 'danger');
                    console.error('加载日志失败:', err);
                });
        }

        // 获取最新数据
        function fetchData() {
            if (!currentAccount) {
                showNotification('错误', '请先选择账号', 'danger');
                return;
            }

            fetch(`?api=fetch-data&accountNo=${currentAccount}`)
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        updateBetDataUI(data.betData);
                        elements.previewContent.textContent = data.preview;
                        elements.lastRefreshTime.textContent = `上次刷新: ${data.timestamp}`;
                    } else {
                        throw new Error(data.error || '刷新数据失败');
                    }
                })
                .catch(err => {
                    showNotification('错误', `刷新失败: ${err.message}`, 'danger');
                    console.error('刷新数据失败:', err);
                });
        }

        // 推送数据
        function pushData(title = '手动推送') {
            if (!currentAccount) {
                showNotification('错误', '请先选择账号', 'danger');
                return;
            }

            showNotification('信息', '正在推送数据...', 'info');
            fetch(`?api=push-data&accountNo=${currentAccount}&title=${encodeURIComponent(title)}&csrf_token=${csrfToken}`, {
                method: 'POST'
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    updateBetDataUI(data.newBetData);
                    elements.previewContent.textContent = data.newPreview;
                    loadPushLogs();
                    showNotification('成功', data.message, 'success');
                } else {
                    throw new Error(data.error || data.message || '推送失败');
                }
            })
            .catch(err => {
              //  showNotification('错误', `推送失败: ${err.message}`, 'danger');
                console.error('推送数据失败:', err);
            });
        }

        // 测试Telegram连接
        function testTelegramConnection() {
            if (!currentAccount) {
                showNotification('错误', '请先选择账号', 'danger');
                return;
            }

            showNotification('信息', '测试连接中...', 'info');
            fetch(`?api=test-connection&accountNo=${currentAccount}`, { method: 'POST' })
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        showNotification('成功', '测试消息发送成功', 'success');
                    } else {
                        throw new Error(data.error || '测试连接失败');
                    }
                })
                .catch(err => {
                    showNotification('失败', `测试失败: ${err.message}`, 'danger');
                    console.error('测试连接失败:', err);
                });
        }

        // 自动刷新相关
        function updateRefreshInterval() {
            const value = parseInt(elements.refreshInterval.value);
            if (value >= 1) {
                refreshInterval = value;
                restartAutoRefresh();
                showNotification('成功', `自动刷新间隔已设为 ${value} 秒`, 'success');
            } else {
                showNotification('错误', '刷新间隔不能小于1秒', 'danger');
                elements.refreshInterval.value = refreshInterval;
            }
        }

        function toggleAutoRefresh() {
            autoRefresh = elements.autoRefresh.checked;
            autoRefresh ? startAutoRefresh() : stopAutoRefresh();
            showNotification('成功', autoRefresh ? '已启用自动刷新' : '已禁用自动刷新', 'success');
        }

        function startAutoRefresh() {
            stopAutoRefresh();
            if (autoRefresh && currentAccount) {
                refreshTimer = setInterval(fetchData, refreshInterval * 1000);
                console.log(`自动刷新已启动，间隔 ${refreshInterval} 秒`);
            }
        }

        function stopAutoRefresh() {
            if (refreshTimer) {
                clearInterval(refreshTimer);
                refreshTimer = null;
                console.log('自动刷新已停止');
            }
        }

        function restartAutoRefresh() {
            stopAutoRefresh();
            startAutoRefresh();
        }

        // 定时任务设置
        function toggleCronType() {
            if (elements.cronType.value === 'interval') {
                elements.intervalConfig.classList.remove('hidden');
                elements.specificConfig.classList.add('hidden');
            } else {
                elements.intervalConfig.classList.add('hidden');
                elements.specificConfig.classList.remove('hidden');
            }
        }

        function saveCronSettings() {
            try {
                const config = {
                    cronType: elements.cronType.value,
                    intervalTime: parseInt(elements.intervalTime.value) || 1,
                    specificTime: elements.specificTime.value,
                    enableCron: elements.enableCron.checked
                };

                if (config.enableCron) {
                    if (config.cronType === 'interval' && config.intervalTime < 1) {
                        throw new Error('间隔时间不能小于1秒');
                    }
                    if (config.cronType === 'specific') {
                        const times = config.specificTime.split(',').map(t => t.trim());
                        times.forEach(time => {
                            if (time && !/^\d{1,2}:\d{2}$/.test(time)) {
                                throw new Error('时间格式错误，请使用HH:MM');
                            }
                        });
                    }
                }

                localStorage.setItem('cronConfig', JSON.stringify(config));
                setupCronJob();
                showNotification('成功', '定时设置已保存', 'success');
            } catch (err) {
                showNotification('错误', err.message, 'danger');
            }
        }

        function setupCronJob() {
            clearCronJob();
            const config = JSON.parse(localStorage.getItem('cronConfig') || '{"enableCron": false}');
            
            if (config.enableCron && currentAccount) {
                if (config.cronType === 'interval') {
                    cronTimer = setInterval(() => pushData('定时推送'), config.intervalTime * 1000);
                    showNotification('信息', `已设置间隔推送（${config.intervalTime}秒）`, 'info');
                } else {
                    setupSpecificTimeCron(config.specificTime);
                    showNotification('信息', `已设置定时推送（${config.specificTime}）`, 'info');
                }
            } else {
                showNotification('信息', '定时推送已禁用', 'info');
            }
        }

        function setupSpecificTimeCron(times) {
            const now = new Date();
            times.forEach(time => {
                if (!time) return;
                const [hours, minutes] = time.split(':').map(Number);
                if (isNaN(hours) || isNaN(minutes)) return;

                const scheduled = new Date(now);
                scheduled.setHours(hours, minutes, 0, 0);
                if (scheduled <= now) scheduled.setDate(scheduled.getDate() + 1);
                
                const delay = scheduled - now;
                console.log(`定时推送将在${new Date(scheduled).toLocaleString()}执行`);
                
                const timer = setTimeout(() => {
                    pushData('定时推送');
                    setupSpecificTimeCron(times); // 重复设置明天的定时器
                }, delay);
                dailyCronTimers.push(timer);
            });
        }

        function clearCronJob() {
            if (cronTimer) clearInterval(cronTimer);
            dailyCronTimers.forEach(timer => clearTimeout(timer));
            dailyCronTimers = [];
        }

        // 无数据推送配置
        function toggleNoDataPushType() {
            const isInterval = elements.noDataPushType.value === 'interval';
            elements.noDataIntervalConfig.classList.toggle('hidden', !isInterval);
            elements.noDataSpecificConfig.classList.toggle('hidden', isInterval);
        }

        function saveNoDataConfig() {
            if (!currentAccount) {
                showNotification('错误', '请先选择账号', 'danger');
                return;
            }

            try {
                const config = {
                    enable: elements.enableNoDataPush.checked,
                    type: elements.noDataPushType.value,
                    interval: parseInt(elements.noDataInterval.value) || 3600,
                    specificTime: elements.noDataSpecificTime.value.trim() || '12:00'
                };

                // 验证配置
                if (config.enable) {
                    if (config.type === 'interval' && config.interval < 1) {
                        throw new Error('间隔时间不能小于1秒');
                    }
                    if (config.type === 'specific') {
                        const times = config.specificTime.split(',').map(t => t.trim());
                        times.forEach(time => {
                            if (time && !/^\d{1,2}:\d{2}$/.test(time)) {
                                throw new Error('时间格式错误，请使用HH:MM');
                            }
                        });
                    }
                }

                // 保存到localStorage（按账号区分）
                localStorage.setItem(`noDataConfig_${currentAccount}`, JSON.stringify(config));
                setupNoDataCronJob(); // 重新初始化定时任务
                showNotification('成功', '无数据推送配置已保存', 'success');
            } catch (err) {
                showNotification('错误', err.message, 'danger');
            }
        }

        function loadNoDataConfig() {
            if (!currentAccount) return;
            
            const config = JSON.parse(localStorage.getItem(`noDataConfig_${currentAccount}`) || '{"enable": false}');
            elements.enableNoDataPush.checked = config.enable;
            elements.noDataPushType.value = config.type || 'interval';
            elements.noDataInterval.value = config.interval || 3600;
            elements.noDataSpecificTime.value = config.specificTime || '12:00';
            
            toggleNoDataPushType(); // 切换显示正确的配置区域
            setupNoDataCronJob(); // 启动定时任务
        }

        function setupNoDataCronJob() {
            // 清除现有任务
            if (noDataTimer) clearInterval(noDataTimer);
            noDataTimer = null;
            
            const config = JSON.parse(localStorage.getItem(`noDataConfig_${currentAccount}`) || '{"enable": false}');
            if (!config.enable || !currentAccount) return;
            
            // 间隔推送
            if (config.type === 'interval') {
                noDataTimer = setInterval(() => {
                    checkAndPushNoData();
                }, config.interval * 1000);
                console.log(`无数据推送定时器启动（间隔${config.interval}秒）`);
            } 
            // 特定时间推送
            else {
                setupSpecificNoDataPush(config.specificTime);
            }
        }

        function checkAndPushNoData() {
            fetch(`?api=check-data-exists&accountNo=${currentAccount}`)
                .then(res => res.json())
                .then(data => {
                    if (!data.hasData) { // 无数据时执行推送
                        pushNoDataContent();
                    }
                })
                .catch(err => {
                    console.error('检查数据失败:', err);
                });
        }

        function pushNoDataContent() {
            fetch(`?api=push-no-data&accountNo=`+currentAccount, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({accountNo: currentAccount})
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    showNotification('信息', '无数据定时推送成功', 'info');
                    loadPushLogs(); // 刷新日志
                } else {
                    showNotification('错误', `无数据推送失败: ${data.error || data.message}`, 'danger');
                }
            })
            .catch(err => {
                showNotification('错误', `无数据推送失败: ${err.message}`, 'danger');
                console.error('无数据推送失败:', err);
            });
        }

        function setupSpecificNoDataPush(timeStr) {
            const times = timeStr.split(',').map(t => t.trim());
            const now = new Date();
            times.forEach(time => {
                if (!time) return;
                const [hours, minutes] = time.split(':').map(Number);
                if (isNaN(hours) || isNaN(minutes)) return;

                const pushTime = new Date(now);
                pushTime.setHours(hours, minutes, 0, 0);
                if (pushTime <= now) pushTime.setDate(pushTime.getDate() + 1);
                
                const delay = pushTime - now;
                console.log(`无数据推送将在${new Date(pushTime).toLocaleString()}执行`);
                
                setTimeout(() => {
                    checkAndPushNoData();
                    setupSpecificNoDataPush(timeStr); // 重复设置明天的定时器
                }, delay);
            });
        }

        // 模态框控制
        function showPushModal() {
            elements.pushModal.classList.remove('hidden');
            elements.pushTitle.value = '';
            elements.pushTitle.focus();
        }

        function hidePushModal() {
            elements.pushModal.classList.add('hidden');
        }

        function confirmPush() {
            const title = elements.pushTitle.value.trim() || '带标题推送';
            hidePushModal();
            pushData(title);
        }

        function showLogDetail(id) {
            fetch(`?api=push-log-detail&id=${id}`)
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        const log = data.data;
                        const content = JSON.parse(log.Contents || '{}');
                        
                        elements.logDetailContent.innerHTML = `
                            <p><strong>标题:</strong> ${content.title || '无标题'}</p>
                            <p><strong>状态:</strong> ${content.status || '未知'}</p>
                            <p><strong>时间:</strong> ${formatDateTime(log.CreateDate)}</p>
                            <p><strong>消息:</strong> ${content.message || '无'}</p>
                            <p><strong>内容:</strong></p>
                            <pre class="bg-gray-50 p-3 rounded">${content.content || '无内容'}</pre>
                        `;
                        elements.logDetailModal.classList.remove('hidden');
                    } else {
                        throw new Error(data.error || '获取日志详情失败');
                    }
                })
                .catch(err => {
                    showNotification('错误', `加载详情失败: ${err.message}`, 'danger');
                    console.error('加载日志详情失败:', err);
                });
        }

        function hideLogDetailModal() {
            elements.logDetailModal.classList.add('hidden');
        }

        // 错误和通知
        function showErrorModal(message) {
            elements.errorMessage.textContent = message;
            elements.errorModal.classList.remove('hidden');
        }

        function hideErrorModal() {
            elements.errorModal.classList.add('hidden');
        }

        function showNotification(title, message, type) {
            const iconClass = {
                success: 'fa-check-circle text-success',
                danger: 'fa-exclamation-circle text-danger',
                info: 'fa-info-circle text-primary'
            }[type] || 'fa-info-circle text-primary';

            elements.notificationIcon.className = `fa ${iconClass}`;
            elements.notificationTitle.textContent = title;
            elements.notificationMessage.textContent = message;
            elements.notification.classList.remove('translate-x-full', 'opacity-0');

            setTimeout(hideNotification, 5000);
        }

        function hideNotification() {
            elements.notification.classList.add('translate-x-full', 'opacity-0');
        }

        // 工具函数
        function formatDate(dateString) {
            const date = new Date(dateString);
            return date.toLocaleDateString();
        }

        function formatDateTime(dateString) {
            const date = new Date(dateString);
            return date.toLocaleString();
        }

        function escapeHtml(unsafe) {
            if (!unsafe) return '';
            return unsafe
                .replace(/&/g, "&amp;")
                .replace(/</g, "&lt;")
                .replace(/>/g, "&gt;")
                .replace(/"/g, "&quot;")
                .replace(/'/g, "&#039;");
        }
    </script>
</body>
</html>