<?php
/**
 * 投注添加API接口
 * 处理投注数据并创建相应记录
 */

// 设置HTTP头部信息
header('Content-Type: application/json; charset=utf-8');
header("Access-Control-Allow-Origin: *"); // 允许所有域名
header("Access-Control-Allow-Methods: GET, POST"); // 允许的请求方法
header("Access-Control-Allow-Headers: Content-Type"); // 允许的头部

// 错误处理设置
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// 定义响应结构
$response = [
    'status' => 'error',
    'message' => '未知错误',
    'success_count' => 0,
    'fail_count' => 0,
    'skip_count' => 0,
    'error_details' => []
];

// 数据库连接尝试
try {
    require_once '../db_connect.php';
    if (!isset($pdo) || !$pdo) {
        throw new Exception('数据库连接失败');
    }
} catch (Exception $e) {
    $response['message'] = '系统初始化失败: ' . $e->getMessage();
    $response['error_details'][] = [
        'type' => 'system',
        'message' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ];
    echo json_encode($response);
    exit;
}

// 确保只接受POST请求
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $response['message'] = '只接受POST请求';
    echo json_encode($response);
    exit;
}

// 处理请求数据
try {
    $inputData = file_get_contents('php://input');
    if ($inputData === false) {
        throw new Exception('无法读取请求数据');
    }
    
    $data = json_decode($inputData, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('请求数据格式无效: ' . json_last_error_msg());
    }
    
    // 检查是否有results数组
    if (!isset($data['results']) || !is_array($data['results'])) {
        throw new Exception('缺少必要的投注数据');
    }
    
    $bets = $data['results'];
    $total = count($bets);
    $successCount = 0;
    $failCount = 0;
    $skipCount = 0;
    $autoBetCount = 0;
    $errors = [];
    
    // 开始事务处理
    try {
        $pdo->beginTransaction();
        
        // 遍历处理每条投注数据
        foreach ($bets as $index => $bet) {
            try {
                // 验证投注数据完整性
                validateBetData($bet, $index);
                
                // 检查BetNo是否已存在
                if (isBetNoExists($pdo, $bet['BetNo'])) {
                    $skipCount++;
                    $errors[] = "投注单号 {$bet['BetNo']} 已存在，跳过处理";
                    continue;
                }
                
                // 构建BetContent内容
                $betContent = buildBetContent($bet);
                
                // 处理账户信息
                $accountNos = explode(',', $bet['AccountNo'] ?? '');
                $insertData = prepareInsertData($bet, $betContent, $accountNos);
                
                if (empty($insertData)) {
                    $skipCount++;
                    $errors[] = "投注单号 {$bet['BetNo']} 没有有效的账户信息，跳过处理";
                    continue;
                }
                
                // 执行批量插入
                if (batchInsertBetList($pdo, $insertData)) {
                    $successCount += count($insertData);
                    
                    // 尝试创建自动投注记录
                    try {
                        $autoBetCount += createAutoBet($pdo, $bet, $sysConfig);
                    } catch (Exception $e) {
                        // 自动投注记录创建失败，记录错误但不影响主流程
                        $errors[] = "自动投注记录创建失败 ({$bet['BetNo']}): " . $e->getMessage();
                        error_log('Auto bet creation failed: ' . $e->getMessage() . ' in ' . $e->getFile() . ' on line ' . $e->getLine());
                    }
                } else {
                    $failCount += count($insertData);
                    $errors[] = "投注单号 {$bet['BetNo']} 数据插入失败";
                }
            } catch (Exception $e) {
                $failCount++;
                $errors[] = "处理投注数据 (索引: {$index}) 失败: " . $e->getMessage();
                error_log('Bet processing error: ' . $e->getMessage() . ' in ' . $e->getFile() . ' on line ' . $e->getLine());
                continue;
            }
        }
        
        // 提交事务
        $pdo->commit();
        
        // 构建成功响应
        $response = [
            'status' => 'success',
            'message' => '投注数据处理完成',
            'total_count' => $total,
            'success_count' => $successCount,
            'fail_count' => $failCount,
            'skip_count' => $skipCount,
            'auto_bet_count' => $autoBetCount,
            'error_details' => $errors
        ];
    } catch (PDOException $e) {
        // 数据库事务失败，回滚
        $pdo->rollBack();
        $response['message'] = '数据库操作失败，请稍后再试';
        $response['error_details'][] = [
            'type' => 'database',
            'message' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ];
        error_log('Database transaction error: ' . $e->getMessage() . ' in ' . $e->getFile() . ' on line ' . $e->getLine());
    } catch (Exception $e) {
        // 其他事务处理错误
        if (isset($pdo)) {
            $pdo->rollBack();
        }
        $response['message'] = '数据处理过程中发生错误';
        $response['error_details'][] = [
            'type' => 'processing',
            'message' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ];
        error_log('Processing error: ' . $e->getMessage() . ' in ' . $e->getFile() . ' on line ' . $e->getLine());
    }
} catch (Exception $e) {
    $response['message'] = '请求处理失败';
    $response['error_details'][] = [
        'type' => 'request',
        'message' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ];
    error_log('Request error: ' . $e->getMessage() . ' in ' . $e->getFile() . ' on line ' . $e->getLine());
}

// 输出JSON响应
echo json_encode($response, JSON_UNESCAPED_UNICODE);

/**
 * 验证投注数据的完整性
 * @param array $bet 投注数据
 * @param int $index 索引位置
 * @throws Exception 数据验证失败时抛出异常
 */
function validateBetData($bet, $index) {
    // 检查必填字段
    $requiredKeys = ['BetNo', 'BetAccount', 'BetType', 'BetName', 'BetOdds', 'BetTime', 'BetAmount'];
    
    foreach ($requiredKeys as $key) {
        if (!isset($bet[$key]) || empty(trim($bet[$key]))) {
            throw new Exception("必填字段 '{$key}' 缺失或为空");
        }
    }
    
    // 验证数据类型
    if (!is_numeric($bet['BetOdds']) || !is_numeric($bet['BetAmount'])) {
        throw new Exception('赔率或金额必须为数字');
    }
    
    // 验证BetTime格式
    if (!validateDateTime($bet['BetTime'])) {
        throw new Exception('投注时间格式无效');
    }
}

/**
 * 验证日期时间格式
 * @param string $dateTime 日期时间字符串
 * @return bool 是否有效
 */
function validateDateTime($dateTime) {
    $d = DateTime::createFromFormat('Y-m-d H:i:s', $dateTime);
    return $d && $d->format('Y-m-d H:i:s') === $dateTime;
}

/**
 * 检查投注单号是否已存在
 * @param PDO $pdo 数据库连接
 * @param string $betNo 投注单号
 * @return bool 是否已存在
 */
function isBetNoExists($pdo, $betNo) {
    try {
        $checkStmt = $pdo->prepare("SELECT id FROM TelSms_BetList WHERE BetNo = :BetNo");
        $checkStmt->execute([':BetNo' => $betNo]);
        return $checkStmt->fetch() !== false;
    } catch (PDOException $e) {
        throw new Exception('检查投注单号失败: ' . $e->getMessage());
    }
}

/**
 * 构建投注内容文本
 * @param array $bet 投注数据
 * @return string 构建的投注内容
 */
function buildBetContent($bet) {
    $betContent = "";
    $betContent .= "\n时间:  {$bet['BetTime']}\n";
    $betContent .= "会员:  {$bet['BetAccount']}\n";
    $betContent .= "单号:  {$bet['BetNo']}\n";
    $betContent .= "{$bet['BetType']}:" . "  {$bet['BetHandType']}\n";
    $betContent .= "内容：\n";
    $betContent .= "{$bet['BetName']}\n";
    $betContent .= "{$bet['BetTeamNumberV']}\n";
    $betContent .= "{$bet['BetTeamV']}\n";
    
    // 提取球队名称和盘口数据
    $teamWithHandicap = extractTeamWithHandicap($bet['BetContent']);
    if ($teamWithHandicap) {
        $betContent .= "{$teamWithHandicap} @ {$bet['BetOdds']}\n\n";
    } else {
        $betContent .= "{$bet['BetItem']} @ {$bet['BetOdds']}\n\n";
    }
    
    $betContent .= "<code>￥: {$bet['BetAmount']}</code>";
    return $betContent;
}

/**
 * 准备插入数据
 * @param array $bet 投注数据
 * @param string $betContent 投注内容
 * @param array $accountNos 账户列表
 * @return array 准备好的插入数据
 */
function prepareInsertData($bet, $betContent, $accountNos) {
    $insertData = [];
    
    foreach ($accountNos as $accountNo) {
        $accountNo = trim($accountNo);
        if (!empty($accountNo)) {
            $insertData[] = [
                'AccountNo' => $accountNo,
                'HgAdmin' => $bet['HgAdmin'] ?? '',
                'BetAccount' => $bet['BetAccount'],
                'BetNo' => $bet['BetNo'],
                'BetType' => $bet['BetType'],
                'BetName' => $bet['BetName'],
                'BetHandType' => $bet['BetHandType'] ?? '',
                'BetItem' => $bet['BetItem'] ?? '',
                'BetOdds' => $bet['BetOdds'],
                'BetTime' => $bet['BetTime'],
                'BetTeamV' => $bet['BetTeamV'] ?? '',
                'BetTeamNumberV' => $bet['BetTeamNumberV'] ?? '',
                'BetAmount' => $bet['BetAmount'],
                'BetContent' => $betContent,
                'BetResult' => (int)($bet['BetResult'] ?? 0),
                'BetStatus' => (int)($bet['BetStatus'] ?? 0),
                'SmsStatus' => (int)($bet['SmsStatus'] ?? 0)
            ];
        }
    }
    
    return $insertData;
}

/**
 * 批量插入投注列表数据
 * @param PDO $pdo 数据库连接
 * @param array $insertData 插入数据数组
 * @return bool 是否插入成功
 */
function batchInsertBetList($pdo, $insertData) {
    try {
        // 构建批量插入的 SQL 语句
        $columns = "AccountNo, HgAdmin, BetAccount, BetNo, BetType, BetName, BetHandType, BetItem, BetOdds, BetTime, BetTeamV, BetTeamNumberV, BetAmount, BetContent, BetResult, BetStatus, SmsStatus, CreateDate";
        $placeholders = [];
        
        // 准备占位符
        foreach ($insertData as $index => $item) {
            $placeholders[] = "(:AccountNo_{$index}, :HgAdmin_{$index}, :BetAccount_{$index}, :BetNo_{$index}, :BetType_{$index}, :BetName_{$index}, :BetHandType_{$index}, :BetItem_{$index}, :BetOdds_{$index}, :BetTime_{$index}, :BetTeamV_{$index}, :BetTeamNumberV_{$index}, :BetAmount_{$index}, :BetContent_{$index}, :BetResult_{$index}, :BetStatus_{$index}, :SmsStatus_{$index}, NOW())";
        }
        
        $sql = "INSERT INTO TelSms_BetList ({$columns}) VALUES " . implode(', ', $placeholders);
        $stmt = $pdo->prepare($sql);
        
        // 绑定参数
        foreach ($insertData as $index => $item) {
            $stmt->bindValue(":AccountNo_{$index}", $item['AccountNo'], PDO::PARAM_STR);
            $stmt->bindValue(":HgAdmin_{$index}", $item['HgAdmin'], PDO::PARAM_STR);
            $stmt->bindValue(":BetAccount_{$index}", $item['BetAccount'], PDO::PARAM_STR);
            $stmt->bindValue(":BetNo_{$index}", $item['BetNo'], PDO::PARAM_STR);
            $stmt->bindValue(":BetType_{$index}", $item['BetType'], PDO::PARAM_STR);
            $stmt->bindValue(":BetName_{$index}", $item['BetName'], PDO::PARAM_STR);
            $stmt->bindValue(":BetHandType_{$index}", $item['BetHandType'], PDO::PARAM_STR);
            $stmt->bindValue(":BetItem_{$index}", $item['BetItem'], PDO::PARAM_STR);
            $stmt->bindValue(":BetOdds_{$index}", $item['BetOdds'], PDO::PARAM_STR);
            $stmt->bindValue(":BetTime_{$index}", $item['BetTime'], PDO::PARAM_STR);
            $stmt->bindValue(":BetTeamV_{$index}", $item['BetTeamV'], PDO::PARAM_STR);
            $stmt->bindValue(":BetTeamNumberV_{$index}", $item['BetTeamNumberV'], PDO::PARAM_STR);
            $stmt->bindValue(":BetAmount_{$index}", $item['BetAmount'], PDO::PARAM_STR);
            $stmt->bindValue(":BetContent_{$index}", $item['BetContent'], PDO::PARAM_STR);
            $stmt->bindValue(":BetResult_{$index}", $item['BetResult'], PDO::PARAM_INT);
            $stmt->bindValue(":BetStatus_{$index}", $item['BetStatus'], PDO::PARAM_INT);
            $stmt->bindValue(":SmsStatus_{$index}", $item['SmsStatus'], PDO::PARAM_INT);
        }
        
        return $stmt->execute();
    } catch (PDOException $e) {
        throw new Exception('批量插入数据失败: ' . $e->getMessage());
    }
}

/**
 * 创建自动投注记录
 * @param PDO $pdo 数据库连接
 * @param array $bet 投注数据
 * @param array $sysConfig 系统配置
 * @return int 创建成功的数量
 */
function createAutoBet($pdo, $bet, &$sysConfig = null) {
    try {
        // 获取系统参数配置
        if ($sysConfig === null) {
            $sysConfigStmt = $pdo->prepare("SELECT * FROM bets_sysconfig WHERE hgAdmin = :hgAdmin ORDER BY Id DESC LIMIT 1");
            $sysConfigStmt->execute([':hgAdmin' => $bet['HgAdmin'] ?? '']);
            $sysConfig = $sysConfigStmt->fetch(PDO::FETCH_ASSOC);
        }
        
        // 检查是否找到匹配的参数配置
        if (empty($sysConfig)) {
            throw new Exception('未找到对应的系统配置');
        }
        
        // 确定投注金额
        $betMoney = 0;
        if ($sysConfig['betBeishu'] != 0) {
            $betMoney = floatval($bet['BetAmount']) * floatval($sysConfig['betBeishu']);
        } else {
            $betMoney = floatval($sysConfig['betAmount']);
        }
        
        // 初始化变量
        $betTypeValue = ''; // 投注类型 足球或者篮球
        $betItem = $bet['BetItem'] ?? '';
        $bet_Type = 'handicap'; // 默认让球盘类型
        
        // 将"主队名称 v 客队名称"格式分解为主队和客队名称
        $teamParts = explode(' v ', $bet['BetTeamV'] ?? '');
        $homeTeam = $teamParts[0] ?? ''; // 主队名称
        $awayTeam = $teamParts[1] ?? ''; // 客队名称
        // 去除主队名称中括号内的比分（如：拉里萨(1 - 0) -> 拉里萨）
        $homeTeam = preg_replace('/\(\d+\s*-\s*\d+\)/', '', $homeTeam);
        $homeTeam = trim($homeTeam);
        // 去除客队名称中括号内的比分（如：拉里萨(1 - 0) -> 拉里萨）
        $awayTeam = preg_replace('/\(\d+\s*-\s*\d+\)/', '', $awayTeam);
        $awayTeam = trim($awayTeam);
        
        // 根据投注类型处理
        if ($bet['BetType'] == '足球') {
            $betTypeValue = 'ft';
            
            if ($bet['BetHandType'] == '(今日)让球' || $bet['BetHandType'] == '(滚球) 让球') {
                $extracted = extractTeamWithHandicap2($bet['BetContent']);
                $betItem = $extracted !== null ? $extracted : $betItem;
            } elseif ($bet['BetHandType'] == '(今日)大 / 小' || $bet['BetHandType'] == '(滚球) 大 / 小') {
                $formatted = formatBetItem($bet['BetItem']);
                $betItem = $formatted !== null ? $formatted : $betItem;
                $bet_Type = 'overunder';
            }
            
            // 处理反向跟单
            if (isset($sysConfig['betIsFlower']) && $sysConfig['betIsFlower'] == 1) {
                $betItem = processReverseBet($betItem, $bet_Type, $homeTeam, $awayTeam);
            }
        } elseif ($bet['BetType'] == '篮球 / 美式足球') {
            $betTypeValue = 'bk';
            // 篮球类型的处理逻辑可以在这里扩展
        }
        
        // 插入自动投注记录
        $autoBetStmt = $pdo->prepare("INSERT INTO bets (
            betno, task_id, league, game_id, home_team, away_team, 
            bet_money, bet_type, bet_item, bet_time, game_time, 
            bet_mode, created_by, odds, hgname, hgpass, betType
        ) VALUES (
            :betno, :task_id, :league, :game_id, :home_team, :away_team, 
            :bet_money, :bet_type, :bet_item, :bet_time, :game_time, 
            :bet_mode, :created_by, :odds, :hgname, :hgpass, :betType
        )");
        
        $autoBetStmt->execute([
            ':betno' => $bet['BetNo'],
            ':task_id' => 0, // 默认值
            ':league' => $bet['BetName'] ?? '',
            ':game_id' => $bet['GameId'] ?? '', // 默认值
            ':home_team' => $homeTeam, // 从bet数据中提取
            ':away_team' => $awayTeam, // 从bet数据中提取
            ':bet_money' => $betMoney,
            ':bet_type' => $bet_Type,
            ':bet_item' => $betItem,
            ':bet_time' => date('Y-m-d H:i:s'),
            ':game_time' => date('Y-m-d H:i:s'), // 应该从bet数据中提取
            ':bet_mode' => 'simulation',
            ':created_by' => 'system',
            ':odds' => floatval($bet['BetOdds']),
            ':hgname' => $sysConfig['betHgName'] ?? '',
            ':hgpass' => $sysConfig['betHgPassword'] ?? '',
            ':betType' => $betTypeValue
        ]);
        
        return 1; // 成功创建一条记录
    } catch (Exception $e) {
        throw new Exception('创建自动投注记录失败: ' . $e->getMessage());
    }
}

/**
 * 处理反向投注
 * @param string $betItem 投注项
 * @param string $bet_Type 投注类型
 * @param string $homeTeam 主队名称
 * @param string $awayTeam 客队名称
 * @return string 处理后的投注项
 */
function processReverseBet($betItem, $bet_Type, $homeTeam, $awayTeam) {
    try {
        if ($bet_Type == 'overunder') {
            // 大小球反向处理：大->小，小->大
            if (strpos($betItem, '大@') !== false) {
                $betItem = str_replace('大@', '小@', $betItem);
            } else if (strpos($betItem, '小@') !== false) {
                $betItem = str_replace('小@', '大@', $betItem);
            }
        } else if ($bet_Type == 'handicap') {
            // 让球盘反向处理：交换主队客队，反转盘口值
            // 提取球队名称和盘口值
            $parts = explode('@', $betItem);
            if (count($parts) == 2) {
                $teamName = $parts[0];
                $handicapValue = $parts[1];
                
                // 判断是主队还是客队
                if ($teamName == $homeTeam) {
                    // 主队变客队
                    $newTeamName = $awayTeam;
                } else if ($teamName == $awayTeam) {
                    // 客队变主队
                    $newTeamName = $homeTeam;
                } else {
                    $newTeamName = $teamName;
                }
                
                // 基于字符串符号判断并反转盘口值
                if (strpos($handicapValue, '+') === 0) {
                    // 如果包含+号，则换成-号
                    $formattedHandicap = '-' . substr($handicapValue, 1);
                } else if (strpos($handicapValue, '-') === 0) {
                    // 如果包含-号，则换成+号
                    $formattedHandicap = '+' . substr($handicapValue, 1);
                } else {
                    // 如果没有任何符号，表示是0，则保持不变
                    $formattedHandicap = '0';
                }
                
                // 重新组合投注项
                $betItem = $newTeamName . '@' . $formattedHandicap;
            }
        }
        
        return $betItem;
    } catch (Exception $e) {
        // 反向投注处理失败，返回原始投注项
        error_log('Reverse bet processing error: ' . $e->getMessage());
        return $betItem;
    }
}

/**
 * 提取带盘口的球队名称（空格分隔版）
 * @param string $text 文本内容
 * @return string|null 提取结果或null
 */
function extractTeamWithHandicap($text) {
    try {
        // 正则表达式适配多种格式，包括li标签内的结构
        $pattern = '/([\x{4e00}-\x{9fa5}A-Za-z0-9()\s]+?)\s+(?:<[^>]+>\s*)*<tt class="word_red">([^<]+)<\/tt>/u';
        
        if (preg_match($pattern, $text, $matches)) {
            // 去除多余空格并组合结果
            return trim($matches[1]) . '  ' . trim($matches[2]);
        }
        
        return null;
    } catch (Exception $e) {
        error_log('Extract team with handicap error: ' . $e->getMessage());
        return null;
    }
}

/**
 * 提取带盘口的球队名称（@分隔版）
 * @param string $text 文本内容
 * @return string|null 提取结果或null
 */
function extractTeamWithHandicap2($text) {
    try {
        // 正则表达式适配多种格式，包括li标签内的结构
        $pattern = '/([\x{4e00}-\x{9fa5}A-Za-z0-9()\s]+?)\s+(?:<[^>]+>\s*)*<tt class="word_red">([^<]+)<\/tt>/u';
        
        if (preg_match($pattern, $text, $matches)) {
            // 去除多余空格并组合结果
            
            return trim($matches[1]) . '@' . trim($matches[2]);
        }
        
        return null;
    } catch (Exception $e) {
        error_log('Extract team with handicap 2 error: ' . $e->getMessage());
        return null;
    }
}

/**
 * 格式化投注项
 * @param string $betItem 投注项
 * @return string 格式化后的投注项
 */
function formatBetItem($betItem) {
    try {
        // 支持"大 3.5"或"小 2.5"格式
        $pattern1 = '/^(大|小)\s+(\d+(?:\.\d+)?)$/';
        // 支持"大 2 / 2.5"或"小 3 / 3.5"格式
        $pattern2 = '/^(大|小)\s+(\d+(?:\.\d+)?\s*\/\s*\d+(?:\.\d+)?)$/';
        
        if (preg_match($pattern1, $betItem, $matches)) {
            // 处理单个数字格式
            return $matches[1] . '@' . $matches[2];
        } else if (preg_match($pattern2, $betItem, $matches)) {
            // 处理带斜杠的范围格式，使用extractHandicapValue函数计算平均值
            $handicapValue = extractHandicapValue($matches[2]);
            return $matches[1] . '@' . $handicapValue;
        }
        
        // 如果不匹配预期格式，则返回原始值
        return $betItem;
    } catch (Exception $e) {
        error_log('Format bet item error: ' . $e->getMessage());
        return $betItem;
    }
}



/**
 * 从盘口描述中提取盘口值
 */
function extractHandicapValue($ratioString) {
    // 格式示例: "2.5 / 3" 或 "0.5"
    if (strpos($ratioString, '/') !== false) {
        // 如果包含'/'，计算前后两个数字的平均值
        $parts = explode('/', $ratioString);
        $firstValue = floatval(trim($parts[0]));
        $secondValue = floatval(trim($parts[1]));
        return ($firstValue + $secondValue) / 2;
    } else {
        // 如果不包含'/'，直接返回当前值
        return floatval(trim($ratioString));
    }
}

?>