<?php

require_once '../db.php';

/**
 * 比赛结果数据批量处理接口（支持多AccountNo拆分）
 */

// 设置适当的超时时间和内存限制（针对大数据量）
set_time_limit(60);
ini_set('memory_limit', '256M');
ini_set('post_max_size', '100M');
ini_set('upload_max_filesize', '100M');

// 响应头设置
header("Content-Type: application/json; charset=utf-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST");
header("Access-Control-Allow-Headers: Content-Type");



// 连接数据库
try {
     $pdo = new PDO("mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
     $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    exit(json_encode([
        'status' => 'error',
        'message' => '数据库连接失败',
        'error_code' => $e->getCode()
    ]));
}

// 日期格式转换函数
function convertToMysqlDatetime($dateString) {
    if (empty($dateString)) {
        return null;
    }
    
    try {
        $formats = ['Y-m-d H:i:s', 'Y-m-d\TH:i:s.u\Z', 'Y-m-d\TH:i:s\Z', 'YmdHis'];
        foreach ($formats as $format) {
            $datetime = DateTime::createFromFormat($format, $dateString);
            if ($datetime !== false) {
                return $datetime->format('Y-m-d H:i:s');
            }
        }
        
        $datetime = new DateTime($dateString);
        return $datetime->format('Y-m-d H:i:s');
    } catch (Exception $e) {
        return null;
    }
}

// 读取并验证输入数据
$input = file_get_contents('php://input');
if ($input === false) {
    exit(json_encode(['status' => 'error', 'message' => '无法读取输入数据']));
}

// JSON解析
$data = json_decode($input, true, 512, JSON_BIGINT_AS_STRING);
if (json_last_error() !== JSON_ERROR_NONE) {
    $errorMsg = '';
    switch (json_last_error()) {
        case JSON_ERROR_DEPTH: $errorMsg = 'JSON结构深度超过限制'; break;
        case JSON_ERROR_SYNTAX: $errorMsg = 'JSON语法错误'; break;
        case JSON_ERROR_CTRL_CHAR: $errorMsg = 'JSON包含控制字符'; break;
        case JSON_ERROR_INF_OR_NAN: $errorMsg = 'JSON包含无效的浮点数'; break;
        case JSON_ERROR_UNSUPPORTED_TYPE: $errorMsg = 'JSON包含不支持的类型'; break;
        default: $errorMsg = 'JSON解析错误 (错误码: ' . json_last_error() . ')';
    }
    exit(json_encode([
        'status' => 'error',
        'message' => '无效的JSON格式: ' . $errorMsg,
        'input_length' => strlen($input)
    ]));
}

// 验证数据结构
if (!$data || !isset($data['results']) || !is_array($data['results'])) {
    exit(json_encode([
        'status' => 'error',
        'message' => '无效的数据格式，需要包含results数组'
    ]));
}

$results = $data['results'];
$totalOriginal = count($results); // 原始记录总数
$successCount = 0;
$duplicateCount = 0;
$errorCount = 0;
$errorMessages = [];

// 分页处理
$page = isset($data['page']) ? (int)$data['page'] : 1;
$pageSize = isset($data['page_size']) ? (int)$data['page_size'] : 1000;
$offset = ($page - 1) * $pageSize;
$originalResultsToProcess = array_slice($results, $offset, $pageSize);

// 核心修改1：拆分逗号分隔的AccountNo，生成扩展记录
$expandedResults = [];
foreach ($originalResultsToProcess as $originalItem) {
    // 处理AccountNo拆分 - 只保留第一个有效值
    $accountNos = [];
    if (isset($originalItem['AccountNo']) && !empty($originalItem['AccountNo'])) {
        $splitAccounts = explode(',', $originalItem['AccountNo']);
        // 清理空值和空格
        $cleanedAccounts = array_filter(array_map('trim', $splitAccounts), function($acc) {
            return !empty($acc);
        });
        // 只取第一个有效值
        $firstAccount = reset($cleanedAccounts);
        if (!empty($firstAccount)) {
            $accountNos = [$firstAccount];
        }
    }
    
    // 验证拆分后的AccountNo
    if (empty($accountNos)) {
        $errorCount++;
        $wagerId = $originalItem['WagerId'] ?? '未知';
        $errorMessages[] = "原始记录 WagerId: {$wagerId} 缺少有效的AccountNo（拆分后为空）";
        continue;
    }
    
    // 核心修改2：为每个AccountNo创建单独记录
    foreach ($accountNos as $acc) {
        $newItem = $originalItem;
        $newItem['AccountNo'] = $acc; // 替换为拆分后的单个AccountNo
        $expandedResults[] = $newItem;
    }
}

$currentBatchExpandedCount = count($expandedResults); // 拆分后的记录数

if ($currentBatchExpandedCount > 0) {
    try {
        $pdo->beginTransaction();
        
        // 核心修改3：收集(WagerId, AccountNo)组合用于重复检查
        $checkPairs = [];
        foreach ($expandedResults as $item) {
            if (isset($item['WagerId'], $item['AccountNo'])) {
                $checkPairs[] = [
                    'WagerId' => $item['WagerId'],
                    'AccountNo' => $item['AccountNo']
                ];
            }
        }
        
        // 查询已存在的联合记录
        $existingPairs = [];
        if (!empty($checkPairs)) {
            // 构造(?,?)占位符
            $placeholders = rtrim(str_repeat('(?,?),', count($checkPairs)), ',');
            $stmt = $pdo->prepare(
                "SELECT WagerId, AccountNo FROM TelSms_BetResults 
                 WHERE (WagerId, AccountNo) IN ($placeholders)"
            );
            
            // 扁平化参数数组
            $params = [];
            foreach ($checkPairs as $pair) {
                $params[] = $pair['WagerId'];
                $params[] = $pair['AccountNo'];
            }
            
            $stmt->execute($params);
            $existingRecords = $stmt->fetchAll();
            
            // 转为便于查询的键值对（格式：WagerId|AccountNo）
            foreach ($existingRecords as $rec) {
                $key = "{$rec['WagerId']}|{$rec['AccountNo']}";
                $existingPairs[$key] = true;
            }
        }
        
        // 准备插入语句（保持原有字段，AccountNo已拆分）
        $insertSql = "INSERT INTO TelSms_BetResults (
            WagerId, AccountNo, HgAdmin, Datetime, ResultTime, Username, Alias,
            WagerType, GameTime, BetAmount, OddsType, Tid, GameType, League,
            TeamH, TeamC, TeamHId, TeamCId, Odds, Score, OrderType, OrderCon,
            ResultWl, ResultStatus, IsSettled, WinGold, BetContent, CreatedAt
        ) VALUES (
            :WagerId, :AccountNo, :HgAdmin, :Datetime, :ResultTime, :Username, :Alias,
            :WagerType, :GameTime, :BetAmount, :OddsType, :Tid, :GameType, :League,
            :TeamH, :TeamC, :TeamHId, :TeamCId, :Odds, :Score, :OrderType, :OrderCon,
            :ResultWl, :ResultStatus, :IsSettled, :WinGold, :BetContent, :CreatedAt
        )";
        
        $insertStmt = $pdo->prepare($insertSql);
        
        // 处理每条扩展记录
        foreach ($expandedResults as $item) {
            // 验证必填字段
            if (!isset($item['WagerId']) || empty($item['WagerId'])) {
                $errorCount++;
                $errorMessages[] = "AccountNo: {$item['AccountNo']} 缺少必填字段 WagerId";
                continue;
            }
            
            // 检查联合重复
            $currentKey = "{$item['WagerId']}|{$item['AccountNo']}";
            if (isset($existingPairs[$currentKey])) {
                $duplicateCount++;
                continue;
            }
            
            try {
                // 日期转换
                $createdAt = isset($item['CreatedAt']) ? convertToMysqlDatetime($item['CreatedAt']) : date('Y-m-d H:i:s');
                $datetime = isset($item['Datetime']) ? convertToMysqlDatetime($item['Datetime']) : null;
                $resultTime = isset($item['ResultTime']) ? convertToMysqlDatetime($item['ResultTime']) : null;
                $gameTime = isset($item['GameTime']) ? convertToMysqlDatetime($item['GameTime']) : null;
                
                // 验证日期格式
                if ($datetime === null && isset($item['Datetime'])) {
                    $errorCount++;
                    $errorMessages[] = "WagerId: {$item['WagerId']}, AccountNo: {$item['AccountNo']} 的 Datetime 格式无效";
                    continue;
                }
                
                // 绑定参数
                $insertStmt->bindValue(':WagerId', $item['WagerId']);
                $insertStmt->bindValue(':AccountNo', $item['AccountNo']); // 已拆分的单个AccountNo
                $insertStmt->bindValue(':HgAdmin', $item['HgAdmin'] ?? null);
                $insertStmt->bindValue(':Datetime', $datetime);
                $insertStmt->bindValue(':ResultTime', $resultTime);
                $insertStmt->bindValue(':Username', $item['Username'] ?? null);
                $insertStmt->bindValue(':Alias', $item['Alias'] ?? null);
                $insertStmt->bindValue(':WagerType', $item['WagerType'] ?? null);
                $insertStmt->bindValue(':GameTime', $gameTime);
                $insertStmt->bindValue(':BetAmount', (float)($item['BetAmount'] ?? 0.00));
                $insertStmt->bindValue(':OddsType', $item['OddsType'] ?? null);
                $insertStmt->bindValue(':Tid', $item['Tid'] ?? null);
                $insertStmt->bindValue(':GameType', $item['GameType'] ?? null);
                $insertStmt->bindValue(':League', $item['League'] ?? null);
                $insertStmt->bindValue(':TeamH', $item['TeamH'] ?? null);
                $insertStmt->bindValue(':TeamC', $item['TeamC'] ?? null);
                $insertStmt->bindValue(':TeamHId', $item['TeamHId'] ?? null);
                $insertStmt->bindValue(':TeamCId', $item['TeamCId'] ?? null);
                $insertStmt->bindValue(':Odds', (float)($item['Odds'] ?? 0.00));
                $insertStmt->bindValue(':Score', $item['Score'] ?? null);
                $insertStmt->bindValue(':OrderType', $item['OrderType'] ?? null);
                $insertStmt->bindValue(':OrderCon', $item['OrderCon'] ?? null);
                $insertStmt->bindValue(':ResultWl', $item['ResultWl'] ?? null);
                $insertStmt->bindValue(':ResultStatus', (int)($item['ResultStatus'] ?? 0));
                $insertStmt->bindValue(':IsSettled', (int)($item['IsSettled'] ?? 0));
                $insertStmt->bindValue(':WinGold', (float)($item['WinGold'] ?? 0.00));
                $insertStmt->bindValue(':BetContent', $item['BetContent'] ?? null);
                $insertStmt->bindValue(':CreatedAt', $createdAt);
                
                $insertStmt->execute();
                $successCount++;
                
                // 标记为已存在，避免同批次内重复
                $existingPairs[$currentKey] = true;
                
            } catch (PDOException $e) {
                if ($e->getCode() == 1062) {
                    // 联合唯一键冲突（需确保表有联合索引）
                    $duplicateCount++;
                } else {
                    $errorCount++;
                    $errorMessages[] = "WagerId: {$item['WagerId']}, AccountNo: {$item['AccountNo']} 插入失败";
                    error_log("插入错误: " . $e->getMessage() . " | {$currentKey}");
                }
            }
        }
        
        $pdo->commit();
        
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("事务失败: " . $e->getMessage());
        exit(json_encode([
            'status' => 'error',
            'message' => '批量处理事务失败',
            'total_original' => $totalOriginal,
            'batch_expanded' => $currentBatchExpandedCount,
            'success' => $successCount,
            'duplicate' => $duplicateCount,
            'errors' => $errorCount
        ]));
    }
}

// 分页信息
$totalPages = ceil($totalOriginal / $pageSize);
$hasMore = $page < $totalPages;

// 返回结果
exit(json_encode([
    'status' => 'success',
    'message' => '数据处理完成',
    'pagination' => [
        'page' => $page,
        'page_size' => $pageSize,
        'total_pages' => $totalPages,
        'has_more' => $hasMore
    ],
    'statistics' => [
        'total_original' => $totalOriginal, // 原始记录总数
        'batch_original' => count($originalResultsToProcess), // 本批次原始记录数
        'batch_expanded' => $currentBatchExpandedCount, // 拆分后本批次记录数
        'success' => $successCount,
        'duplicate' => $duplicateCount,
        'errors' => $errorCount
    ],
    'error_details' => count($errorMessages) > 10 
        ? array_slice($errorMessages, 0, 10) + ['还有 ' . (count($errorMessages) - 10) . ' 个错误'] 
        : $errorMessages
]));
?>