<?php
/**
 * 远程控制Chrome浏览器核心功能函数
 * 修复启动卡住和实例列表不显示问题
 */

// 临时设置超时时间（单次操作最大60秒）
set_time_limit(60);

/**
 * 生成唯一实例ID
 */
function generateInstanceId() {
    return uniqid('inst_', true);
}

/**
 * 获取所有实例数据
 */
function getInstances() {
    $instancesFile = INSTANCES_FILE;
    
    // 确保文件存在
    if (!file_exists($instancesFile)) {
        file_put_contents($instancesFile, '[]');
        return [];
    }
    
    // 读取并验证文件内容
    $content = file_get_contents($instancesFile);
    $instances = json_decode($content, true);
    
    // 验证JSON格式
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("实例数据文件损坏，重置为空白");
        file_put_contents($instancesFile, '[]');
        return [];
    }
    
    return is_array($instances) ? $instances : [];
}

/**
 * 保存实例数据
 */
function saveInstances($instances) {
    $instancesFile = INSTANCES_FILE;
    
    // 加锁写入，防止并发问题
    $file = fopen($instancesFile, 'w');
    if (flock($file, LOCK_EX)) {
        fwrite($file, json_encode($instances, JSON_PRETTY_PRINT));
        flock($file, LOCK_UN);
    }
    fclose($file);
}

/**
 * 获取运行中的实例（只保留真实运行的）
 */
function getRunningInstances() {
    $instances = getInstances();
    $running = [];
    
    foreach ($instances as $instance) {
        // 简单有效的进程检查
        if (isProcessRunning($instance['pid'])) {
            $running[] = $instance;
        }
    }
    
    // 保存过滤后的实例列表
    saveInstances($running);
    return $running;
}

/**
 * 检查进程是否在运行（快速版）
 */
function isProcessRunning($pid) {
    if (empty($pid)) return false;
    
    // 使用WMIC命令快速检查PID是否存在
    exec("wmic process where processid={$pid} get processid 2>nul", $output);
    return count($output) > 1; // 第一行是标题，第二行是PID（如果存在）
}

/**
 * 打开浏览器实例（彻底重构版）
 */
function openBrowsers($botId, $count, $urls) {
    $result = ['success' => false, 'count' => 0, 'error' => ''];
    $instances = getRunningInstances();
    
    // 端口配置（限制范围，防止无限查找）
    $minPort = 9322;
    $maxPort = 9350; // 只允许28个实例，避免资源耗尽
    $usedPorts = array_column($instances, 'port');
    $currentPort = $minPort;
    
    // 查找第一个可用端口
    while (in_array($currentPort, $usedPorts) && $currentPort <= $maxPort) {
        $currentPort++;
    }
    
    // 检查端口是否可用
    if ($currentPort > $maxPort) {
        $result['error'] = "端口资源耗尽（已使用 {$minPort}-{$maxPort} 范围）";
        return $result;
    }
    
    // 批量打开浏览器
    for ($i = 0; $i < $count; $i++) {
        // 每次循环重置超时（关键！）
        set_time_limit(30);
        
        $instanceId = generateInstanceId();
        $userDataDir = USER_DATA_BASE . '\instance_' . $instanceId;
        
        // 清理可能的残留目录（限时3秒）
        if (is_dir($userDataDir)) {
            deleteDirectory($userDataDir, 3);
        }
        
        // 构建URL
        $url = $urls . "?AccountNo={$botId}";
        
        // 简化的启动命令（减少冲突）
        $command = '"' . CHROME_PATH . '" ' .
            '--remote-debugging-port=' . $currentPort . ' ' .
            '--user-data-dir="' . $userDataDir . '" ' .
           // '--load-extension="' . EXTENSION_PATH . '" ' .
            '--no-first-run ' .
            '--no-default-browser-check ' .
            '--new-window ' .
            // 窗口大小设置（宽度,高度）
        //    '--window-size=124,268 ' .  // 可修改为需要的尺寸，如1280,720
           // 可选：设置窗口位置（x,y坐标）
         //   '--window-position=100,100 ' .  // 窗口左上角在屏幕上的位置
              // 忽略证书错误的核心参数
            '--ignore-certificate-errors ' .  // 忽略所有证书错误
            '--ignore-urlfetcher-cert-requests ' .  // 忽略URL请求的证书验证
            '--allow-insecure-localhost ' .  // 允许本地主机的不安全连接
            // 禁用后台/遮挡状态下的节流
            '--disable-background-timer-throttling ' .  // 禁用后台定时器节流
            '--disable-backgrounding-occluded-windows ' .  // 禁用遮挡窗口后台化
           // '--disable-features=CalculateNativeWinOcclusion,BackgroundThrottling ' .  // 禁用原生遮挡计算
            '--disable-renderer-backgrounding ' .  // 禁用渲染器后台化
            '--enable-background-mode ' .  // 强制启用后台模式

            '"' . $url . '"';
        
        // 启动进程并获取PID（最多等待5秒）
        $startTime = time();
        $pid = startChromeProcess($command, 5);
        
        // 检查是否超时或失败
        if ($pid === 0 || time() - $startTime > 8) {
            $result['error'] = "第 " . ($i+1) . " 个实例启动失败或超时";
            addLog($result['error'] . "，命令: " . $command);
            continue;
        }
        
        // 最终验证进程是否真的在运行
        if (!isProcessRunning($pid)) {
            $result['error'] = "实例启动后立即终止（PID: {$pid}）";
            addLog($result['error']);
            continue;
        }
        
        // 记录实例信息
        $instances[] = [
            'id' => $instanceId,
            'botId' => $botId,
            'url' => $url,
            'pid' => $pid,
            'userDataDir' => $userDataDir,
            'port' => $currentPort,
            'startTime' => time()
        ];
        
        $currentPort++;
        $result['count']++;
        addLog("成功打开实例 {$instanceId}（PID: {$pid}）");
        
        // 防止一次打开太多，短暂休息
        usleep(500000); // 500ms
    }
    
    // 保存实例数据（无论成功失败都更新）
    saveInstances($instances);
    $result['success'] = $result['count'] > 0;
    return $result;
}

/**
 * 启动Chrome进程并获取PID（更可靠的方式）
 */
function startChromeProcess($command, $timeout = 5) {
    $startTime = time();
    $pid = 0;
    
    // 使用WMIC直接获取新进程的PID（最可靠的方式）
    $wmicCommand = 'wmic process call create "' . str_replace('"', '\\"', $command) . '" 2>&1';
    exec($wmicCommand, $output);
    
    // 解析WMIC输出获取PID
    foreach ($output as $line) {
        if (preg_match('/ProcessId = (\d+)/', $line, $matches)) {
            $pid = (int)$matches[1];
            break;
        }
    }
    
    // 如果获取到PID，等待进程稳定
    if ($pid > 0) {
        // 最多等待指定时间，确认进程存活
        while (time() - $startTime < $timeout) {
            if (isProcessRunning($pid)) {
                return $pid; // 进程确认存活
            }
            usleep(300000); // 300ms
        }
        // 超时后返回0表示失败
        return 0;
    }
    
    // 尝试备选方法获取PID（通过端口）
    if (preg_match('/--remote-debugging-port=(\d+)/', $command, $matches)) {
        $port = $matches[1];
        while (time() - $startTime < $timeout) {
            exec("netstat -ano | findstr :{$port}", $netOutput);
            if (!empty($netOutput)) {
                $lastLine = end($netOutput);
                if (preg_match('/(\d+)\s*$/', $lastLine, $pidMatches)) {
                    return (int)$pidMatches[1];
                }
            }
            usleep(300000);
        }
    }
    
    return 0; // 所有方法都失败
}

/**
 * 关闭单个浏览器实例
 */
function closeBrowser($instanceId) {
    $instances = getInstances();
    $updatedInstances = [];
    $success = false;
    
    foreach ($instances as $instance) {
        if ($instance['id'] == $instanceId) {
            // 杀死进程
            if (isProcessRunning($instance['pid'])) {
                exec("taskkill /F /PID {$instance['pid']} 2>nul", $output, $returnVar);
                if ($returnVar === 0) {
                    addLog("成功关闭进程（PID: {$instance['pid']}）");
                    $success = true;
                }
            }
            
            // 删除用户数据目录（限时5秒）
            if (!empty($instance['userDataDir']) && is_dir($instance['userDataDir'])) {
                deleteDirectory($instance['userDataDir'], 5);
            }
            
            addLog("实例 {$instanceId} 处理完成");
        } else {
            $updatedInstances[] = $instance;
        }
    }
    
    saveInstances($updatedInstances);
    return $success;
}

/**
 * 关闭所有浏览器实例
 */
function closeAllBrowsers() {
    $instances = getInstances();
    $count = count($instances);
    
    foreach ($instances as $instance) {
        if (isProcessRunning($instance['pid'])) {
            exec("taskkill /F /PID {$instance['pid']} 2>nul");
        }
        // 异步删除目录，不等待完成
        if (!empty($instance['userDataDir']) && is_dir($instance['userDataDir'])) {
            deleteDirectoryAsync($instance['userDataDir']);
        }
    }
    
    saveInstances([]);
    addLog("已尝试关闭所有 {$count} 个实例");
    return $count;
}

/**
 * 带超时的递归删除目录
 */
function deleteDirectory($dir, $maxSeconds = 5) {
    $startTime = time();
    
    if (!file_exists($dir) || !is_dir($dir)) {
        return true;
    }
    
    // 简单删除，不重试（避免耗时）
    $files = scandir($dir);
    foreach ($files as $file) {
        if ($file == '.' || $file == '..') continue;
        
        $fullPath = $dir . DIRECTORY_SEPARATOR . $file;
        
        // 超时检查
        if (time() - $startTime > $maxSeconds) {
            return false;
        }
        
        if (is_dir($fullPath)) {
            deleteDirectory($fullPath, $maxSeconds - (time() - $startTime));
        } else {
            @unlink($fullPath);
        }
    }
    
    return @rmdir($dir);
}

/**
 * 异步删除目录（不阻塞主程序）
 */
function deleteDirectoryAsync($dir) {
    if (!file_exists($dir) || !is_dir($dir)) return;
    
    // 创建临时批处理文件
    $batFile = sys_get_temp_dir() . '\delete_' . uniqid() . '.bat';
    $content = "@echo off\n";
    $content .= "timeout /t 2 /nobreak >nul\n"; // 延迟2秒
    $content .= "rmdir /s /q \"{$dir}\"\n";
    $content .= "del \"{$batFile}\"\n"; // 自删除
    
    file_put_contents($batFile, $content);
    
    // 后台执行批处理
    pclose(popen("start /b \"\" \"{$batFile}\"", 'r'));
}

/**
 * 记录日志
 */
function addLog($message) {
    $logDir = LOG_DIR;
    $logFile = $logDir . '\chrome_control_' . date('Ymd') . '.log';
    
    // 确保日志目录存在
    if (!is_dir($logDir)) {
        mkdir($logDir, 0777, true);
    }
    
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[{$timestamp}] {$message}\n";
    file_put_contents($logFile, $logEntry, FILE_APPEND);
}

/**
 * 更新插件并刷新所有实例
 */
function updateAndReloadExtension($newContent) {
    // 保存新的插件内容
    $extPath = EXTENSION_PATH . '\content.js';
    file_put_contents($extPath, $newContent);
    chmod($extPath, 0666);
    
    $instances = getRunningInstances();
    $successCount = 0;
    
    foreach ($instances as $instance) {
        // 使用简单的HTTP请求触发刷新（替代复杂的WebSocket）
        $refreshUrl = "http://127.0.0.1:{$instance['port']}/json";
        $response = @file_get_contents($refreshUrl);
        
        if ($response) {
            $tabs = json_decode($response, true);
            if (is_array($tabs) && !empty($tabs)) {
                // 刷新第一个标签页
                $tabUrl = $tabs[0]['webSocketDebuggerUrl'];
                if (sendReloadCommand($tabUrl)) {
                    $successCount++;
                }
            }
        }
    }
    
    addLog("插件更新完成，成功刷新 {$successCount}/" . count($instances) . " 个实例");
    return $successCount;
}

/**
 * 发送页面刷新命令（简化版）
 */
function sendReloadCommand($debugUrl) {
    try {
        // 使用原生PHP实现简单的WebSocket客户端
        $urlParts = parse_url($debugUrl);
        if (!$urlParts || !isset($urlParts['host'], $urlParts['port'], $urlParts['path'])) {
            return false;
        }
        
        $socket = fsockopen($urlParts['host'], $urlParts['port'], $errno, $errstr, 3);
        if (!$socket) return false;
        
        // WebSocket握手
        $key = base64_encode(random_bytes(16));
        fwrite($socket, "GET {$urlParts['path']}?{$urlParts['query']} HTTP/1.1\r\n");
        fwrite($socket, "Host: {$urlParts['host']}:{$urlParts['port']}\r\n");
        fwrite($socket, "Upgrade: websocket\r\n");
        fwrite($socket, "Connection: Upgrade\r\n");
        fwrite($socket, "Sec-WebSocket-Key: {$key}\r\n");
        fwrite($socket, "Sec-WebSocket-Version: 13\r\n\r\n");
        
        // 快速发送刷新命令
        $command = json_encode(['id' => 1, 'method' => 'Page.reload']);
        $frame = chr(0x81) . chr(strlen($command)) . $command;
        fwrite($socket, $frame);
        fclose($socket);
        
        return true;
    } catch (Exception $e) {
        return false;
    }
}
    